"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const core_1 = require("@aws-cdk/core");
const iam_generated_1 = require("./iam.generated");
const policy_1 = require("./policy");
const principals_1 = require("./principals");
const util_1 = require("./util");
/**
 * Define a new IAM user
 */
class User extends core_1.Resource {
    constructor(scope, id, props = {}) {
        super(scope, id, {
            physicalName: props.userName,
        });
        this.grantPrincipal = this;
        this.assumeRoleAction = 'sts:AssumeRole';
        this.groups = new Array();
        this.managedPolicies = new Array();
        this.attachedPolicies = new util_1.AttachedPolicies();
        this.managedPolicies.push(...props.managedPolicies || []);
        this.permissionsBoundary = props.permissionsBoundary;
        const user = new iam_generated_1.CfnUser(this, 'Resource', {
            userName: this.physicalName,
            groups: util_1.undefinedIfEmpty(() => this.groups),
            managedPolicyArns: core_1.Lazy.listValue({ produce: () => this.managedPolicies.map(p => p.managedPolicyArn) }, { omitEmpty: true }),
            path: props.path,
            permissionsBoundary: this.permissionsBoundary ? this.permissionsBoundary.managedPolicyArn : undefined,
            loginProfile: this.parseLoginProfile(props)
        });
        this.userName = this.getResourceNameAttribute(user.ref);
        this.userArn = this.getResourceArnAttribute(user.attrArn, {
            region: '',
            service: 'iam',
            resource: 'user',
            resourceName: this.physicalName,
        });
        this.policyFragment = new principals_1.ArnPrincipal(this.userArn).policyFragment;
        if (props.groups) {
            props.groups.forEach(g => this.addToGroup(g));
        }
    }
    /**
     * Import an existing user given a username.
     *
     * @param scope construct scope
     * @param id construct id
     * @param userName the username of the existing user to import
     */
    static fromUserName(scope, id, userName) {
        const arn = core_1.Stack.of(scope).formatArn({
            service: 'iam',
            region: '',
            resource: 'user',
            resourceName: userName
        });
        class Import extends core_1.Resource {
            constructor() {
                super(...arguments);
                this.grantPrincipal = this;
                this.userName = userName;
                this.userArn = arn;
                this.assumeRoleAction = 'sts:AssumeRole';
                this.policyFragment = new principals_1.ArnPrincipal(arn).policyFragment;
            }
            addToPolicy(statement) {
                if (!this.defaultPolicy) {
                    this.defaultPolicy = new policy_1.Policy(this, 'Policy');
                    this.defaultPolicy.attachToUser(this);
                }
                this.defaultPolicy.addStatements(statement);
                return true;
            }
            addToGroup(_group) {
                throw new Error('Cannot add imported User to Group');
            }
            attachInlinePolicy(_policy) {
                throw new Error('Cannot add inline policy to imported User');
            }
            addManagedPolicy(_policy) {
                throw new Error('Cannot add managed policy to imported User');
            }
        }
        return new Import(scope, id);
    }
    /**
     * Adds this user to a group.
     */
    addToGroup(group) {
        this.groups.push(group.groupName);
    }
    /**
     * Attaches a managed policy to the user.
     * @param policy The managed policy to attach.
     */
    addManagedPolicy(policy) {
        if (this.managedPolicies.find(mp => mp === policy)) {
            return;
        }
        this.managedPolicies.push(policy);
    }
    /**
     * Attaches a policy to this user.
     */
    attachInlinePolicy(policy) {
        this.attachedPolicies.attach(policy);
        policy.attachToUser(this);
    }
    /**
     * Adds an IAM statement to the default policy.
     *
     * @returns true
     */
    addToPolicy(statement) {
        if (!this.defaultPolicy) {
            this.defaultPolicy = new policy_1.Policy(this, 'DefaultPolicy');
            this.defaultPolicy.attachToUser(this);
        }
        this.defaultPolicy.addStatements(statement);
        return true;
    }
    parseLoginProfile(props) {
        if (props.password) {
            return {
                password: props.password.toString(),
                passwordResetRequired: props.passwordResetRequired
            };
        }
        if (props.passwordResetRequired) {
            throw new Error('Cannot set "passwordResetRequired" without specifying "initialPassword"');
        }
        return undefined; // no console access
    }
}
exports.User = User;
//# sourceMappingURL=data:application/json;base64,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