"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
require("@aws-cdk/assert/jest");
const core_1 = require("@aws-cdk/core");
const lib_1 = require("../lib");
// tslint:disable:object-literal-key-quotes
const roleAccount = '123456789012';
const notRoleAccount = '012345678901';
describe('IAM Role.fromRoleArn', () => {
    let app;
    beforeEach(() => {
        app = new core_1.App();
    });
    let roleStack;
    let importedRole;
    describe('imported with a static ARN', () => {
        const roleName = 'MyRole';
        describe('into an env-agnostic stack', () => {
            beforeEach(() => {
                roleStack = new core_1.Stack(app, 'RoleStack');
                importedRole = lib_1.Role.fromRoleArn(roleStack, 'ImportedRole', `arn:aws:iam::${roleAccount}:role/${roleName}`);
            });
            test('correctly parses the imported role ARN', () => {
                expect(importedRole.roleArn).toBe(`arn:aws:iam::${roleAccount}:role/${roleName}`);
            });
            test('correctly parses the imported role name', () => {
                expect(importedRole.roleName).toBe(roleName);
            });
            describe('then adding a PolicyStatement to it', () => {
                let addToPolicyResult;
                beforeEach(() => {
                    addToPolicyResult = importedRole.addToPolicy(somePolicyStatement());
                });
                test('returns true', () => {
                    expect(addToPolicyResult).toBe(true);
                });
                test("generates a default Policy resource pointing at the imported role's physical name", () => {
                    assertRoleHasDefaultPolicy(roleStack, roleName);
                });
            });
            describe('then attaching a Policy to it', () => {
                let policyStack;
                describe('that belongs to the same stack as the imported role', () => {
                    beforeEach(() => {
                        importedRole.attachInlinePolicy(somePolicy(roleStack, 'MyPolicy'));
                    });
                    test('correctly attaches the Policy to the imported role', () => {
                        assertRoleHasAttachedPolicy(roleStack, roleName, 'MyPolicy');
                    });
                });
                describe('that belongs to a different env-agnostic stack', () => {
                    beforeEach(() => {
                        policyStack = new core_1.Stack(app, 'PolicyStack');
                        importedRole.attachInlinePolicy(somePolicy(policyStack, 'MyPolicy'));
                    });
                    test('correctly attaches the Policy to the imported role', () => {
                        assertRoleHasAttachedPolicy(policyStack, roleName, 'MyPolicy');
                    });
                });
                describe('that belongs to a targeted stack, with account set to', () => {
                    describe('the same account as in the ARN of the imported role', () => {
                        beforeEach(() => {
                            policyStack = new core_1.Stack(app, 'PolicyStack', { env: { account: roleAccount } });
                            importedRole.attachInlinePolicy(somePolicy(policyStack, 'MyPolicy'));
                        });
                        test('correctly attaches the Policy to the imported role', () => {
                            assertRoleHasAttachedPolicy(policyStack, roleName, 'MyPolicy');
                        });
                    });
                    describe('a different account than in the ARN of the imported role', () => {
                        beforeEach(() => {
                            policyStack = new core_1.Stack(app, 'PolicyStack', { env: { account: notRoleAccount } });
                            importedRole.attachInlinePolicy(somePolicy(policyStack, 'MyPolicy'));
                        });
                        test('does NOT attach the Policy to the imported role', () => {
                            assertPolicyDidNotAttachToRole(policyStack, 'MyPolicy');
                        });
                    });
                });
            });
        });
        describe('into a targeted stack with account set to', () => {
            describe('the same account as in the ARN the role was imported with', () => {
                beforeEach(() => {
                    roleStack = new core_1.Stack(app, 'RoleStack', { env: { account: roleAccount } });
                    importedRole = lib_1.Role.fromRoleArn(roleStack, 'ImportedRole', `arn:aws:iam::${roleAccount}:role/${roleName}`);
                });
                describe('then adding a PolicyStatement to it', () => {
                    let addToPolicyResult;
                    beforeEach(() => {
                        addToPolicyResult = importedRole.addToPolicy(somePolicyStatement());
                    });
                    test('returns true', () => {
                        expect(addToPolicyResult).toBe(true);
                    });
                    test("generates a default Policy resource pointing at the imported role's physical name", () => {
                        assertRoleHasDefaultPolicy(roleStack, roleName);
                    });
                });
                describe('then attaching a Policy to it', () => {
                    describe('that belongs to the same stack as the imported role', () => {
                        beforeEach(() => {
                            importedRole.attachInlinePolicy(somePolicy(roleStack, 'MyPolicy'));
                        });
                        test('correctly attaches the Policy to the imported role', () => {
                            assertRoleHasAttachedPolicy(roleStack, roleName, 'MyPolicy');
                        });
                    });
                    describe('that belongs to an env-agnostic stack', () => {
                        let policyStack;
                        beforeEach(() => {
                            policyStack = new core_1.Stack(app, 'PolicyStack');
                            importedRole.attachInlinePolicy(somePolicy(policyStack, 'MyPolicy'));
                        });
                        test('correctly attaches the Policy to the imported role', () => {
                            assertRoleHasAttachedPolicy(policyStack, roleName, 'MyPolicy');
                        });
                    });
                    describe('that belongs to a targeted stack, with account set to', () => {
                        let policyStack;
                        describe('the same account as in the imported role ARN and in the stack the imported role belongs to', () => {
                            beforeEach(() => {
                                policyStack = new core_1.Stack(app, 'PolicyStack', { env: { account: roleAccount } });
                                importedRole.attachInlinePolicy(somePolicy(policyStack, 'MyPolicy'));
                            });
                            test('correctly attaches the Policy to the imported role', () => {
                                assertRoleHasAttachedPolicy(policyStack, roleName, 'MyPolicy');
                            });
                        });
                        describe('a different account than in the imported role ARN and in the stack the imported role belongs to', () => {
                            beforeEach(() => {
                                policyStack = new core_1.Stack(app, 'PolicyStack', { env: { account: notRoleAccount } });
                                importedRole.attachInlinePolicy(somePolicy(policyStack, 'MyPolicy'));
                            });
                            test('does NOT attach the Policy to the imported role', () => {
                                assertPolicyDidNotAttachToRole(policyStack, 'MyPolicy');
                            });
                        });
                    });
                });
            });
            describe('a different account than in the ARN the role was imported with', () => {
                beforeEach(() => {
                    roleStack = new core_1.Stack(app, 'RoleStack', { env: { account: notRoleAccount } });
                    importedRole = lib_1.Role.fromRoleArn(roleStack, 'ImportedRole', `arn:aws:iam::${roleAccount}:role/${roleName}`);
                });
                describe('then adding a PolicyStatement to it', () => {
                    let addToPolicyResult;
                    beforeEach(() => {
                        addToPolicyResult = importedRole.addToPolicy(somePolicyStatement());
                    });
                    test('pretends to succeed', () => {
                        expect(addToPolicyResult).toBe(true);
                    });
                    test("does NOT generate a default Policy resource pointing at the imported role's physical name", () => {
                        expect(roleStack).not.toHaveResourceLike('AWS::IAM::Policy');
                    });
                });
                describe('then attaching a Policy to it', () => {
                    describe('that belongs to the same stack as the imported role', () => {
                        beforeEach(() => {
                            importedRole.attachInlinePolicy(somePolicy(roleStack, 'MyPolicy'));
                        });
                        test('does NOT attach the Policy to the imported role', () => {
                            assertPolicyDidNotAttachToRole(roleStack, 'MyPolicy');
                        });
                    });
                    describe('that belongs to an env-agnostic stack', () => {
                        let policyStack;
                        beforeEach(() => {
                            policyStack = new core_1.Stack(app, 'PolicyStack');
                            importedRole.attachInlinePolicy(somePolicy(policyStack, 'MyPolicy'));
                        });
                        test('does NOT attach the Policy to the imported role', () => {
                            assertPolicyDidNotAttachToRole(policyStack, 'MyPolicy');
                        });
                    });
                    describe('that belongs to a different targeted stack, with account set to', () => {
                        let policyStack;
                        describe('the same account as in the ARN of the imported role', () => {
                            beforeEach(() => {
                                policyStack = new core_1.Stack(app, 'PolicyStack', { env: { account: roleAccount } });
                                importedRole.attachInlinePolicy(somePolicy(policyStack, 'MyPolicy'));
                            });
                            test('does NOT attach the Policy to the imported role', () => {
                                assertPolicyDidNotAttachToRole(policyStack, 'MyPolicy');
                            });
                        });
                        describe('the same account as in the stack the imported role belongs to', () => {
                            beforeEach(() => {
                                policyStack = new core_1.Stack(app, 'PolicyStack', { env: { account: notRoleAccount } });
                                importedRole.attachInlinePolicy(somePolicy(policyStack, 'MyPolicy'));
                            });
                            test('does NOT attach the Policy to the imported role', () => {
                                assertPolicyDidNotAttachToRole(policyStack, 'MyPolicy');
                            });
                        });
                        describe('a third account, different from both the role and scope stack accounts', () => {
                            beforeEach(() => {
                                policyStack = new core_1.Stack(app, 'PolicyStack', { env: { account: 'some-random-account' } });
                                importedRole.attachInlinePolicy(somePolicy(policyStack, 'MyPolicy'));
                            });
                            test('does NOT attach the Policy to the imported role', () => {
                                assertPolicyDidNotAttachToRole(policyStack, 'MyPolicy');
                            });
                        });
                    });
                });
            });
        });
        describe('and with mutable=false', () => {
            beforeEach(() => {
                roleStack = new core_1.Stack(app, 'RoleStack');
                importedRole = lib_1.Role.fromRoleArn(roleStack, 'ImportedRole', `arn:aws:iam::${roleAccount}:role/${roleName}`, { mutable: false });
            });
            describe('then adding a PolicyStatement to it', () => {
                let addToPolicyResult;
                beforeEach(() => {
                    addToPolicyResult = importedRole.addToPolicy(somePolicyStatement());
                });
                test('pretends to succeed', () => {
                    expect(addToPolicyResult).toBe(true);
                });
                test("does NOT generate a default Policy resource pointing at the imported role's physical name", () => {
                    expect(roleStack).not.toHaveResourceLike('AWS::IAM::Policy');
                });
            });
            describe('then attaching a Policy to it', () => {
                let policyStack;
                describe('that belongs to a stack with account equal to the account in the imported role ARN', () => {
                    beforeEach(() => {
                        policyStack = new core_1.Stack(app, 'PolicyStack', { env: { account: roleAccount } });
                        importedRole.attachInlinePolicy(somePolicy(policyStack, 'MyPolicy'));
                    });
                    test('does NOT attach the Policy to the imported role', () => {
                        assertPolicyDidNotAttachToRole(policyStack, 'MyPolicy');
                    });
                });
            });
        });
    });
    describe('imported with a dynamic ARN', () => {
        const dynamicValue = core_1.Lazy.stringValue({ produce: () => 'role-arn' });
        const roleName = {
            'Fn::Select': [1,
                {
                    'Fn::Split': ['/',
                        {
                            'Fn::Select': [5,
                                { 'Fn::Split': [':', 'role-arn'] },
                            ]
                        },
                    ],
                },
            ],
        };
        describe('into an env-agnostic stack', () => {
            beforeEach(() => {
                roleStack = new core_1.Stack(app, 'RoleStack');
                importedRole = lib_1.Role.fromRoleArn(roleStack, 'ImportedRole', dynamicValue);
            });
            test('correctly parses the imported role ARN', () => {
                expect(importedRole.roleArn).toBe(dynamicValue);
            });
            test('correctly parses the imported role name', () => {
                new lib_1.Role(roleStack, 'AnyRole', {
                    roleName: 'AnyRole',
                    assumedBy: new lib_1.ArnPrincipal(importedRole.roleName),
                });
                expect(roleStack).toHaveResourceLike('AWS::IAM::Role', {
                    'AssumeRolePolicyDocument': {
                        'Statement': [
                            {
                                'Action': 'sts:AssumeRole',
                                'Effect': 'Allow',
                                'Principal': {
                                    'AWS': roleName,
                                },
                            },
                        ],
                    },
                });
            });
            describe('then adding a PolicyStatement to it', () => {
                let addToPolicyResult;
                beforeEach(() => {
                    addToPolicyResult = importedRole.addToPolicy(somePolicyStatement());
                });
                test('returns true', () => {
                    expect(addToPolicyResult).toBe(true);
                });
                test("generates a default Policy resource pointing at the imported role's physical name", () => {
                    assertRoleHasDefaultPolicy(roleStack, roleName);
                });
            });
            describe('then attaching a Policy to it', () => {
                let policyStack;
                describe('that belongs to the same stack as the imported role', () => {
                    beforeEach(() => {
                        importedRole.attachInlinePolicy(somePolicy(roleStack, 'MyPolicy'));
                    });
                    test('correctly attaches the Policy to the imported role', () => {
                        assertRoleHasAttachedPolicy(roleStack, roleName, 'MyPolicy');
                    });
                });
                describe('that belongs to a different env-agnostic stack', () => {
                    beforeEach(() => {
                        policyStack = new core_1.Stack(app, 'PolicyStack');
                        importedRole.attachInlinePolicy(somePolicy(policyStack, 'MyPolicy'));
                    });
                    test('correctly attaches the Policy to the imported role', () => {
                        assertRoleHasAttachedPolicy(policyStack, roleName, 'MyPolicy');
                    });
                });
                describe('that belongs to a targeted stack', () => {
                    beforeEach(() => {
                        policyStack = new core_1.Stack(app, 'PolicyStack', { env: { account: roleAccount } });
                        importedRole.attachInlinePolicy(somePolicy(policyStack, 'MyPolicy'));
                    });
                    test('correctly attaches the Policy to the imported role', () => {
                        assertRoleHasAttachedPolicy(policyStack, roleName, 'MyPolicy');
                    });
                });
            });
        });
        describe('into a targeted stack with account set', () => {
            beforeEach(() => {
                roleStack = new core_1.Stack(app, 'RoleStack', { env: { account: roleAccount } });
                importedRole = lib_1.Role.fromRoleArn(roleStack, 'ImportedRole', dynamicValue);
            });
            describe('then adding a PolicyStatement to it', () => {
                let addToPolicyResult;
                beforeEach(() => {
                    addToPolicyResult = importedRole.addToPolicy(somePolicyStatement());
                });
                test('returns true', () => {
                    expect(addToPolicyResult).toBe(true);
                });
                test("generates a default Policy resource pointing at the imported role's physical name", () => {
                    assertRoleHasDefaultPolicy(roleStack, roleName);
                });
            });
            describe('then attaching a Policy to it', () => {
                let policyStack;
                describe('that belongs to the same stack as the imported role', () => {
                    beforeEach(() => {
                        importedRole.attachInlinePolicy(somePolicy(roleStack, 'MyPolicy'));
                    });
                    test('correctly attaches the Policy to the imported role', () => {
                        assertRoleHasAttachedPolicy(roleStack, roleName, 'MyPolicy');
                    });
                });
                describe('that belongs to an env-agnostic stack', () => {
                    beforeEach(() => {
                        policyStack = new core_1.Stack(app, 'PolicyStack');
                        importedRole.attachInlinePolicy(somePolicy(policyStack, 'MyPolicy'));
                    });
                    test('correctly attaches the Policy to the imported role', () => {
                        assertRoleHasAttachedPolicy(policyStack, roleName, 'MyPolicy');
                    });
                });
                describe('that belongs to a different targeted stack, with account set to', () => {
                    describe('the same account as the stack the role was imported into', () => {
                        beforeEach(() => {
                            policyStack = new core_1.Stack(app, 'PolicyStack', { env: { account: roleAccount } });
                            importedRole.attachInlinePolicy(somePolicy(policyStack, 'MyPolicy'));
                        });
                        test('correctly attaches the Policy to the imported role', () => {
                            assertRoleHasAttachedPolicy(policyStack, roleName, 'MyPolicy');
                        });
                    });
                    describe('a different account than the stack the role was imported into', () => {
                        beforeEach(() => {
                            policyStack = new core_1.Stack(app, 'PolicyStack', { env: { account: notRoleAccount } });
                            importedRole.attachInlinePolicy(somePolicy(policyStack, 'MyPolicy'));
                        });
                        test('correctly attaches the Policy to the imported role', () => {
                            assertRoleHasAttachedPolicy(policyStack, roleName, 'MyPolicy');
                        });
                    });
                });
            });
        });
    });
    describe('imported with the ARN of a service role', () => {
        beforeEach(() => {
            roleStack = new core_1.Stack();
            importedRole = lib_1.Role.fromRoleArn(roleStack, 'Role', `arn:aws:iam::${roleAccount}:role/service-role/codebuild-role`);
        });
        it("correctly strips the 'service-role' prefix from the role name", () => {
            new lib_1.Policy(roleStack, 'Policy', {
                statements: [somePolicyStatement()],
                roles: [importedRole],
            });
            expect(roleStack).toHaveResourceLike('AWS::IAM::Policy', {
                'Roles': [
                    'codebuild-role',
                ],
            });
        });
    });
});
function somePolicyStatement() {
    return new lib_1.PolicyStatement({
        actions: ['s3:*'],
        resources: ['xyz'],
    });
}
function somePolicy(policyStack, policyName) {
    const someRole = new lib_1.Role(policyStack, 'SomeExampleRole', {
        assumedBy: new lib_1.AnyPrincipal(),
    });
    const roleResource = someRole.node.defaultChild;
    roleResource.overrideLogicalId('SomeRole'); // force a particular logical ID in the Ref expression
    return new lib_1.Policy(policyStack, 'MyPolicy', {
        policyName,
        statements: [somePolicyStatement()],
        // need at least one of user/group/role, otherwise validation fails
        roles: [someRole],
    });
}
function assertRoleHasDefaultPolicy(stack, roleName) {
    _assertStackContainsPolicyResource(stack, [roleName], undefined);
}
function assertRoleHasAttachedPolicy(stack, roleName, attachedPolicyName) {
    _assertStackContainsPolicyResource(stack, [{ Ref: 'SomeRole' }, roleName], attachedPolicyName);
}
function assertPolicyDidNotAttachToRole(stack, policyName) {
    _assertStackContainsPolicyResource(stack, [{ Ref: 'SomeRole' }], policyName);
}
function _assertStackContainsPolicyResource(stack, roleNames, nameOfPolicy) {
    const expected = {
        PolicyDocument: {
            Statement: [
                {
                    Action: 's3:*',
                    Effect: 'Allow',
                    Resource: 'xyz',
                },
            ],
        },
        Roles: roleNames,
    };
    if (nameOfPolicy) {
        expected.PolicyName = nameOfPolicy;
    }
    expect(stack).toHaveResourceLike('AWS::IAM::Policy', expected);
}
//# sourceMappingURL=data:application/json;base64,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