"""CRC 8 ATM LUT for use in adc_voltage.py"""

# this table is generated using `generate_crc_8_table`, see
# adc_voltage.py.
CRC_8_ATM_LUT = [
    0,
    7,
    14,
    9,
    28,
    27,
    18,
    21,
    56,
    63,
    54,
    49,
    36,
    35,
    42,
    45,
    112,
    119,
    126,
    121,
    108,
    107,
    98,
    101,
    72,
    79,
    70,
    65,
    84,
    83,
    90,
    93,
    224,
    231,
    238,
    233,
    252,
    251,
    242,
    245,
    216,
    223,
    214,
    209,
    196,
    195,
    202,
    205,
    144,
    151,
    158,
    153,
    140,
    139,
    130,
    133,
    168,
    175,
    166,
    161,
    180,
    179,
    186,
    189,
    199,
    192,
    201,
    206,
    219,
    220,
    213,
    210,
    255,
    248,
    241,
    246,
    227,
    228,
    237,
    234,
    183,
    176,
    185,
    190,
    171,
    172,
    165,
    162,
    143,
    136,
    129,
    134,
    147,
    148,
    157,
    154,
    39,
    32,
    41,
    46,
    59,
    60,
    53,
    50,
    31,
    24,
    17,
    22,
    3,
    4,
    13,
    10,
    87,
    80,
    89,
    94,
    75,
    76,
    69,
    66,
    111,
    104,
    97,
    102,
    115,
    116,
    125,
    122,
    137,
    142,
    135,
    128,
    149,
    146,
    155,
    156,
    177,
    182,
    191,
    184,
    173,
    170,
    163,
    164,
    249,
    254,
    247,
    240,
    229,
    226,
    235,
    236,
    193,
    198,
    207,
    200,
    221,
    218,
    211,
    212,
    105,
    110,
    103,
    96,
    117,
    114,
    123,
    124,
    81,
    86,
    95,
    88,
    77,
    74,
    67,
    68,
    25,
    30,
    23,
    16,
    5,
    2,
    11,
    12,
    33,
    38,
    47,
    40,
    61,
    58,
    51,
    52,
    78,
    73,
    64,
    71,
    82,
    85,
    92,
    91,
    118,
    113,
    120,
    127,
    106,
    109,
    100,
    99,
    62,
    57,
    48,
    55,
    34,
    37,
    44,
    43,
    6,
    1,
    8,
    15,
    26,
    29,
    20,
    19,
    174,
    169,
    160,
    167,
    178,
    181,
    188,
    187,
    150,
    145,
    152,
    159,
    138,
    141,
    132,
    131,
    222,
    217,
    208,
    215,
    194,
    197,
    204,
    203,
    230,
    225,
    232,
    239,
    250,
    253,
    244,
    243,
]

class CRCCheckError(Exception):
    """Raised if CRC check fails to match CRC code generated by ADS1263"""


CRC_BYTE_SIZE = 1
# use this with `generate_crc_8_table` below to get LUT for CRC-8-ATM polynomial
CRC_8_ATM_GEN = 0x07


def generate_crc_8_table(crc_polynomial: int):
    """
    Generates LUT containing CRC-8 values for all possible byte (8 bit) values, i.e.
    every entry is the CRC-8 value for the ith byte value from 0,...,255.

    Adapted from:
    https://github.com/osensa/rtu-adc/blob/6fac17efb955c2e8bb40740dd46e48ae51e1d5ff/crc8atm.py

    Args:
        `crc_polynomial`: the CRC-8 polynomial used to generate this LUT

    Returns:
        `list`: the LUT corresponding to this CRC-8 polynomial
    """
    table = list(range(256))

    # for every byte value from 0 to 255
    for index, crc in enumerate(table):
        for _ in range(8):
            # scan bitwise through byte to align leading 1 with CRC polynomial
            # perform divison (XOR) operation once aligned
            crc = (crc << 1) ^ (crc_polynomial if (crc & 0x80) else 0)

        # store only low byte of computed value to account for left shifting
        table[index] = crc & 0xFF

    return table


def check_crc(voltage_bytes: list, crc_code: int):
    """
    Performs CRC-8 check on a list of data bytes. By default uses
    CRC-8-ATM polynomial table

    Adapted from:
    https://github.com/osensa/rtu-adc/blob/6fac17efb955c2e8bb40740dd46e48ae51e1d5ff/crc8atm.py

    Args:
        `data_bytes` (list): a list of bytes ordered as
            [data_1, data_2,..., data_n, crc_8]

        `crc_table`: a previously generated LUT for the CRC-8-ATM
            polynomial
    """
    crc = 0

    data_bytes = voltage_bytes + [crc_code]

    # compute CRC value of data bytes byte wise using LUT
    # which holds CRC values for all possible byte values
    for byte in data_bytes:
        index = crc ^ byte
        crc = CRC_8_ATM_LUT[index]

    if crc != 0:
        raise CRCCheckError(
            f"CRC check failed with codes: computed={hex(crc)}, expected={hex(crc_code)}"
        )

def get_crc(data: list):
    """
    Generate a crc byte of a given data and insert it at the end of the data
    Args:
        data (list): list of data
    Return:
        data_crc (list): list of data + crc byte
    """
    crc = 0

    # compute CRC value of data bytes byte wise using LUT
    # which holds CRC values for all possible byte values
    for byte in data:
        index = crc ^ byte
        crc = CRC_8_ATM_LUT[index]

    # TODO: why not append?
    data_crc = data+[crc]
    return data_crc
