import contextlib
import dataclasses

import httpx

from unico_device_setuper.lib import (
    aapt,
    adb,
    api,
    auth,
    cnsl,
    datadir,
    env,
    sygic,
    uninav,
    unitech,
)


@dataclasses.dataclass
class Args:
    unitech_client_strategy: auth.ClientStrategy
    sygic_products_names: list[str] | None
    unitech_env: env.UnitechEnv
    sygic_env: env.SygicEnv
    uninav_version: uninav.VersionName | None
    device_owner: str | None


async def make_sygic_client(sygic_env: env.SygicEnv, http_client: httpx.AsyncClient):
    return sygic.Client(
        base_url=sygic_env.api_base_url,
        http_client=http_client,
        api_key=await auth.get_sygic_api_key(sygic_env, http_client),
    )


def get_default_device_owner():
    if datadir.is_release_version():
        return 'uni'
    return 'tmp'


@dataclasses.dataclass
class UndevicedSetup:
    args: Args
    api_client: api.Client
    http_client: httpx.AsyncClient
    undeviced_adb: adb.UndevicedAdb
    _unitech_client: unitech.Client
    _sygic_client: sygic.Client | None
    _uninav_version: uninav.VersionName | None
    _device_owner: str | None

    @contextlib.asynccontextmanager
    @staticmethod
    async def make(args: Args):
        cnsl.print_gray(f'Environement Unitech: {args.unitech_env.value}')
        cnsl.print_gray(f'Environement Sygic: {args.sygic_env.value}')
        auth_token = await auth.get_unitech_auth_token(
            args.unitech_env, client_strategy=auth.PICK_ANY
        )
        assert auth_token is not None  # should not fail with PICK_ANY, otherwise cannot do much

        async with (
            unitech.Client(base_url=str(args.unitech_env.api_base_url)) as unitech_client,
            api.Client(
                base_url=str(args.unitech_env.device_setuper_base_url),
                headers={'Authorization': f'Bearer {auth_token}'},
            ) as api_client,
            httpx.AsyncClient() as http_client,
            adb.UndevicedAdb.make(http_client) as undeviced_adb,
        ):
            yield UndevicedSetup(
                args=args,
                api_client=api_client,
                http_client=http_client,
                undeviced_adb=undeviced_adb,
                _unitech_client=unitech_client,
                _sygic_client=None,
                _uninav_version=args.uninav_version,
                _device_owner=args.device_owner,
            )

    @contextlib.asynccontextmanager
    async def with_device(self, device: adb.Device):
        adb_ = self.undeviced_adb.with_device(device)
        async with aapt.Aapt.make(adb_, self.http_client) as aapt_:
            yield Setup(self, adb_, aapt_, uninav.Uninav(adb_, aapt_))

    async def get_unitech_client(self):
        headers = self._unitech_client.get_async_httpx_client().headers
        auth_header_name = 'Authorization'
        if headers.get(auth_header_name) is None:
            auth_token = await auth.get_unitech_auth_token(
                self.args.unitech_env, self.args.unitech_client_strategy
            )
            if auth_token is None:
                return None
            headers[auth_header_name] = f'Bearer {auth_token}'
        return self._unitech_client

    async def get_sygic_client(self):
        if self._sygic_client is None:
            self._sygic_client = await make_sygic_client(self.args.sygic_env, self.http_client)
        return self._sygic_client

    async def get_uninav_version(self):
        if self._uninav_version is None:
            unitech_client = await self.get_unitech_client()
            if unitech_client is None:
                return None
            self._uninav_version = await uninav.VersionName.choose(
                unitech_env=self.args.unitech_env,
                unitech_client=unitech_client,
                http_client=self.http_client,
            )
        return self._uninav_version

    async def get_device_owner(self):
        if self._device_owner is None:
            default = get_default_device_owner()
            self._device_owner = await (
                cnsl.input(f"Propriétaire de l'appareil (par défaut: {default}): ") or default
            )
        return self._device_owner


@dataclasses.dataclass
class Setup:
    _undeviced_setup: UndevicedSetup
    adb: adb.Adb
    aapt: aapt.Aapt
    uninav: uninav.Uninav

    def get_unitech_client(self):
        return self._undeviced_setup.get_unitech_client()

    def get_sygic_client(self):
        return self._undeviced_setup.get_sygic_client()

    def get_uninav_version(self):
        return self._undeviced_setup.get_uninav_version()

    def get_device_owner(self):
        return self._undeviced_setup.get_device_owner()

    @property
    def args(self):
        return self._undeviced_setup.args

    @property
    def api_client(self):
        return self._undeviced_setup.api_client

    @property
    def http_client(self):
        return self._undeviced_setup.http_client
