"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Bundling = void 0;
const os = require("os");
const path = require("path");
const aws_lambda_1 = require("aws-cdk-lib/aws-lambda");
const cdk = require("aws-cdk-lib/core");
const util_1 = require("./util");
/**
 * Bundling
 */
class Bundling {
    static bundle(options) {
        const bundling = new Bundling(options);
        return aws_lambda_1.Code.fromAsset(path.dirname(options.moduleDir), {
            assetHashType: options.assetHashType ?? cdk.AssetHashType.OUTPUT,
            assetHash: options.assetHash,
            bundling: {
                image: bundling.image,
                command: bundling.command,
                environment: bundling.environment,
                local: bundling.local,
                entrypoint: bundling.entrypoint,
                volumes: bundling.volumes,
                volumesFrom: bundling.volumesFrom,
                workingDirectory: bundling.workingDirectory,
                user: bundling.user,
                securityOpt: bundling.securityOpt,
                network: bundling.network,
                bundlingFileAccess: bundling.bundlingFileAccess,
            },
        });
    }
    static clearRunsLocallyCache() {
        this.runsLocally = undefined;
    }
    constructor(props) {
        this.props = props;
        Bundling.runsLocally = Bundling.runsLocally
            ?? (0, util_1.getGoBuildVersion)()
            ?? false;
        const projectRoot = path.dirname(props.moduleDir);
        this.relativeEntryPath = `./${path.relative(projectRoot, path.resolve(props.entry))}`;
        const cgoEnabled = props.cgoEnabled ? '1' : '0';
        const environment = {
            CGO_ENABLED: cgoEnabled,
            GO111MODULE: 'on',
            GOARCH: props.architecture.dockerPlatform.split('/')[1],
            GOOS: 'linux',
            ...props.environment,
        };
        if (props.goProxies) {
            environment.GOPROXY = props.goProxies.join(',');
        }
        // Docker bundling
        const shouldBuildImage = props.forcedDockerBundling || !Bundling.runsLocally;
        this.image = shouldBuildImage
            ? props.dockerImage ?? cdk.DockerImage.fromBuild(path.join(__dirname, '..', 'lib'), {
                buildArgs: {
                    ...props.buildArgs ?? {},
                    IMAGE: aws_lambda_1.Runtime.GO_1_X.bundlingImage.image, // always use the GO_1_X build image
                },
                platform: props.architecture.dockerPlatform,
            })
            : cdk.DockerImage.fromRegistry('dummy'); // Do not build if we don't need to
        const bundlingCommand = this.createBundlingCommand(cdk.AssetStaging.BUNDLING_INPUT_DIR, cdk.AssetStaging.BUNDLING_OUTPUT_DIR);
        this.command = props.command ?? ['bash', '-c', bundlingCommand];
        this.environment = environment;
        this.entrypoint = props.entrypoint;
        this.volumes = props.volumes;
        this.volumesFrom = props.volumesFrom;
        this.workingDirectory = props.workingDirectory;
        this.user = props.user;
        this.securityOpt = props.securityOpt;
        this.network = props.network;
        this.bundlingFileAccess = props.bundlingFileAccess;
        // Local bundling
        if (!props.forcedDockerBundling) { // only if Docker is not forced
            const osPlatform = os.platform();
            const createLocalCommand = (outputDir) => this.createBundlingCommand(projectRoot, outputDir, osPlatform);
            this.local = {
                tryBundle(outputDir) {
                    if (Bundling.runsLocally == false) {
                        process.stderr.write('go build cannot run locally. Switching to Docker bundling.\n');
                        return false;
                    }
                    const localCommand = createLocalCommand(outputDir);
                    (0, util_1.exec)(osPlatform === 'win32' ? 'cmd' : 'bash', [
                        osPlatform === 'win32' ? '/c' : '-c',
                        localCommand,
                    ], {
                        env: { ...process.env, ...environment ?? {} },
                        stdio: [
                            'ignore', // ignore stdio
                            process.stderr, // redirect stdout to stderr
                            'inherit', // inherit stderr
                        ],
                        cwd: path.dirname(props.moduleDir),
                        windowsVerbatimArguments: osPlatform === 'win32',
                    });
                    return true;
                },
            };
        }
    }
    createBundlingCommand(inputDir, outputDir, osPlatform = 'linux') {
        const pathJoin = osPathJoin(osPlatform);
        const hasVendor = (0, util_1.findUp)('vendor', path.dirname(this.props.entry));
        const goBuildCommand = [
            'go', 'build',
            hasVendor ? '-mod=vendor' : '',
            '-o', `"${pathJoin(outputDir, 'bootstrap')}"`,
            `${this.props.goBuildFlags ? this.props.goBuildFlags.join(' ') : ''}`,
            `${this.relativeEntryPath.replace(/\\/g, '/')}`,
        ].filter(c => !!c).join(' ');
        return chain([
            ...this.props.commandHooks?.beforeBundling(inputDir, outputDir) ?? [],
            goBuildCommand,
            ...this.props.commandHooks?.afterBundling(inputDir, outputDir) ?? [],
        ]);
    }
}
exports.Bundling = Bundling;
/**
 * Platform specific path join
 */
function osPathJoin(platform) {
    return function (...paths) {
        const joined = path.join(...paths);
        // If we are on win32 but need posix style paths
        if (os.platform() === 'win32' && platform !== 'win32') {
            return joined.replace(/\\/g, '/');
        }
        return joined;
    };
}
function chain(commands) {
    return commands.filter(c => !!c).join(' && ');
}
//# sourceMappingURL=data:application/json;base64,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