#
# Licensed to the Apache Software Foundation (ASF) under one or more
# contributor license agreements.  See the NOTICE file distributed with
# this work for additional information regarding copyright ownership.
# The ASF licenses this file to You under the Apache License, Version 2.0
# (the "License"); you may not use this file except in compliance with
# the License.  You may obtain a copy of the License at
#
#    http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
# See the License for the specific language governing permissions and
# limitations under the License.
#

"""
A collections of builtin avro functions
"""


from typing import Dict, Optional, TYPE_CHECKING, cast

from pyspark.sql.column import Column, _to_java_column
from pyspark.errors import PySparkTypeError
from pyspark.sql.utils import get_active_spark_context, try_remote_avro_functions
from pyspark.util import _print_missing_jar

if TYPE_CHECKING:
    from pyspark.sql._typing import ColumnOrName


@try_remote_avro_functions
def from_avro(
    # BEGIN-EDGE
    data: "ColumnOrName",
    jsonFormatSchema: Optional[str] = None,
    options: Optional[Dict[str, str]] = None,
    subject: Optional[str] = None,
    schemaRegistryAddress: Optional[str] = None,
    # END-EDGE
) -> Column:
    """
    Converts a binary column of Avro format into its corresponding catalyst value.
    The specified schema must match the read data, otherwise the behavior is undefined:
    it may fail or return arbitrary result.
    To deserialize the data with a compatible and evolved schema, the expected Avro schema can be
    set via the option avroSchema.

    If `jsonFormatSchema` is not provided but both `subject` and `schemaRegistryAddress`
    are provided, the function converts a binary column of Schema-Registry avro format into its
    corresponding catalyst value. The schema of the given subject in Schema-Registry should not
    change in an incompatible way, otherwise exception will be thrown at runtime when Spark
    consumes data with new schema.

    .. versionadded:: 3.0.0

    .. versionchanged:: 3.5.0
        Supports Spark Connect.

    Parameters
    ----------
    data : :class:`~pyspark.sql.Column` or str
        the binary column.
    jsonFormatSchema : str, optional
        the avro schema in JSON string format.
    options : dict, optional
        options to control how the Avro record is parsed and configs for schema registry client.

    subject: str, optional
        the subject in Schema-Registry that these data belong to.

        .. versionadded:: 3.4.0

    schemaRegistryAddress : str, optional
        the address(host and port) of Schema-Registry.

        .. versionadded:: 3.4.0

    Notes
    -----
    Avro is built-in but external data source module since Spark 2.4. Please deploy the
    application as per the deployment section of "Apache Avro Data Source Guide".

    Examples
    --------
    >>> from pyspark.sql import Row
    >>> from pyspark.sql.avro.functions import from_avro, to_avro
    >>> data = [(1, Row(age=2, name='Alice'))]
    >>> df = spark.createDataFrame(data, ("key", "value"))
    >>> avroDf = df.select(to_avro(df.value).alias("avro"))
    >>> avroDf.collect()
    [Row(avro=bytearray(b'\\x00\\x00\\x04\\x00\\nAlice'))]

    >>> jsonFormatSchema = '''{"type":"record","name":"topLevelRecord","fields":
    ...     [{"name":"avro","type":[{"type":"record","name":"value","namespace":"topLevelRecord",
    ...     "fields":[{"name":"age","type":["long","null"]},
    ...     {"name":"name","type":["string","null"]}]},"null"]}]}'''
    >>> avroDf.select(from_avro(avroDf.avro, jsonFormatSchema).alias("value")).collect()
    [Row(value=Row(avro=Row(age=2, name='Alice')))]
    >>> import org.apache.avro.{Schema, SchemaBuilder}
    >>> schemaRegistryClient.register("input", SchemaBuilder.builder().stringType())

    >>> from pyspark.sql.functions import col, lit, struct
    >>> from pyspark.sql.avro.functions import from_avro, to_avro
    >>> options = {"confluent.schema.registry.basic.auth.credentials.source": 'USER_INFO',
    >>>     "confluent.schema.registry.basic.auth.user.info": f"{key}:{secret}"}
    >>> avroDF = spark.range(5)
    ...   .select(col("id").cast("STRING").alias("str"))
    ...   .select(to_avro("str", None, lit("input"), schemaRegistryAddress, options)
    ...           .alias("value")).collect()
    [Row(value=bytearray(b'\x00\x00\x00\x00\x01\x020')),
     Row(value=bytearray(b'\x00\x00\x00\x00\x01\x021')),
     Row(value=bytearray(b'\x00\x00\x00\x00\x01\x022')),
     Row(value=bytearray(b'\x00\x00\x00\x00\x01\x023')),
     Row(value=bytearray(b'\x00\x00\x00\x00\x01\x024'))]

    >>> avroDF = spark.range(5)
    ...   .select(col("id").cast("STRING").alias("str"))
    ...   .select(to_avro("str", None, lit("input"), schemaRegistryAddress, options)
    ...           .alias("value"))
    ...   .select(from_avro("value", None, options, "input", schemaRegistryAddress))
    >>> avroDF.show()
    +----------------+
    |from_avro(value)|
    +----------------+
    |               0|
    |               1|
    |               2|
    |               3|
    |               4|
    +----------------+
    """
    from py4j.java_gateway import JVMView

    if not isinstance(data, (Column, str)):
        raise PySparkTypeError(
            errorClass="INVALID_TYPE",
            messageParameters={
                "arg_name": "data",
                "arg_type": "pyspark.sql.Column or str",
            },
        )
    if jsonFormatSchema is not None and not isinstance(jsonFormatSchema, str):
        raise PySparkTypeError(
            errorClass="INVALID_TYPE",
            messageParameters={"arg_name": "jsonFormatSchema", "arg_type": "str, optional"},
        )
    if options is not None and not isinstance(options, dict):
        raise PySparkTypeError(
            errorClass="INVALID_TYPE",
            messageParameters={"arg_name": "options", "arg_type": "dict, optional"},
        )

    sc = get_active_spark_context()
    try:
        # BEGIN-EDGE
        if subject is not None and not isinstance(subject, str):
            raise PySparkTypeError(
                errorClass="INVALID_TYPE",
                messageParameters={"arg_name": "subject", "arg_type": "str, optional"},
            )
        if schemaRegistryAddress is not None and not isinstance(schemaRegistryAddress, str):
            raise PySparkTypeError(
                errorClass="INVALID_TYPE",
                messageParameters={
                    "arg_name": "schemaRegistryAddress",
                    "arg_type": "str, optional",
                },
            )

        if not jsonFormatSchema:
            if subject and schemaRegistryAddress:
                jc = cast(JVMView, sc._jvm).org.apache.spark.sql.avro.functions.from_avro(
                    _to_java_column(data), subject, schemaRegistryAddress, options or {}
                )
            else:
                raise ValueError(
                    "Both subject and schemaRegistryAddress should be provided for Schema Registry"
                )
        else:
            jc = getattr(cast(JVMView, sc._jvm), "org.apache.spark.sql.avro.functions").from_avro(
                _to_java_column(data), jsonFormatSchema, options or {}
            )
        # END-EDGE
    except TypeError as e:
        if str(e) == "'JavaPackage' object is not callable":
            _print_missing_jar("Avro", "avro", "avro", sc.version)
        raise
    return Column(jc)


@try_remote_avro_functions
# BEGIN-EDGE
def to_avro(
    data: "ColumnOrName",
    jsonFormatSchema: Optional[str] = None,
    subject: Optional["ColumnOrName"] = None,
    schemaRegistryAddress: Optional[str] = None,
    options: Optional[Dict[str, str]] = None,
) -> Column:
    # END-EDGE
    """
    Converts a column into binary of avro format.

    If both `subject` and `schemaRegistryAddress` are provided, the function converts a column into
    binary of Schema-Registry Avro format. The input data schema must have been registered to the
    given subject in Schema-Registry, or the query will fail at runtime.

    .. versionadded:: 3.0.0

    .. versionchanged:: 3.5.0
        Supports Spark Connect.

    Parameters
    ----------
    data : :class:`~pyspark.sql.Column` or str
        the data column.
    jsonFormatSchema : str, optional
        user-specified output avro schema in JSON string format.

    subject : :class:`~pyspark.sql.Column` or str, optional
        the subject in Schema-Registry that these data belong to.

        .. versionadded:: 3.4.0

    schemaRegistryAddress : str, optional
        the address(host and port) of Schema-Registry.

        .. versionadded:: 3.4.0

    options : dict, optional
        options to control how the Avro record is parsed and configs for schema registry client.

        .. versionadded:: 3.4.0

    Notes
    -----
    Avro is built-in but external data source module since Spark 2.4. Please deploy the
    application as per the deployment section of "Apache Avro Data Source Guide".

    Examples
    --------
    >>> from pyspark.sql import Row
    >>> from pyspark.sql.avro.functions import to_avro
    >>> data = ['SPADES']
    >>> df = spark.createDataFrame(data, "string")
    >>> df.select(to_avro(df.value).alias("suite")).collect()
    [Row(suite=bytearray(b'\\x00\\x0cSPADES'))]

    >>> jsonFormatSchema = '''["null", {"type": "enum", "name": "value",
    ...     "symbols": ["SPADES", "HEARTS", "DIAMONDS", "CLUBS"]}]'''
    >>> df.select(to_avro(df.value, jsonFormatSchema).alias("suite")).collect()
    [Row(suite=bytearray(b'\\x02\\x00'))]
    >>> import org.apache.avro.{Schema, SchemaBuilder}
    >>> schemaRegistryClient.register("input", SchemaBuilder.builder().stringType())

    >>> from pyspark.sql.functions import col, lit, struct
    >>> from pyspark.sql.avro.functions import from_avro, to_avro
    >>> options = {"confluent.schema.registry.basic.auth.credentials.source": 'USER_INFO',
    >>>     "confluent.schema.registry.basic.auth.user.info": f"{key}:{secret}"}
    >>> avroDF = spark.range(5)
    ...   .select(col("id").cast("STRING").alias("str"))
    ...   .select(to_avro("str", None, lit("input"), schemaRegistryAddress, options)
    ...           .alias("value")).collect()
    [Row(value=bytearray(b'\x00\x00\x00\x00\x01\x020')),
     Row(value=bytearray(b'\x00\x00\x00\x00\x01\x021')),
     Row(value=bytearray(b'\x00\x00\x00\x00\x01\x022')),
     Row(value=bytearray(b'\x00\x00\x00\x00\x01\x023')),
     Row(value=bytearray(b'\x00\x00\x00\x00\x01\x024'))]

    >>> avroDF = spark.range(5)
    ...   .select(col("id").cast("STRING").alias("str"))
    ...   .select(to_avro("str", None, lit("input"), schemaRegistryAddress, options)
    ...           .alias("value"))
    ...   .select(from_avro("value", None, options, "input", schemaRegistryAddress))
    >>> avroDF.show()
    +----------------+
    |from_avro(value)|
    +----------------+
    |               0|
    |               1|
    |               2|
    |               3|
    |               4|
    +----------------+
    """
    from py4j.java_gateway import JVMView

    if not isinstance(data, (Column, str)):
        raise PySparkTypeError(
            errorClass="INVALID_TYPE",
            messageParameters={
                "arg_name": "data",
                "arg_type": "pyspark.sql.Column or str",
            },
        )
    if jsonFormatSchema is not None and not isinstance(jsonFormatSchema, str):
        raise PySparkTypeError(
            errorClass="INVALID_TYPE",
            messageParameters={"arg_name": "jsonFormatSchema", "arg_type": "str, optional"},
        )

    sc = get_active_spark_context()
    try:
        # BEGIN-EDGE
        if subject is not None and not isinstance(subject, (Column, str)):
            raise PySparkTypeError(
                errorClass="INVALID_TYPE",
                messageParameters={
                    "arg_name": "subject",
                    "arg_type": "pyspark.sql.Column or str, optional",
                },
            )
        if schemaRegistryAddress is not None and not isinstance(schemaRegistryAddress, str):
            raise PySparkTypeError(
                errorClass="INVALID_TYPE",
                messageParameters={
                    "arg_name": "schemaRegistryAddress",
                    "arg_type": "str, optional",
                },
            )
        if options is not None and not isinstance(options, dict):
            raise PySparkTypeError(
                errorClass="INVALID_TYPE",
                messageParameters={"arg_name": "options", "arg_type": "dict, optional"},
            )

        if jsonFormatSchema:
            if subject is not None and schemaRegistryAddress:
                jc = getattr(cast(JVMView, sc._jvm), "org.apache.spark.sql.avro.functions").to_avro(
                    _to_java_column(data),
                    _to_java_column(subject),
                    schemaRegistryAddress,
                    options or {},
                    jsonFormatSchema,
                )
            else:
                jc = getattr(cast(JVMView, sc._jvm), "org.apache.spark.sql.avro.functions").to_avro(
                    _to_java_column(data), jsonFormatSchema
                )
        else:
            if subject is not None and schemaRegistryAddress:
                jc = getattr(cast(JVMView, sc._jvm), "org.apache.spark.sql.avro.functions").to_avro(
                    _to_java_column(data),
                    _to_java_column(subject),
                    schemaRegistryAddress,
                    options or {},
                )
            else:
                jc = getattr(cast(JVMView, sc._jvm), "org.apache.spark.sql.avro.functions").to_avro(
                    _to_java_column(data)
                )
        # END-EDGE
    except TypeError as e:
        if str(e) == "'JavaPackage' object is not callable":
            _print_missing_jar("Avro", "avro", "avro", sc.version)
        raise
    return Column(jc)


def _test() -> None:
    import os
    import sys
    from pyspark.testing.sqlutils import search_jar

    avro_jar = search_jar("connector/avro", "spark-avro", "spark-avro")
    if avro_jar is None:
        print(
            "Skipping all Avro Python tests as the optional Avro project was "
            "not compiled into a JAR. To run these tests, "
            "you need to build Spark with 'build/sbt -Pavro package' or "
            "'build/mvn -Pavro package' before running this test."
        )
        sys.exit(0)
    else:
        existing_args = os.environ.get("PYSPARK_SUBMIT_ARGS", "pyspark-shell")
        jars_args = "--jars %s" % avro_jar
        os.environ["PYSPARK_SUBMIT_ARGS"] = " ".join([jars_args, existing_args])

    import doctest
    from pyspark.sql import SparkSession
    import pyspark.sql.avro.functions

    globs = pyspark.sql.avro.functions.__dict__.copy()
    spark = (
        SparkSession.builder.master("local[4]").appName("sql.avro.functions tests").getOrCreate()
    )
    globs["spark"] = spark
    (failure_count, test_count) = doctest.testmod(
        pyspark.sql.avro.functions,
        globs=globs,
        optionflags=doctest.ELLIPSIS | doctest.NORMALIZE_WHITESPACE,
    )
    spark.stop()
    if failure_count:
        sys.exit(-1)


if __name__ == "__main__":
    _test()
