"use strict";
/**
 *  Copyright 2019 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
Object.defineProperty(exports, "__esModule", { value: true });
const assert_1 = require("@aws-cdk/assert");
const lib_1 = require("../lib");
const lambda = require("@aws-cdk/aws-lambda");
const cdk = require("@aws-cdk/core");
require("@aws-cdk/assert/jest");
function deployStack(stack) {
    const props = {
        lambdaFunctionProps: {
            code: lambda.Code.asset(`${__dirname}/lambda`),
            runtime: lambda.Runtime.NODEJS_10_X,
            handler: 'index.handler'
        },
        iotTopicRuleProps: {
            topicRulePayload: {
                ruleDisabled: false,
                description: "Processing of DTC messages from the AWS Connected Vehicle Solution.",
                sql: "SELECT * FROM 'connectedcar/dtc/#'",
                actions: []
            }
        }
    };
    return new lib_1.IotToLambdaToDynamoDB(stack, 'test-iot-lambda-dynamodb-stack', props);
}
test('snapshot test IotToLambdaToDynamoDB default params', () => {
    const stack = new cdk.Stack();
    deployStack(stack);
    expect(assert_1.SynthUtils.toCloudFormation(stack)).toMatchSnapshot();
});
test('check lambda function properties', () => {
    const stack = new cdk.Stack();
    deployStack(stack);
    expect(stack).toHaveResource('AWS::Lambda::Function', {
        Handler: "index.handler",
        Role: {
            "Fn::GetAtt": [
                "testiotlambdadynamodbstackIotToLambdaLambdaFunctionServiceRoleC57F7FDA",
                "Arn"
            ]
        },
        Runtime: "nodejs10.x",
        Environment: {
            Variables: {
                AWS_NODEJS_CONNECTION_REUSE_ENABLED: "1",
                DDB_TABLE_NAME: {
                    Ref: "testiotlambdadynamodbstackLambdaToDynamoDBDynamoTableE17E5733"
                }
            }
        }
    });
});
test('check lambda function permission', () => {
    const stack = new cdk.Stack();
    deployStack(stack);
    expect(stack).toHaveResource('AWS::Lambda::Permission', {
        Action: "lambda:InvokeFunction",
        FunctionName: {
            "Fn::GetAtt": [
                "testiotlambdadynamodbstackIotToLambdaLambdaFunctionDFEAF894",
                "Arn"
            ]
        },
        Principal: "iot.amazonaws.com",
        SourceArn: {
            "Fn::GetAtt": [
                "testiotlambdadynamodbstackIotToLambdaIotTopic74F5E3BB",
                "Arn"
            ]
        }
    });
});
test('check iot lambda function role', () => {
    const stack = new cdk.Stack();
    deployStack(stack);
    expect(stack).toHaveResource('AWS::IAM::Role', {
        AssumeRolePolicyDocument: {
            Statement: [
                {
                    Action: "sts:AssumeRole",
                    Effect: "Allow",
                    Principal: {
                        Service: "lambda.amazonaws.com"
                    }
                }
            ],
            Version: "2012-10-17"
        },
        Policies: [
            {
                PolicyDocument: {
                    Statement: [
                        {
                            Action: [
                                "logs:CreateLogGroup",
                                "logs:CreateLogStream",
                                "logs:PutLogEvents"
                            ],
                            Effect: "Allow",
                            Resource: {
                                "Fn::Join": [
                                    "",
                                    [
                                        "arn:aws:logs:",
                                        {
                                            Ref: "AWS::Region"
                                        },
                                        ":",
                                        {
                                            Ref: "AWS::AccountId"
                                        },
                                        ":log-group:/aws/lambda/*"
                                    ]
                                ]
                            }
                        }
                    ],
                    Version: "2012-10-17"
                },
                PolicyName: "LambdaFunctionServiceRolePolicy"
            }
        ]
    });
});
test('check iot topic rule properties', () => {
    const stack = new cdk.Stack();
    deployStack(stack);
    expect(stack).toHaveResource('AWS::IoT::TopicRule', {
        TopicRulePayload: {
            Actions: [
                {
                    Lambda: {
                        FunctionArn: {
                            "Fn::GetAtt": [
                                "testiotlambdadynamodbstackIotToLambdaLambdaFunctionDFEAF894",
                                "Arn"
                            ]
                        }
                    }
                }
            ],
            Description: "Processing of DTC messages from the AWS Connected Vehicle Solution.",
            RuleDisabled: false,
            Sql: "SELECT * FROM 'connectedcar/dtc/#'"
        }
    });
});
test('check dynamo table properties', () => {
    const stack = new cdk.Stack();
    deployStack(stack);
    expect(stack).toHaveResource('AWS::DynamoDB::Table', {
        KeySchema: [
            {
                AttributeName: "id",
                KeyType: "HASH"
            }
        ],
        AttributeDefinitions: [
            {
                AttributeName: "id",
                AttributeType: "S"
            }
        ],
        BillingMode: "PAY_PER_REQUEST",
        SSESpecification: {
            SSEEnabled: true
        }
    });
});
test('check lambda function policy ', () => {
    const stack = new cdk.Stack();
    deployStack(stack);
    expect(stack).toHaveResource('AWS::IAM::Policy', {
        PolicyDocument: {
            Statement: [
                {
                    Action: [
                        "xray:PutTraceSegments",
                        "xray:PutTelemetryRecords"
                    ],
                    Effect: "Allow",
                    Resource: "*"
                },
                {
                    Action: [
                        "dynamodb:BatchGetItem",
                        "dynamodb:GetRecords",
                        "dynamodb:GetShardIterator",
                        "dynamodb:Query",
                        "dynamodb:GetItem",
                        "dynamodb:Scan",
                        "dynamodb:BatchWriteItem",
                        "dynamodb:PutItem",
                        "dynamodb:UpdateItem",
                        "dynamodb:DeleteItem"
                    ],
                    Effect: "Allow",
                    Resource: [
                        {
                            "Fn::GetAtt": [
                                "testiotlambdadynamodbstackLambdaToDynamoDBDynamoTableE17E5733",
                                "Arn"
                            ]
                        },
                        {
                            Ref: "AWS::NoValue"
                        }
                    ]
                }
            ],
            Version: "2012-10-17"
        }
    });
});
test('check properties', () => {
    const stack = new cdk.Stack();
    const construct = deployStack(stack);
    expect(construct.lambdaFunction !== null);
    expect(construct.dynamoTable !== null);
    expect(construct.iotTopicRule !== null);
});
test('check exception for Missing existingObj from props for deploy = false', () => {
    const stack = new cdk.Stack();
    const props = {
        iotTopicRuleProps: {
            topicRulePayload: {
                ruleDisabled: false,
                description: "Processing of DTC messages from the AWS Connected Vehicle Solution.",
                sql: "SELECT * FROM 'connectedcar/dtc/#'",
                actions: []
            }
        }
    };
    try {
        new lib_1.IotToLambdaToDynamoDB(stack, 'test-iot-lambda-integration', props);
    }
    catch (e) {
        expect(e).toBeInstanceOf(Error);
    }
});
//# sourceMappingURL=data:application/json;base64,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