# Author: Cameron F. Abrams, <cfa22@drexel.edu>

"""
Definition of the :class:`PSFContents` class for parsing and managing PSF topology files.
This class reads a PSF file, extracts topology information such as atoms, bonds, angles,
dihedrals, and patches, and provides methods for accessing and manipulating this data."""

import logging
import numpy as np
import os
from .psftopoelement import LineList
from .psfatom import PSFAtom, PSFAtomList
from .psfbond import PSFBondList
from .psfangle import PSFAngleList
from .psfdihedral import PSFDihedralList

from ..objs.ssbond import SSBond, SSBondList
from ..objs.link import Link, LinkList

logger=logging.getLogger(__name__)

def get_toppar_from_psf(filename):
    """
    Extracts the topology file names from a PSF file.
    This function reads the PSF file and looks for lines that contain topology information,
    specifically those that start with ' REMARKS topology'. It collects the unique topology file names
    and returns them as a list.  Such lines are written by the psfgen program, so this function expects
    to operate on a PSF file generated by psfgen.
    """
    lines=[]
    with open(filename,'r') as f:
        for line in f:
            if '!NATOM' in line:
                break
            lines.append(line)
    toppars=[]
    for l in lines:
        if l[0:17]==' REMARKS topology':
            tokens=l.split()
            top=tokens[2]
            if 'toppar_' in top:
                fn=os.path.basename(top)
                toppars.append(fn)
    return list(set(toppars))  

class PSFContents:
    """
    A class for parsing and managing PSF topology files.
    This class reads a PSF file, extracts topology information such as atoms, bonds, angles,
    dihedrals, and patches, and provides methods for accessing and manipulating this data.
    
    Attributes
    ----------
    atoms : PSFAtomList
        A list of atoms parsed from the PSF file, represented as instances of the :class:`PSFAtom` class.
    atomserials : list
        A list of serial numbers corresponding to the atoms in the PSF file.
    ssbonds : SSBondList
        A list of disulfide bonds parsed from the PSF file, represented as instances of the :class:`SSBond <pestifer.objs.ssbond.SSBond>` class.  These appear in psfgen-generated PSF files as patches of type 'DISU'.
    links : LinkList
        A list of links parsed from the PSF file, represented as instances of the :class:`Link <pestifer.objs.link.Link>` class. These appear in psfgen-generated PSF files as patches recognized as creating covalent bonds between atoms in two different residues OTHER than disulfides.
    patches : dict
        A dictionary containing patches defined in the PSF file, where keys are patch types (e.g., 'NTER', 'CTER') and values are lists of patch definitions.
    bonds : PSFBondList, optional
        A list of bonds parsed from the PSF file, represented as instances of the :class:`PSFBond <.psfbond.PSFBond>` class.
        This attribute is only set if the ``parse_topology`` parameter includes ``bonds``.
    angles : PSFAngleList, optional
        A list of angles parsed from the PSF file, represented as instances of the :class:`PSFAngle <.psfangle.PSFAngle>` class.
        This attribute is only set if the ``parse_topology`` parameter includes ``angles``.
    dihedrals : PSFDihedralList, optional
        A list of dihedrals parsed from the PSF file, represented as instances of the :class:`PSFDihedral <.psfdihedral.PSFDihedral>` class.
        This attribute is only set if the ``parse_topology`` parameter includes ``dihedrals``.

    Parameters
    ----------
    filename : str
        The path to the PSF file to be read.
    topology_segtypes : list, optional
        A list of segment types to include in the topology parsing. If provided, only atoms of these segment types will be included in the topology.
        Default is an empty list, which means all atoms will be included.
    parse_topology : list, optional
        A list of topology elements to parse from the PSF file. Possible values are 'bonds', 'angles', 'dihedrals', and 'impropers'.
        If provided, the class will parse the specified topology elements from the PSF file.
        Default is an empty list, which means no topology elements will be parsed.
    """
    def __init__(self,filename,topology_segtypes=[],parse_topology=[]):
        logger.debug(f'Reading {filename}...')
        with open(filename,'r') as f:
            psflines=f.read().split('\n')
        logger.debug(f'{len(psflines)} lines...')
        self.token_idx={}
        self.token_count={}
        self.patches={}
        for i,l in enumerate(psflines):
            toktst=[x.strip() for x in l.split()]
            if len(toktst)>=2 and toktst[1][0]=='!':
                token_name=toktst[1][2:]
                if token_name[-1]==':':
                    token_name=token_name[:-1]
                self.token_idx[token_name]=i
                self.token_count[token_name]=int(toktst[0])
            elif len(toktst)>1 and toktst[0]=="REMARKS":
                remark_type=toktst[1]
                if remark_type=='patch':
                    patch_type=toktst[2]
                    if patch_type not in ['NTER','CTER']:
                        if not patch_type in self.patches:
                            self.patches[patch_type]=[]
                        self.patches[patch_type].append(toktst[3:])
        self.token_lines={}
        for (k,l0),l1 in zip(self.token_idx.items(),list(self.token_idx.values())[1:]+[len(psflines)]):
            fl=l0+1
            ll=l1-1
            self.token_lines[k]=psflines[fl:ll]
        logger.debug(f'{len(self.token_lines)} tokensets:')
        logger.debug(f'{", ".join([x for x in self.token_lines.keys()])}')
        self.atoms=PSFAtomList([PSFAtom(x) for x in self.token_lines['ATOM']])
        self.atomserials=[x.serial for x in self.atoms]
        logger.debug(f'{len(self.atoms)} total atoms...')
        self.ssbonds=SSBondList([SSBond(L) for L in self.patches.get('DISU',[])])
        self.links=LinkList([])
        for patchtype,patchlist in self.patches.items():
            if patchtype in Link.patch_atomnames:
                for patch in patchlist:
                    self.links.append(Link([patchtype]+patch))
        if parse_topology:
            include_serials=[]
            if topology_segtypes:
                self.included_atoms=PSFAtomList([])
                for segtype in topology_segtypes:
                    sublist=self.atoms.get(segtype=segtype)
                    logger.debug(f'{len(sublist)} atoms of segtype {segtype}...')
                    self.included_atoms.extend(sublist)
                include_serials=[x.segtype in topology_segtypes for x in self.atoms]
                logger.debug(f'Including {include_serials.count(True)}/{len(include_serials)} topologically active atoms ({len(self.included_atoms)}) from segtypes {topology_segtypes}')
            if 'bonds' in parse_topology:
                self.bonds=PSFBondList(LineList(self.token_lines['BOND']),include_serials=include_serials)
                # logger.debug(f'Creating graph from {len(self.bonds)} bonds...')
                self.G=self.bonds.to_graph()
                logger.debug(f'extending atom instances with ligands...')
                self.add_ligands()
                logger.debug(f'done')
                # logger.debug(f'Parsed {len(self.bonds)} bonds.')
            if 'angles' in parse_topology:
                self.angles=PSFAngleList(LineList(self.token_lines['THETA']),include_serials=include_serials)
            if 'dihedrals' in parse_topology:
                self.dihedrals=PSFDihedralList(LineList(self.token_lines['PHI']),include_serials=include_serials)
            if 'impropers' in parse_topology:
                self.dihedrals=PSFDihedralList(LineList(self.token_lines['IMPHI']),include_serials=include_serials)
    
    def add_ligands(self):
        """
        Add ligands to each atom based on the bonds defined in the PSF file.
        This method iterates through the atoms and bonds, establishing ligand relationships
        by adding each bonded atom to the ligand list of the corresponding atom.
        It ensures that each atom's ligands are correctly populated based on the bonds defined in the PSF file.
        
        Raises
        ------
        AssertionError
            If the `bonds` attribute is not set or if the atom serial numbers do not match the expected indices.
        """
        assert hasattr(self,'bonds')
        for i,a in enumerate(self.atoms):
            a.ligands=[]
            assert a.serial==i+1
        for b in self.bonds:
            i,j=b.serial1,b.serial2
            aix,ajx=i-1,j-1#self.atomserials.index(i),self.atomserials.index(j)
            ai,aj=self.atoms[aix],self.atoms[ajx]
            ai.add_ligand(aj)
            aj.add_ligand(ai)

    def get_charge(self):
        """
        Calculate the total charge of the system by summing the charges of all atoms.
        
        Returns
        -------
        float
            The total charge of the system, calculated as the sum of the charges of all atoms in the PSF file.
        """
        return np.sum([x.charge for x in self.atoms])