"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.ConfigMap = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const configmap = require("fs");
const path = require("path");
const cdk8s_1 = require("cdk8s");
const constructs_1 = require("constructs");
const minimatch_1 = require("minimatch");
const base = require("./base");
const k8s = require("./imports/k8s");
const utils_1 = require("./utils");
class ImportedConfigMap extends constructs_1.Construct {
    constructor(scope, id, name) {
        super(scope, id);
        this.resourceType = 'configmaps';
        this._name = name;
    }
    get name() {
        return this._name;
    }
    get apiVersion() {
        return k8s.KubeConfigMap.GVK.apiVersion;
    }
    get apiGroup() {
        return '';
    }
    get kind() {
        return k8s.KubeConfigMap.GVK.kind;
    }
    get resourceName() {
        return this.name;
    }
}
/**
 * ConfigMap holds configuration data for pods to consume.
 */
class ConfigMap extends base.Resource {
    constructor(scope, id, props = {}) {
        super(scope, id);
        this.resourceType = 'configmaps';
        this._binaryData = {};
        this._data = {};
        this.immutable = props.immutable ?? false;
        this.apiObject = new k8s.KubeConfigMap(this, 'Resource', {
            metadata: props.metadata,
            // we need lazy here because we filter empty
            data: cdk8s_1.Lazy.any({ produce: () => this.synthesizeData() }),
            binaryData: cdk8s_1.Lazy.any({ produce: () => this.synthesizeBinaryData() }),
            immutable: this.immutable,
        });
        for (const [k, v] of Object.entries(props.data ?? {})) {
            this.addData(k, v);
        }
        for (const [k, v] of Object.entries(props.binaryData ?? {})) {
            this.addBinaryData(k, v);
        }
    }
    /**
     * Represents a ConfigMap created elsewhere.
     */
    static fromConfigMapName(scope, id, name) {
        return new ImportedConfigMap(scope, id, name);
    }
    /**
     * Adds a data entry to the config map.
     * @param key The key
     * @param value The value
     *
     * @throws if there is either a `data` or `binaryData` entry with the same key
     */
    addData(key, value) {
        this.verifyKeyAvailable(key);
        this._data[key] = value;
    }
    /**
     * The data associated with this config map.
     *
     * Returns an copy. To add data records, use `addData()` or `addBinaryData()`.
     */
    get data() {
        return { ...this._data };
    }
    /**
     * Adds a binary data entry to the config map. BinaryData can contain byte
     * sequences that are not in the UTF-8 range.
     * @param key The key
     * @param value The value
     *
     * @throws if there is either a `data` or `binaryData` entry with the same key
     */
    addBinaryData(key, value) {
        this.verifyKeyAvailable(key);
        this._binaryData[key] = value;
    }
    /**
     * The binary data associated with this config map.
     *
     * Returns a copy. To add data records, use `addBinaryData()` or `addData()`.
     */
    get binaryData() {
        return { ...this._binaryData };
    }
    /**
     * Adds a file to the ConfigMap.
     * @param localFile The path to the local file
     * @param key The ConfigMap key (default to the file name).
     */
    addFile(localFile, key) {
        key = key ?? path.basename(localFile);
        const value = configmap.readFileSync(localFile, 'utf-8');
        this.addData(key, value);
    }
    /**
     * Adds a directory to the ConfigMap.
     * @param localDir A path to a local directory
     * @param options Options
     */
    addDirectory(localDir, options = {}) {
        const exclude = options.exclude ?? [];
        const shouldInclude = (file) => {
            for (const pattern of exclude) {
                const mm = new minimatch_1.Minimatch(pattern);
                if (mm.match(file)) {
                    return false;
                }
            }
            return true;
        };
        const keyPrefix = options.keyPrefix ?? '';
        for (const file of configmap.readdirSync(localDir)) {
            const filePath = path.join(localDir, file);
            if (configmap.statSync(filePath).isDirectory()) {
                continue;
            }
            if (!shouldInclude(file)) {
                continue;
            }
            const relativeFilePath = keyPrefix + file;
            this.addFile(filePath, relativeFilePath);
        }
    }
    verifyKeyAvailable(key) {
        if (key in this._data || key in this._binaryData) {
            throw new Error(`unable to add a ConfigMap entry with key "${key}". It is already used`);
        }
    }
    synthesizeData() {
        return utils_1.undefinedIfEmpty(this._data);
    }
    synthesizeBinaryData() {
        return utils_1.undefinedIfEmpty(this._binaryData);
    }
}
exports.ConfigMap = ConfigMap;
_a = JSII_RTTI_SYMBOL_1;
ConfigMap[_a] = { fqn: "cdk8s-plus-23.ConfigMap", version: "2.2.5" };
//# sourceMappingURL=data:application/json;base64,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