# Copyright 2024-2025 Canonical Ltd.
#
# This program is free software; you can redistribute it and/or
# modify it under the terms of the GNU Lesser General Public
# License version 3 as published by the Free Software Foundation.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
# Lesser General Public License for more details.
#
# You should have received a copy of the GNU Lesser General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.

"""JSON schemas for validating Slurm configuration models."""

__all__ = [
    "ACCT_GATHER_CONFIG_MODEL_SCHEMA",
    "CGROUP_CONFIG_MODEL_SCHEMA",
    "DOWN_NODES_MODEL_SCHEMA",
    "DOWN_NODES_LIST_MODEL_SCHEMA",
    "FRONTEND_NODE_MODEL_SCHEMA",
    "FRONTEND_NODE_MAPPING_MODEL_SCHEMA",
    "GRES_MODEL_SCHEMA",
    "GRES_LIST_MODEL_SCHEMA",
    "GRES_MAPPING_MODEL_SCHEMA",
    "GRES_CONFIG_MODEL_SCHEMA",
    "NODE_MODEL_SCHEMA",
    "NODE_MAPPING_MODEL_SCHEMA",
    "NODESET_MODEL_SCHEMA",
    "NODESET_MAPPING_MODEL_SCHEMA",
    "OCI_CONFIG_MODEL_SCHEMA",
    "PARTITION_MODEL_SCHEMA",
    "PARTITION_MAPPING_MODEL_SCHEMA",
    "SLURM_CONFIG_MODEL_SCHEMA",
    "SLURMDBD_CONFIG_MODEL_SCHEMA",
]

_GLOBAL_SCHEMA_VERSION = "https://json-schema.org/draft/2020-12/schema"

# `acct_gather.conf` data model schema.

ACCT_GATHER_CONFIG_MODEL_SCHEMA = {
    "$schema": _GLOBAL_SCHEMA_VERSION,
    "type": "object",
    "properties": {
        "energyipmicalcadjustment": {"type": "boolean"},
        "energyipmipowersensors": {
            "type": "object",
            "patternProperties": {
                "^.*$": {
                    "type": "array",
                    "items": {"type": "integer"},
                    "uniqueItems": True,
                }
            },
        },
        "energyipmiusername": {"type": "string"},
        "energyipmipassword": {"type": "string"},
        "energyipmifrequency": {"type": "integer", "minimum": 0},
        "energyipmitimeout": {"type": "integer", "minimum": 0},
        "profilehdf5dir": {"type": "string"},
        "profilehdf5default": {
            "type": "array",
            "items": {
                "type": "string",
                "enum": [
                    "all",
                    "none",
                    "energy",
                    "filesystem",
                    "network",
                    "task",
                ],
            },
            "uniqueItems": True,
        },
        "profileinfluxdbdatabase": {"type": "string"},
        "profileinfluxdbdefault": {
            "type": "array",
            "items": {
                "type": "string",
                "enum": [
                    "all",
                    "none",
                    "energy",
                    "filesystem",
                    "network",
                    "task",
                ],
            },
            "uniqueItems": True,
        },
        "profileinfluxdbhost": {"type": "string"},
        "profileinfluxdbpass": {"type": "string"},
        "profileinfluxdbrtpolicy": {"type": "string"},
        "profileinfluxdbuser": {"type": "string"},
        "profileinfluxdbtimeout": {"type": "integer", "minimum": 0},
        "infinibandofedport": {"type": "integer", "minimum": 0},
        "sysfsinterfaces": {
            "type": "array",
            "items": {"type": "string"},
            "uniqueItems": True,
        },
    },
    "additionalProperties": False,
}

# `cgroup.conf` data model schema.

CGROUP_CONFIG_MODEL_SCHEMA = {
    "$schema": _GLOBAL_SCHEMA_VERSION,
    "type": "object",
    "properties": {
        "cgroupmountpoint": {"type": "string"},
        "cgroupplugin": {
            "type": "string",
            "enum": ["cgroup/v1", "cgroup/v2", "autodetect", "disabled"],
        },
        "systemdtimeout": {"type": "integer", "minimum": 0},
        "ignoresystemd": {"type": "boolean"},
        "ignoreystemdonfailure": {"type": "boolean"},
        "enablecontrollers": {"type": "boolean"},
        "allowedramspace": {"type": "integer", "minimum": 0},
        "allowedswapspace": {"type": "integer", "minimum": 0},
        "constraincores": {"type": "boolean"},
        "constraindevices": {"type": "boolean"},
        "constrainramspace": {"type": "boolean"},
        "constrainswapspace": {"type": "boolean"},
        "maxrampercent": {"type": "number", "minimum": 0, "maximum": 100},
        "maxswappercent": {"type": "number", "minimum": 0, "maximum": 100},
        "memoryswappiness": {"type": "integer", "minimum": 0},
        "minramspace": {"type": "integer", "minimum": 0},
        "signalchildrenprocesses": {"type": "boolean"},
    },
    "additionalProperties": False,
}

# `gres.conf` data model schemas.

GRES_MODEL_SCHEMA = {
    "$schema": _GLOBAL_SCHEMA_VERSION,
    "type": "object",
    "properties": {
        "name": {"type": "string"},
        "autodetect": {
            "type": "string",
            "enum": [
                "nrt",
                "nvidia",
                "nvml",
                "off",
                "oneapi",
                "rsmi",
            ],
        },
        "count": {
            "oneOf": [
                {"type": "string"},
                {"type": "integer", "minimum": 0},
            ],
        },
        "cores": {
            "type": "array",
            "items": {
                "oneOf": [
                    {"type": "string"},
                    {"type": "integer"},
                ],
            },
            "uniqueItems": True,
        },
        "file": {"type": "string"},
        "flags": {
            "type": "array",
            "items": {
                "type": "string",
                "enum": [
                    "countonly",
                    "explicit",
                    "one_sharing",
                    "all_sharing",
                    "nvidia_gpu_env",
                    "amd_gpu_env",
                    "intel_gpu_env",
                    "opencl_env",
                    "no_gpu_env",
                ],
            },
            "uniqueItems": True,
        },
        "links": {
            "type": "array",
            "items": {
                "type": "integer",
                "minimum": -1,
            },
        },
        "multiplefiles": {
            "type": "array",
            "items": {
                "type": "string",
            },
            "uniqueItems": True,
        },
        "nodename": {
            "type": "string",
        },
        "type": {"type": "string"},
    },
    "additionalProperties": False,
}

GRES_LIST_MODEL_SCHEMA = {
    "$schema": _GLOBAL_SCHEMA_VERSION,
    "type": "array",
    "items": GRES_MODEL_SCHEMA,
    "uniqueItems": True,
}

GRES_MAPPING_MODEL_SCHEMA = {
    "$schema": _GLOBAL_SCHEMA_VERSION,
    "type": "object",
    "patternProperties": {"^.+$": GRES_LIST_MODEL_SCHEMA},
}

GRES_CONFIG_MODEL_SCHEMA = {
    "$schema": _GLOBAL_SCHEMA_VERSION,
    "type": "object",
    "properties": {
        "autodetect": GRES_MODEL_SCHEMA["properties"]["autodetect"],
        "gres": GRES_MAPPING_MODEL_SCHEMA,
    },
    "additionalProperties": False,
}

# `oci.conf` data model schemas.

OCI_CONFIG_MODEL_SCHEMA = {
    "$schema": _GLOBAL_SCHEMA_VERSION,
    "type": "object",
    "properties": {
        "containerpath": {"type": "string"},
        "createenvfile": {"type": "string", "enum": ["null", "newline", "disable"]},
        "debugflags": {
            "type": "array",
            "items": {
                "type": "string",
                "enum": [
                    "accrue",
                    "agent",
                    "auditrpcs",
                    "backfill",
                    "backfillmap",
                    "burstbuffer",
                    "cgroup",
                    "conmgr",
                    "cpu_bind",
                    "cpufrequency",
                    "data",
                    "dbd_agent",
                    "dependency",
                    "energy",
                    "federation",
                    "frontend",
                    "gres",
                    "hetjob",
                    "gang",
                    "glob_silence",
                    "jobaccountgather",
                    "jobcomp",
                    "jobcontainer",
                    "license",
                    "network",
                    "networkraw",
                    "nodefeatures",
                    "no_conf_hash",
                    "power",
                    "priority",
                    "profile",
                    "protocol",
                    "reservation",
                    "route",
                    "script",
                    "selecttype",
                    "steps",
                    "switch",
                    "tls",
                    "tracejobs",
                    "triggers",
                ],
            },
            "uniqueItems": True,
        },
        "disablecleanup": {"type": "boolean"},
        "disablehooks": {"type": "array", "items": {"type": "string"}, "uniqueItems": True},
        "envexclude": {"type": "string"},
        "mountspooldir": {"type": "string"},
        "runtimeenvexclude": {"type": "string"},
        "filedebug": {
            "type": "string",
            "enum": [
                "quiet",
                "fatal",
                "error",
                "info",
                "verbose",
                "debug",
                "debug2",
                "debug3",
                "debug4",
                "debug5",
            ],
        },
        "ignorefileconfigjson": {"type": "boolean"},
        "runtimecreate": {"type": "string"},
        "runtimedelete": {"type": "string"},
        "runtimekill": {"type": "string"},
        "runtimequery": {"type": "string"},
        "runtimerun": {"type": "string"},
        "runtimestart": {"type": "string"},
        "srunpath": {"type": "string"},
        "srunargs": {"type": "array", "items": {"type": "string"}, "uniqueItems": True},
        "stdiodebug": {
            "type": "string",
            "enum": [
                "quiet",
                "fatal",
                "error",
                "info",
                "verbose",
                "debug",
                "debug2",
                "debug3",
                "debug4",
                "debug5",
            ],
        },
        "syslogdebug": {
            "type": "string",
            "enum": [
                "quiet",
                "fatal",
                "error",
                "info",
                "verbose",
                "debug",
                "debug2",
                "debug3",
                "debug4",
                "debug5",
            ],
        },
    },
    "additionalProperties": False,
}

# `slurm.conf` data model schemas.

DOWN_NODES_MODEL_SCHEMA = {
    "$schema": _GLOBAL_SCHEMA_VERSION,
    "type": "object",
    "properties": {
        "downnodes": {"type": "array", "items": {"type": "string"}, "uniqueItems": True},
        "reason": {"type": "string"},
        "state": {"type": "string", "enum": ["down", "drain", "fail", "failing", "future"]},
    },
    "additionalProperties": False,
}

FRONTEND_NODE_MODEL_SCHEMA = {
    "$schema": _GLOBAL_SCHEMA_VERSION,
    "type": "object",
    "properties": {
        "frontendname": {"type": "string"},
        "frontendaddr": {"type": "string"},
        "allowgroups": {"type": "array", "items": {"type": "string"}, "uniqueItems": True},
        "allowusers": {"type": "array", "items": {"type": "string"}, "uniqueItems": True},
        "denygroups": {"type": "array", "items": {"type": "string"}, "uniqueItems": True},
        "denyusers": {"type": "array", "items": {"type": "string"}, "uniqueItems": True},
        "port": {"type": "integer", "minimum": 0},
        "reason": {"type": "string"},
        "state": {"type": "string", "enum": ["down", "drain", "fail", "failing", "unknown"]},
    },
    "additionalProperties": False,
}

NODE_MODEL_SCHEMA = {
    "$schema": _GLOBAL_SCHEMA_VERSION,
    "type": "object",
    "properties": {
        "nodename": {"type": "string"},
        "nodehostname": {"type": "string"},
        "nodeaddr": {"type": "string"},
        "bcastaddr": {"type": "string"},
        "boards": {"type": "integer", "minimum": 1},
        "corespeccount": {"type": "integer", "minimum": 1},
        "corespersocket": {"type": "integer", "minimum": 1},
        "cpubind": {"type": "string", "enum": ["none", "socket", "ldom", "core", "thread"]},
        "cpus": {"type": "integer", "minimum": 0},
        "cpuspeclist": {"type": "array", "items": {"type": "integer"}, "uniqueItems": True},
        "features": {"type": "array", "items": {"type": "string"}, "uniqueItems": True},
        "gres": {"type": "array", "items": {"type": "string"}, "uniqueItems": True},
        "memspeclimit": {"type": "integer", "minimum": 1},
        "port": {"type": "integer"},
        "procs": {"type": "integer", "minimum": 1},
        "realmemory": {"type": "integer", "minimum": 1},
        "reason": {"type": "string"},
        "sockets": {"type": "integer", "minimum": 1},
        "socketsperboard": {"type": "integer", "minimum": 1},
        "state": {
            "type": "string",
            "enum": ["cloud", "down", "drain", "fail", "failing", "future", "unknown"],
        },
        "threadspercore": {"type": "integer", "minimum": 1},
        "tmpdisk": {"type": "integer", "minimum": 1},
        "weight": {"type": "integer", "minimum": 1},
    },
    "additionalProperties": False,
}

NODESET_MODEL_SCHEMA = {
    "$schema": _GLOBAL_SCHEMA_VERSION,
    "type": "object",
    "properties": {
        "nodeset": {"type": "string"},
        "feature": {"type": "string"},
        "nodes": {"type": "array", "items": {"type": "string"}, "uniqueItems": True},
    },
    "additionalProperties": False,
}

PARTITION_MODEL_SCHEMA = {
    "$schema": _GLOBAL_SCHEMA_VERSION,
    "type": "object",
    "properties": {
        "partitionname": {"type": "string"},
        "allocnodes": {"type": "array", "items": {"type": "string"}, "uniqueItems": True},
        "allowaccounts": {"type": "array", "items": {"type": "string"}, "uniqueItems": True},
        "allowgroups": {"type": "array", "items": {"type": "string"}, "uniqueItems": True},
        "allowqos": {"type": "array", "items": {"type": "string"}, "uniqueItems": True},
        "alternate": {"type": "string"},
        "cpubind": {"type": "string", "enum": ["none", "socket", "ldom", "core", "thread"]},
        "default": {"type": "boolean"},
        "defaulttime": {
            "oneOf": [{"type": "integer", "minimum": 0}, {"type": "string"}],
        },
        "defcpupergpu": {"type": "integer", "minimum": 0},
        "defmempercpu": {"type": "integer", "minimum": 0},
        "defmempergpu": {"type": "integer", "minimum": 0},
        "defmempernode": {"type": "integer", "minimum": 0},
        "denyaccounts": {"type": "array", "items": {"type": "string"}, "uniqueItems": True},
        "denyqos": {"type": "array", "items": {"type": "string"}, "uniqueItems": True},
        "disablerootjobs": {"type": "boolean"},
        "exclusivetopo": {"type": "boolean"},
        "exclusiveuser": {"type": "boolean"},
        "gracetime": {"type": "number", "minimum": 0},
        "hidden": {"type": "boolean"},
        "lln": {"type": "boolean"},
        "maxcpuspernode": {"type": "integer", "minimum": 0},
        "maxcpuspersocket": {"type": "integer", "minimum": 0},
        "maxmempercpu": {"type": "integer", "minimum": 0},
        "maxmempernode": {"type": "integer", "minimum": 0},
        "maxnodes": {
            "oneOf": [
                {"type": "integer", "minimum": -1},
                {"type": "string", "enum": ["unlimited"]},
            ]
        },
        "maxtime": {
            "oneOf": [{"type": "integer", "minimum": 0}, {"type": "string"}],
        },
        "minnodes": {"type": "integer", "minimum": 0},
        "nodes": {"type": "array", "items": {"type": "string"}, "uniqueItems": True},
        "oversubscribe": {"type": "string", "enum": ["exclusive", "force", "yes", "no"]},
        "overtimelimit": {
            "oneOf": [
                {"type": "integer", "minimum": 0, "maximum": 65533},
                {"type": "string", "enum": ["unlimited"]},
            ]
        },
        "powerdownonidle": {"type": "boolean"},
        "preemptmode": {"type": "string", "enum": ["off", "cancel", "gang", "requeue", "suspend"]},
        "priorityjobfactor": {"type": "integer", "maximum": 65533},
        "prioritytier": {"type": "integer", "maximum": 65533},
        "qos": {"type": "string"},
        "reqresv": {"type": "boolean"},
        "resumetimeout": {"type": "integer", "minimum": 0},
        "rootonly": {"type": "boolean"},
        "selecttypeparameters": {
            "type": "array",
            "items": {
                "type": "string",
                "enum": ["cr_core", "cr_core_memory", "cr_socket", "cr_socket_memory"],
            },
        },
        "state": {"type": "string", "enum": ["up", "down", "drain", "inactive"]},
        "suspendtime": {
            "oneOf": [
                {"type": "integer", "minimum": -1},
                {"type": "string", "enum": ["infinite"]},
            ],
        },
        "suspendtimeout": {"type": "integer", "minimum": 0},
        "tresbillingweights": {
            "type": "object",
            "patternProperties": {"^.*$": {"type": "number"}},
        },
    },
    "additionalProperties": False,
}

NODE_MAPPING_MODEL_SCHEMA = {
    "$schema": _GLOBAL_SCHEMA_VERSION,
    "type": "object",
    "patternProperties": {"^.+$": NODE_MODEL_SCHEMA},
}

DOWN_NODES_LIST_MODEL_SCHEMA = {
    "$schema": _GLOBAL_SCHEMA_VERSION,
    "type": "array",
    "items": DOWN_NODES_MODEL_SCHEMA,
    "uniqueItems": True,
}

FRONTEND_NODE_MAPPING_MODEL_SCHEMA = {
    "$schema": _GLOBAL_SCHEMA_VERSION,
    "type": "object",
    "patternProperties": {"^.+$": FRONTEND_NODE_MODEL_SCHEMA},
}

NODESET_MAPPING_MODEL_SCHEMA = {
    "$schema": _GLOBAL_SCHEMA_VERSION,
    "type": "object",
    "patternProperties": {"^.+$": NODESET_MODEL_SCHEMA},
}

PARTITION_MAPPING_MODEL_SCHEMA = {
    "$schema": _GLOBAL_SCHEMA_VERSION,
    "type": "object",
    "patternProperties": {"^.+$": PARTITION_MODEL_SCHEMA},
}

SLURM_CONFIG_MODEL_SCHEMA = {
    "$schema": _GLOBAL_SCHEMA_VERSION,
    "type": "object",
    "properties": {
        "accountingstoragebackuphost": {"type": "string"},
        "accountingstorageenforce": {
            "type": "array",
            "items": {
                "type": "string",
                "enum": [
                    "all",
                    "associations",
                    "limits",
                    "nojobs",
                    "nosteps",
                    "qos",
                    "safe",
                    "wckeys",
                ],
            },
            "uniqueItems": True,
        },
        "accountingstorageexternalhost": {"type": "string"},
        "accountingstoragehost": {"type": "string"},
        "accountingstorageparameters": {
            "type": "object",
            "properties": {
                "max_step_records": {"type": "integer"},
                "ssl_cert": {"type": "string"},
                "ssl_ca": {"type": "string"},
                "ssl_capath": {"type": "string"},
                "ssl_key": {"type": "string"},
                "ssl_cipher": {
                    "type": "string",
                    "items": {"type": "string"},
                    "uniqueItems": True,
                },
            },
            "additionalProperties": False,
        },
        "accountingstoragepass": {"type": "string"},
        "accountingstorageport": {"type": "integer", "minimum": 0},
        "accountingstoragetres": {
            "type": "array",
            "items": {"type": "string"},
            "uniqueItems": True,
        },
        "accountingstoragetype": {"type": "string", "enum": ["accounting_storage/slurmdbd"]},
        "accountingstorageuser": {"type": "string"},
        "accountingstoreflags": {
            "type": "array",
            "items": {
                "type": "string",
                "enum": ["job_comment", "job_env", "job_extra", "job_script", "no_stdio"],
            },
        },
        "acctgathernodefreq": {"type": "integer", "minimum": 0},
        "acctgatherenergytype": {
            "type": "string",
            "enum": [
                "acct_gather_energy/gpu",
                "acct_gather_energy/ipmi",
                "acct_gather_energy/pm_counters",
                "acct_gather_energy/rapl",
                "acct_gather_energy/xcc",
            ],
        },
        "acctgatherinterconnecttype": {
            "type": "string",
            "enum": ["acct_gather_interconnect/ofed", "acct_gather_interconnect/sysfs"],
        },
        "acctgatherfilesystemtype": {"type": "string", "enum": ["acct_gather_filesystem/lustre"]},
        "acctgatherprofiletype": {
            "type": "string",
            "enum": ["acct_gather_profile/hdf5", "acct_gather_profile/influxdb"],
        },
        "allowspecresourcesusage": {"type": "boolean"},
        "authalttypes": {"type": "array", "items": {"type": "string", "enum": ["auth/jwt"]}},
        "authaltparameters": {
            "type": "object",
            "properties": {
                "disable_token_creation": {"type": "boolean"},
                "max_token_lifespan": {"type": "integer"},
                "jwks": {"type": "string"},
                "jwt_key": {"type": "string"},
                "userclaimfield": {"type": "string"},
            },
            "additionalProperties": False,
        },
        "authinfo": {
            "type": "object",
            "properties": {
                "cred_expire": {"type": "integer"},
                "socket": {"type": "string"},
                "ttl": {"type": "integer"},
                "use_client_ids": {"type": "boolean"},
            },
            "additionalProperties": False,
        },
        "authtype": {"type": "string", "enum": ["auth/munge", "auth/slurm"]},
        "batchstarttimeout": {"type": "integer", "minimum": 0},
        "bcastexclude": {"type": "array", "items": {"type": "string"}, "uniqueItems": True},
        "bcastparameters": {
            "type": "object",
            "properties": {
                "destdir": {"type": "string"},
                "compression": {"type": "string", "enum": ["lz4", "none"]},
                "send_libs": {"type": "boolean"},
            },
            "additionalProperties": False,
        },
        "burstbuffertype": {
            "type": "string",
            "enum": ["burst_buffer/datawarp", "burst_buffer/lua", "burst_buffer/none"],
        },
        "clifilterplugins": {
            "type": "array",
            "items": {
                "type": "string",
                "enum": ["cli_filter/lua", "cli_filter/syslog", "cli_filter/user_defaults"],
            },
            "uniqueItems": True,
        },
        "clustername": {"type": "string"},
        "communicationparameters": {
            "type": "object",
            "properties": {
                "block_null_hash": {"type": "boolean"},
                "disableipv4": {"type": "boolean"},
                "enableipv6": {"type": "boolean"},
                "getnameinfo_cache_timeout": {"type": "integer", "minimum": 0},
                "keepaliveinterval": {"type": "integer", "minimum": 0},
                "keepaliveprobes": {"type": "integer", "minimum": 0},
                "keepalivetime": {"type": "integer", "minimum": 0},
                "noctldinaddrany": {"type": "boolean"},
                "noinaddrany": {"type": "boolean"},
            },
            "additionalProperties": False,
        },
        "completewait": {"type": "integer", "minimum": 0, "maximum": 65533},
        "cpufreqdef": {
            "type": "string",
            "enum": ["conservative", "ondemand", "performance", "powersave"],
        },
        "cpufreqgovernors": {
            "type": "array",
            "items": {
                "type": "string",
                "enum": [
                    "conservative",
                    "ondemand",
                    "performance",
                    "powersave",
                    "schedutil",
                    "userspace",
                ],
            },
            "uniqueItems": True,
        },
        "credtype": {"type": "string", "enum": ["cred/munge", "cred/slurm"]},
        "debugflags": {
            "type": "array",
            "items": {
                "type": "string",
                "enum": [
                    "accrue",
                    "agent",
                    "auditrpcs",
                    "backfill",
                    "backfillmap",
                    "burstbuffer",
                    "cgroup",
                    "conmgr",
                    "cpu_bind",
                    "cpufrequency",
                    "data",
                    "dbd_agent",
                    "dependency",
                    "energy",
                    "federation",
                    "frontend",
                    "gres",
                    "hetjob",
                    "gang",
                    "glob_silence",
                    "jobaccountgather",
                    "jobcomp",
                    "jobcontainer",
                    "license",
                    "network",
                    "networkraw",
                    "nodefeatures",
                    "no_conf_hash",
                    "power",
                    "priority",
                    "profile",
                    "protocol",
                    "reservation",
                    "route",
                    "script",
                    "selecttype",
                    "steps",
                    "switch",
                    "tls",
                    "tracejobs",
                    "triggers",
                ],
            },
            "uniqueItems": True,
        },
        "defcpupergpu": {"type": "integer", "minimum": 0},
        "defmempercpu": {"type": "integer", "minimum": 0},
        "defmempergpu": {"type": "integer", "minimum": 0},
        "defmempernode": {"type": "integer", "minimum": 0},
        "dependencyparameters": {
            "type": "object",
            "properties": {
                "disable_remote_singleton": {"type": "boolean"},
                "kill_invalid_depend": {"type": "boolean"},
                "max_depend_depth": {"type": "integer", "minimum": 0},
            },
            "additionalProperties": False,
        },
        "disablerootjobs": {"type": "boolean"},
        "eiotimeout": {"type": "integer", "minimum": 0, "maximum": 65533},
        "enforcepartlimits": {"type": "string", "enum": ["all", "any", "no"]},
        "epilog": {"type": "string"},
        "epilogmsgtime": {"type": "integer", "minimum": 0},
        "epilogslurmctld": {"type": "string"},
        "fairsharedampeningfactor": {"type": "integer", "minimum": 1},
        "federationparameters": {
            "type": "object",
            "properties": {
                "fed_display": {"type": "boolean"},
            },
            "additionalProperties": False,
        },
        "firstjobid": {"type": "integer", "minimum": 1},
        "getenvtimeout": {"type": "integer", "minimum": 0},
        "grestypes": {"type": "array", "items": {"type": "string"}, "uniqueItems": True},
        "groupupdateforce": {"type": "boolean"},
        "groupupdatetime": {"type": "integer", "minimum": 0},
        "gpufreqdef": {
            "type": "array",
            "items": {
                "oneOf": [
                    {"type": "string"},
                    {"type": "integer"},
                ]
            },
            "uniqueItems": True,
        },
        "hashplugin": {"type": "string", "enum": ["hash/k12", "hash/sha3"]},
        "healthcheckinterval": {"type": "integer", "minimum": 0},
        "healthchecknodestate": {
            "type": "array",
            "items": {
                "type": "string",
                "enum": ["alloc", "any", "cycle", "idle", "nondrained_idle", "mixed"],
            },
            "uniqueItems": True,
        },
        "healthcheckprogram": {"type": "string"},
        "inactivelimit": {"type": "integer", "minimum": 0, "maximum": 65533},
        "include": {
            "type": "array",
            "items": {
                "type": "string",
            },
            "uniqueItems": True,
        },
        "interactivestepoptions": {"type": "string"},
        "jobacctgathertype": {
            "type": "string",
            "enum": ["jobacct_gather/cgroup", "jobacct_gather/linux", "jobacct_gather/none"],
        },
        "jobacctgatherfrequency": {
            "type": "object",
            "properties": {
                "task": {"type": "integer", "minimum": 0},
                "energy": {"type": "integer", "minimum": 0},
                "network": {"type": "integer", "minimum": 0},
                "filesystem": {"type": "integer", "minimum": 0},
            },
            "additionalProperties": False,
        },
        "jobacctgatherparams": {
            "type": "object",
            "properties": {
                "noshared": {"type": "boolean"},
                "usepss": {"type": "boolean"},
                "overmemorykill": {"type": "boolean"},
                "disablegpuacct": {"type": "boolean"},
            },
            "additionalProperties": False,
        },
        "jobcomphost": {"type": "string"},
        "jobcomploc": {"type": "string"},
        "jobcompparams": {
            "type": "object",
            "properties": {
                "flush_timeout": {"type": "integer", "minimum": 0},
                "poll_interval": {"type": "integer", "minimum": 0},
                "requeue_on_msg_timeout": {"type": "boolean"},
                "topic": {"type": "string"},
            },
            "additionalProperties": False,
        },
        "jobcomppass": {"type": "string"},
        "jobcompport": {"type": "integer", "minimum": 0},
        "jobcomptype": {
            "type": "string",
            "enum": [
                "jobcomp/none",
                "jobcomp/elasticsearch",
                "jobcomp/filetxt",
                "jobcomp/kafka",
                "jobcomp/lua",
                "jobcomp/mysql",
                "jobcomp/script",
            ],
        },
        "jobcompuser": {"type": "string"},
        "jobcontainertype": {"type": "string", "enum": ["job_container/tmpfs"]},
        "jobfileappend": {"type": "boolean"},
        "jobrequeue": {"type": "boolean"},
        "jobsubmitplugins": {
            "type": "array",
            "items": {
                "type": "string",
                "enum": [
                    "all_partitions",
                    "defaults",
                    "logging",
                    "lua",
                    "partition",
                    "pbs",
                    "require_timelimit",
                ],
            },
            "uniqueItems": True,
        },
        "killonbadexit": {"type": "boolean"},
        "killwait": {"type": "integer", "minimum": 0, "maximum": 65533},
        "maxbatchrequeue": {"type": "integer", "minimum": 0},
        "nodefeaturesplugins": {
            "type": "array",
            "items": {
                "type": "string",
                "enum": [
                    "node_features/knl_cray",
                    "node_features/knl_generic",
                    "node_features/helpers",
                ],
            },
            "uniqueItems": True,
        },
        "launchparameters": {
            "type": "object",
            "properties": {
                "batch_step_set_cpu_freq": {"type": "boolean"},
                "cray_net_exclusive": {"type": "boolean"},
                "enable_nss_slurm": {"type": "boolean"},
                "lustre_no_flush": {"type": "boolean"},
                "mem_sort": {"type": "boolean"},
                "mpir_use_nodeaddr": {"type": "boolean"},
                "disable_send_gids": {"type": "boolean"},
                "slurmstepd_memlock_all": {"type": "boolean"},
                "test_exec": {"type": "boolean"},
                "use_interactive_step": {"type": "boolean"},
                "ulimit_pam_adopt": {"type": "boolean"},
            },
            "additionalProperties": False,
        },
        "licenses": {
            "type": "object",
            "patternProperties": {
                "^.*$": {
                    "type": "integer",
                    "minimum": -1,
                },
            },
        },
        "logtimeformat": {
            "type": "string",
            "enum": [
                "iso8601",
                "iso8601,format_stderr",
                "iso8601_ms",
                "iso8601_ms,format_stderr",
                "rfc5424",
                "rfc5424,format_stderr",
                "rfc5424_ms",
                "rfc5424_ms,format_stderr",
                "rfc3339",
                "rfc3339,format_stderr",
                "clock",
                "clock,format_stderr",
                "short",
                "short,format_stderr",
                "thread_id",
                "thread_id,format_stderr",
            ],
        },
        "maildomain": {"type": "string"},
        "mailprog": {"type": "string"},
        "maxarraysize": {"type": "integer", "minimum": 0, "maximum": 4000001},
        "maxdbdmsgs": {"type": "integer", "minimum": 10000},
        "maxjobcount": {"type": "integer", "minimum": 0},
        "maxjobid": {"type": "integer", "minimum": 0},
        "maxmempercpu": {"type": "integer", "minimum": 0},
        "maxmempernode": {"type": "integer", "minimum": 0},
        "maxnodecount": {"type": "integer", "minimum": 0, "maximum": 65533},
        "maxstepcount": {"type": "integer", "minimum": 0},
        "maxtaskspernode": {"type": "integer", "minimum": 0, "maximum": 65533},
        "mcsparameters": {
            "type": "object",
            "properties": {
                "ondemand": {"type": "boolean"},
                "enforced": {"type": "boolean"},
                "noselect": {"type": "boolean"},
                "select": {"type": "boolean"},
                "ondemandselect": {"type": "boolean"},
                "privatedata": {"type": "boolean"},
                "mcs_plugin_parameters": {
                    "type": "array",
                    "items": {"type": "string"},
                    "uniqueItems": True,
                },
            },
            "additionalProperties": False,
        },
        "mcsplugin": {
            "type": "string",
            "enum": ["mcs/none", "mcs/account", "mcs/group", "mcs/user", "mcs/label"],
        },
        "messagetimeout": {"type": "integer", "minimum": 0},
        "minjobage": {"type": "integer", "minimum": 0},
        "mpidefault": {"type": "string", "enum": ["pmi2", "pmix", "none"]},
        "mpiparams": {
            "type": "object",
            "properties": {
                "ports": {"type": "string"},
                "disable_slurm_hydra_bootstrap": {"type": "boolean"},
            },
            "additionalProperties": False,
        },
        "overtimelimit": {
            "oneOf": [
                {"type": "integer", "minimum": 0, "maximum": 65533},
                {"type": "string", "enum": ["unlimited"]},
            ]
        },
        "plugindir": {
            "type": "array",
            "items": {"type": "string"},
            "uniqueItems": True,
        },
        "plugstackconfig": {"type": "string"},
        "preemptmode": {
            "type": "array",
            "items": {
                "type": "string",
                "enum": ["off", "cancel", "gang", "requeue", "suspend", "within"],
            },
        },
        "preemptparameters": {
            "type": "object",
            "properties": {
                "min_exempt_priority": {"type": "integer"},
                "reclaim_licenses": {"type": "boolean"},
                "reorder_count": {"type": "integer", "minimum": 0},
                "send_user_signal": {"type": "boolean"},
                "strict_order": {"type": "boolean"},
                "suspend_grace_time": {"type": "boolean"},
                "youngest_first": {"type": "boolean"},
            },
            "additionalProperties": False,
        },
        "preempttype": {
            "type": "string",
            "enum": ["preempt/partition_prio", "preempt/qos"],
        },
        "preemptexempttime": {
            "oneOf": [{"type": "integer", "minimum": -1}, {"type": "string"}],
        },
        "prepparameters": {"type": "string"},
        "prepplugins": {
            "type": "array",
            "items": {"type": "string"},
            "uniqueItems": True,
        },
        "prioritycalcperiod": {"type": "integer", "minimum": 0},
        "prioritydecayhalflife": {
            "oneOf": [{"type": "integer", "minimum": 0}, {"type": "string"}]
        },
        "priorityfavorsmall": {"type": "boolean"},
        "priorityflags": {
            "type": "array",
            "items": {
                "type": "string",
                "enum": [
                    "accrue_always",
                    "calculate_running",
                    "depth_oblivious",
                    "no_fair_tree",
                    "incr_only",
                    "max_tres",
                    "no_normal_all",
                    "no_normal_assoc",
                    "no_normal_part",
                    "no_normal_qos",
                    "no_normal_tres",
                    "small_relative_to_time",
                ],
            },
            "uniqueItems": True,
        },
        "prioritymaxage": {"oneOf": [{"type": "integer", "minimum": 0}, {"type": "string"}]},
        "priorityparameters": {"type": "string"},
        "prioritysitefactorparameters": {"type": "string"},
        "prioritysitefactorplugin": {"type": "string"},
        "prioritytype": {"type": "string", "enum": ["priority/basic", "priority/multifactor"]},
        "priorityusageresetperiod": {
            "type": "string",
            "enum": [
                "none",
                "now",
                "daily",
                "weekly",
                "monthly",
                "quarterly",
                "yearly",
            ],
        },
        "priorityweightage": {"type": "integer", "minimum": 0},
        "priorityweightassoc": {"type": "integer", "minimum": 0},
        "priorityweightfairshare": {"type": "integer", "minimum": 0},
        "priorityweightjobsize": {"type": "integer", "minimum": 0},
        "priorityweightpartition": {"type": "integer", "minimum": 0},
        "priorityweightqos": {"type": "integer", "minimum": 0},
        "priorityweighttres": {
            "type": "object",
            "patternProperties": {"^.*$": {"type": "integer"}},
        },
        "privatedata": {
            "type": "array",
            "items": {
                "type": "string",
                "enum": [
                    "accounts",
                    "events",
                    "jobs",
                    "nodes",
                    "partitions",
                    "reservations",
                    "usage",
                    "users",
                ],
            },
            "uniqueItems": True,
        },
        "proctracktype": {
            "type": "string",
            "enum": ["proctrack/cgroup", "proctrack/linuxproc", "proctrack/pgid"],
        },
        "prolog": {"type": "string"},
        "prologepilogtimeout": {"type": "integer", "minimum": 0},
        "prologflags": {
            "type": "array",
            "items": {
                "type": "string",
                "enum": [
                    "alloc",
                    "contain",
                    "deferbatch",
                    "nohold",
                    "forcerequeueonfail",
                    "runinjob",
                    "serial",
                    "x11",
                ],
            },
            "uniqueItems": True,
        },
        "prologslurmctld": {"type": "string"},
        "propagateprioprocess": {"type": "integer", "minimum": 0, "maximum": 2},
        "propagateresourcelimits": {
            "type": "array",
            "items": {
                "type": "string",
                "enum": [
                    "all",
                    "none",
                    "as",
                    "core",
                    "cpu",
                    "data",
                    "fsize",
                    "memlock",
                    "nofile",
                    "nproc",
                    "rss",
                    "stack",
                ],
            },
            "uniqueItems": True,
        },
        "propagateresourcelimitsexcept": {
            "type": "array",
            "items": {
                "type": "string",
                "enum": [
                    "all",
                    "none",
                    "as",
                    "core",
                    "cpu",
                    "data",
                    "fsize",
                    "memlock",
                    "nofile",
                    "nproc",
                    "rss",
                    "stack",
                ],
            },
            "uniqueItems": True,
        },
        "rebootprogram": {"type": "string"},
        "reconfigflags": {
            "type": "array",
            "items": {
                "type": "string",
                "enum": ["keeppartinfo", "keeppartstate", "keeppowersavesettings"],
            },
            "uniqueItems": True,
        },
        "requeueexit": {
            "type": "array",
            "items": {"oneOf": [{"type": "integer", "minimum": 0}, {"type": "string"}]},
            "uniqueItems": True,
        },
        "requeueexithold": {
            "type": "array",
            "items": {"oneOf": [{"type": "integer", "minimum": 0}, {"type": "string"}]},
            "uniqueItems": True,
        },
        "resumefailprogram": {"type": "string"},
        "resumeprogram": {"type": "string"},
        "resumerate": {"type": "integer", "minimum": 0},
        "resumetimeout": {"type": "integer", "minimum": 0},
        "resvepilog": {"type": "string"},
        "resvoverrun": {
            "oneOf": [
                {"type": "integer", "minimum": 0, "maximum": 65533},
                {"type": "string", "enum": ["unlimited"]},
            ]
        },
        "resvprolog": {"type": "string"},
        "returntoservice": {"type": "integer", "minimum": 0, "maximum": 2},
        "schedulerparameters": {
            "type": "boolean",
            "properties": {
                "allow_zero_lic": {"type": "boolean"},
                "assoc_limit_stop": {"type": "boolean"},
                "batch_sched_delay": {"type": "integer", "minimum": 0},
                "bb_array_stage_cnt": {"type": "integer", "minimum": 0},
                "bf_allow_magnetic_slot": {"type": "boolean"},
                "bf_busy_nodes": {"type": "boolean"},
                "bf_continue": {"type": "boolean"},
                "bf_hetjob_immediate": {"type": "boolean"},
                "bf_hetjob_prio": {"type": "string", "enum": ["min", "avg", "max"]},
                "bf_interval": {"type": "integer", "minimum": -1, "maximum": 10800},
                "bf_job_part_count_reserve": {"type": "integer", "minimum": 0, "maximum": 100000},
                "bf_licenses": {"type": "boolean"},
                "bf_max_job_array_resv": {"type": "integer", "minimum": 0, "maximum": 1000},
                "bf_max_job_assoc": {"type": "integer", "minimum": 0, "maximum": 1000000},
                "bf_max_job_part": {"type": "integer", "minimum": 0, "maximum": 1000000},
                "bf_max_job_start": {"type": "integer", "minimum": 0, "maximum": 10000},
                "bf_max_job_test": {"type": "integer", "minimum": 1, "maximum": 1000000},
                "bf_max_job_user": {"type": "integer", "minimum": 0},
                "bf_max_job_user_part": {"type": "integer", "minimum": 0},
                "bf_max_time": {"type": "integer", "minimum": 1, "maximum": 3600},
                "bf_min_age_reserve": {"type": "integer", "minimum": 2592000},
                "bf_min_prio_reserve": {"type": "integer", "minimum": 0, "maximum": 2**63},
                "bf_node_space_size": {"type": "integer", "minimum": 1, "maximum": 2000000},
                "bf_one_resv_per_job": {"type": "boolean"},
                "bf_resolution": {"type": "integer", "minimum": 1, "maximum": 3600},
                "bf_running_job_reserve": {"type": "boolean"},
                "bf_topopt_enable": {"type": "boolean"},
                "bf_topopt_iterations": {"type": "integer", "minimum": 0},
                "bf_window": {"type": "integer", "minimum": 1, "maximum": 43200},
                "bf_window_linear": {"type": "integer", "minimum": 0},
                "bf_yield_interval": {"type": "integer", "minimum": 1, "maximum": 10000000},
                "bf_yield_sleep": {"type": "integer", "minimum": 1, "maximum": 10000000},
                "build_queue_timeout": {"type": "integer", "minimum": 0},
                "correspond_after_task_cnt": {"type": "integer", "minimum": 0},
                "default_queue_depth": {"type": "integer", "minimum": 0},
                "defer": {"type": "boolean"},
                "defer_batch": {"type": "boolean"},
                "delay_boot": {"type": "integer", "minimum": 0},
                "disable_job_shrink": {"type": "boolean"},
                "disable_hetjob_steps": {"type": "boolean"},
                "enable_hetjob_steps": {"type": "boolean"},
                "enable_job_state_cache": {"type": "boolean"},
                "enable_user_top": {"type": "boolean"},
                "extra_constraints": {"type": "boolean"},
                "ignore_constraint_validation": {"type": "boolean"},
                "ignore_numa": {"type": "boolean"},
                "ignore_prefer_validation": {"type": "boolean"},
                "max_array_tasks": {"type": "integer", "minimum": 0},
                "max_rpc_cnt": {"type": "integer", "minimum": 0},
                "max_sched_time": {"type": "integer", "minimum": 0},
                "max_script_size": {"type": "integer", "minimum": 0},
                "max_submit_line_size": {"type": "integer", "minimum": 0, "maximum": 2},
                "max_switch_wait": {"type": "integer", "minimum": 0},
                "no_backup_scheduling": {"type": "boolean"},
                "no_env_cache": {"type": "boolean"},
                "nohold_on_prolog_fail": {"type": "boolean"},
                "pack_serial_at_end": {"type": "boolean"},
                "partition_job_depth": {"type": "integer"},
                "reduce_completing_frag": {"type": "boolean"},
                "requeue_setup_env_fail": {"type": "boolean"},
                "salloc_wait_nodes": {"type": "boolean"},
                "sbatch_wait_nodes": {"type": "boolean"},
                "sched_interval": {"type": "integer", "minimum": -1},
                "sched_max_job_start": {"type": "integer", "minimum": 0},
                "sched_min_interval": {"type": "integer", "minimum": 0},
                "spec_cores_first": {"type": "boolean"},
                "step_retry_count": {"type": "integer", "minimum": 0},
                "step_retry_time": {"type": "integer", "minimum": 0},
                "time_min_as_soft_limit": {"type": "boolean"},
                "whole_hetjob": {"type": "boolean"},
            },
            "additionalProperties": False,
        },
        "schedulertimeslice": {"type": "integer", "minimum": 5, "maximum": 65533},
        "schedulertype": {
            "type": "string",
            "enum": ["sched/backfill", "sched/builtin"],
        },
        "scronparameters": {
            "type": "object",
            "properties": {
                "enable": {"type": "boolean"},
                "explicit_scancel": {"type": "boolean"},
            },
            "additionalProperties": False,
        },
        "selecttype": {
            "type": "string",
            "enum": ["select/cons_tres", "select/linear"],
        },
        "selecttypeparameters": {
            "type": "object",
            "properties": {
                "cr_cpu": {"type": "boolean"},
                "cr_cpu_memory": {"type": "boolean"},
                "cr_core": {"type": "boolean"},
                "cr_core_memory": {"type": "boolean"},
                "cr_one_task_per_core": {"type": "boolean"},
                "cr_core_default_dist_block": {"type": "boolean"},
                "cr_lln": {"type": "boolean"},
                "cr_pack_nodes": {"type": "boolean"},
                "ll_shared_gres": {"type": "boolean"},
                "cr_socket": {"type": "boolean"},
                "cr_socket_memory": {"type": "boolean"},
                "cr_memory": {"type": "boolean"},
                "multiple_sharing_gres_pj": {"type": "boolean"},
                "enforce_binding_gres": {"type": "boolean"},
                "one_task_per_sharing_gres": {"type": "boolean"},
            },
            "additionalProperties": False,
        },
        "slurmctldaddr": {"type": "string"},
        "slurmctlddebug": {
            "type": "string",
            "enum": [
                "quiet",
                "fatal",
                "error",
                "info",
                "verbose",
                "debug",
                "debug2",
                "debug3",
                "debug4",
                "debug5",
            ],
        },
        "slurmctldhost": {
            "type": "array",
            "items": {"type": "string"},
            "uniqueItems": True,
        },
        "slurmctldlogfile": {"type": "string"},
        "slurmctldparameters": {
            "type": "object",
            "properties": {
                "allow_user_triggers": {"type": "boolean"},
                "cloud_dns": {"type": "boolean"},
                "conmgr_max_connections": {"type": "integer", "minimum": 0},
                "conmgr_threads": {"type": "integer", "minimum": 0},
                "conmgr_use_poll": {"type": "boolean"},
                "conmgr_connect_timeout": {"type": "integer", "minimum": 0},
                "conmgr_read_timeout": {"type": "integer", "minimum": 0},
                "conmgr_wait_write_delay": {"type": "integer", "minimum": 0},
                "conmgr_write_timeout": {"type": "integer", "minimum": 0},
                "disable_triggers": {"type": "boolean"},
                "enable_configless": {"type": "boolean"},
                "idle_on_node_suspend": {"type": "boolean"},
                "node_reg_mem_percent": {"type": "integer", "minimum": 0},
                "no_quick_restart": {"type": "boolean"},
                "power_save_interval": {"type": "integer", "minimum": 0},
                "power_save_min_interval": {"type": "integer", "minimum": 0},
                "max_powered_nodes": {"type": "integer", "minimum": 0},
                "max_dbd_msg_action": {"type": "string", "enum": ["discard", "exit"]},
                "reboot_from_controller": {"type": "boolean"},
                "rl_bucket_size": {"type": "integer", "minimum": 0},
                "rl_enable": {"type": "boolean"},
                "rl_log_freq": {"type": "integer", "minimum": -1},
                "rl_refill_period": {"type": "integer", "minimum": 0},
                "rl_refill_rate": {"type": "integer", "minimum": 0},
                "rl_table_size": {"type": "integer", "minimum": 0},
                "enable_stepmgr": {"type": "boolean"},
                "user_resv_delete": {"type": "boolean"},
                "validate_nodeaddr_threads": {"type": "integer", "minimum": 0, "maximum": 64},
            },
            "additionalProperties": False,
        },
        "slurmctldpidfile": {"type": "string"},
        "slurmctldport": {"oneOf": [{"type": "integer", "minimum": 0}, {"type": "string"}]},
        "slurmctldprimaryoffprog": {"type": "string"},
        "slurmctldprimaryonprog": {"type": "string"},
        "slurmctldsyslogdebug": {
            "type": "string",
            "enum": [
                "quiet",
                "fatal",
                "error",
                "info",
                "verbose",
                "debug",
                "debug2",
                "debug3",
                "debug4",
                "debug5",
            ],
        },
        "slurmctldtimeout": {"type": "integer", "minimum": 0, "maximum": 65533},
        "slurmddebug": {
            "type": "string",
            "enum": [
                "quiet",
                "fatal",
                "error",
                "info",
                "verbose",
                "debug",
                "debug2",
                "debug3",
                "debug4",
                "debug5",
            ],
        },
        "slurmdlogfile": {"type": "string"},
        "slurmdparameters": {
            "type": "object",
            "properties": {
                "allow_ecores": {"type": "boolean"},
                "config_overrides": {"type": "boolean"},
                "conmgr_max_connections": {"type": "integer", "minimum": 0},
                "conmgr_threads": {"type": "integer", "minimum": 0},
                "conmgr_use_poll": {"type": "boolean"},
                "conmgr_connect_timeout": {"type": "integer", "minimum": 0},
                "conmgr_read_timeout": {"type": "integer", "minimum": 0},
                "conmgr_wait_write_delay": {"type": "integer", "minimum": 0},
                "conmgr_write_timeout": {"type": "integer", "minimum": 0},
                "l3cache_as_socket": {"type": "boolean"},
                "numa_node_as_socket": {"type": "boolean"},
                "shutdown_on_reboot": {"type": "boolean"},
                "contain_spank": {"type": "boolean"},
            },
            "additionalProperties": False,
        },
        "slurmdpidfile": {"type": "string"},
        "slurmdport": {"type": "integer", "minimum": 0},
        "slurmdspooldir": {"type": "string"},
        "slurmdsyslogdebug": {
            "type": "string",
            "enum": [
                "quiet",
                "fatal",
                "error",
                "info",
                "verbose",
                "debug",
                "debug2",
                "debug3",
                "debug4",
                "debug5",
            ],
        },
        "slurmdtimeout": {"type": "integer", "minimum": 0, "maximum": 65533},
        "slurmduser": {"type": "string"},
        "slurmschedlogfile": {"type": "string"},
        "slurmschedloglevel": {"type": "boolean"},
        "slurmuser": {"type": "string"},
        "srunepilog": {"type": "string"},
        "srunportrange": {"type": "string"},
        "srunprolog": {"type": "string"},
        "statesavelocation": {"type": "string"},
        "suspendexcnodes": {
            "type": "array",
            "items": {"type": "string"},
            "uniqueItems": True,
        },
        "suspendexcparts": {
            "type": "array",
            "items": {"type": "string"},
            "uniqueItems": True,
        },
        "suspendexcstates": {
            "type": "array",
            "items": {
                "type": "string",
                "enum": [
                    "cloud",
                    "down",
                    "drain",
                    "dynamic_future",
                    "dynamic_norm",
                    "fail",
                    "invalid_reg",
                    "maintenance",
                    "not_responding",
                    "perfctrs",
                    "planned",
                    "reserved",
                ],
            },
            "uniqueItems": True,
        },
        "suspendprogram": {"type": "string"},
        "suspendrate": {"type": "integer", "minimum": 0},
        "suspendtime": {
            "oneOf": [{"type": "integer", "minimum": -1}, {"type": "string", "enum": ["infinite"]}]
        },
        "suspendtimeout": {"type": "integer", "minimum": 0},
        "switchparameters": {
            "type": "object",
            "properties": {
                "vnis": {"type": "string"},
                "tcs": {
                    "type": "array",
                    "items": {
                        "type": "string",
                        "enum": [
                            "dedicated_access",
                            "low_latency",
                            "bulk_data",
                            "best_effort",
                            "tc_dedicated_access",
                            "tc_low_latency",
                            "tc_bulk_data",
                            "tc_best_effort",
                        ],
                    },
                    "uniqueItems": True,
                },
                "single_node_vni": {"type": "string", "enum": ["all", "user", "none"]},
                "job_vni": {"type": "string", "enum": ["all", "user", "none"]},
                "adjust_limits": {"type": "boolean"},
                "no_adjust_limits": {"type": "boolean"},
                "hwcoll_addrs_per_job": {"type": "boolean"},
                "hwcoll_num_nodes": {"type": "boolean"},
                "fm_url": {"type": "boolean"},
                "fm_auth": {"type": "string", "enum": ["oauth", "basic"]},
                "fm_authdir": {"type": "string"},
                "def_txqs": {"type": "integer", "minimum": 0},
                "res_txqs": {"type": "integer", "minimum": 0, "maximum": 1024},
                "max_txqs": {"type": "integer", "minimum": 0, "maximum": 1024},
                "def_tgqs": {"type": "integer", "minimum": 0},
                "res_tgqs": {"type": "integer", "minimum": 0, "maximum": 512},
                "max_tgqs": {"type": "integer", "minimum": 0, "maximum": 512},
                "def_eqs": {"type": "integer", "minimum": 0},
                "res_eqs": {"type": "integer", "minimum": 0, "maximum": 2047},
                "max_eqs": {"type": "integer", "minimum": 0, "maximum": 2047},
                "def_cts": {"type": "integer", "minimum": 0},
                "res_cts": {"type": "integer", "minimum": 0, "maximum": 2047},
                "max_cts": {"type": "integer", "minimum": 0, "maximum": 2047},
                "def_tles": {"type": "integer", "minimum": 0},
                "res_tles": {"type": "integer", "minimum": 0, "maximum": 2048},
                "max_tles": {"type": "integer", "minimum": 0, "maximum": 2048},
                "def_ptes": {"type": "integer", "minimum": 0},
                "res_ptes": {"type": "integer", "minimum": 0, "maximum": 2048},
                "max_ptes": {"type": "integer", "minimum": 0, "maximum": 2048},
                "def_les": {"type": "integer", "minimum": 0},
                "res_les": {"type": "integer", "minimum": 0, "maximum": 16384},
                "max_les": {"type": "integer", "minimum": 0, "maximum": 16348},
                "def_acs": {"type": "integer", "minimum": 0},
                "res_acs": {"type": "integer", "minimum": 0, "maximum": 1022},
                "max_acs": {"type": "integer", "minimum": 0},
                "imex_channel_count": {"type": "integer", "minimum": 0},
            },
            "additionalProperties": False,
        },
        "switchtype": {"type": "string", "enum": ["switch/hpe_slingshot", "switch/nvidia_imex"]},
        "taskepilog": {"type": "string"},
        "taskplugin": {
            "type": "array",
            "items": {"type": "string", "enum": ["task/affinity", "task/cgroup"]},
            "uniqueItems": True,
        },
        "taskpluginparam": {
            "type": "object",
            "properties": {
                "cores": {"type": "boolean"},
                "none": {"type": "boolean"},
                "sockets": {"type": "boolean"},
                "threads": {"type": "boolean"},
                "slurmdoffspec": {"type": "boolean"},
                "verbose": {"type": "boolean"},
                "autobind": {"type": "string", "enum": ["cores", "sockets", "threads"]},
            },
            "additionalProperties": False,
        },
        "taskprolog": {"type": "string"},
        "tcptimeout": {"type": "integer", "minimum": 0},
        "tmpfs": {"type": "string"},
        "topologyparam": {
            "type": "object",
            "properties": {
                "dragonfly": {"type": "boolean"},
                "routepart": {"type": "boolean"},
                "switchasnoderank": {"type": "boolean"},
                "routetree": {"type": "boolean"},
                "topomaxsizeunroll": {"type": "integer", "minimum": -1},
                "topooptional": {"type": "boolean"},
            },
            "additionalProperties": False,
        },
        "topologyplugin": {
            "type": "array",
            "items": {
                "type": "string",
                "enum": [
                    "topology/3d_torus",
                    "topology/block",
                    "topology/default",
                    "topology/tree",
                ],
            },
            "uniqueItems": True,
        },
        "trackwckey": {"type": "boolean"},
        "treewidth": {"type": "integer", "minimum": 0, "maximum": 65533},
        "unkillablestepprogram": {"type": "string"},
        "unkillablesteptimeout": {"type": "integer", "minimum": 0},
        "usepam": {"type": "boolean"},
        "vsizefactor": {"type": "integer", "minimum": 0, "maximum": 65533},
        "waittime": {"type": "integer", "minimum": 0, "maximum": 65533},
        "x11parameters": {
            "type": "object",
            "properties": {
                "home_xauthority": {"type": "boolean"},
            },
            "additionalProperties": False,
        },
        "downnodes": DOWN_NODES_LIST_MODEL_SCHEMA,
        "frontendnodes": FRONTEND_NODE_MAPPING_MODEL_SCHEMA,
        "nodes": NODE_MAPPING_MODEL_SCHEMA,
        "nodesets": NODESET_MAPPING_MODEL_SCHEMA,
        "partitions": PARTITION_MAPPING_MODEL_SCHEMA,
    },
    "additionalProperties": False,
}


# `slurmdbd.conf` data model schema.

SLURMDBD_CONFIG_MODEL_SCHEMA = {
    "$schema": _GLOBAL_SCHEMA_VERSION,
    "type": "object",
    "properties": {
        "allownodefacct": {"type": "boolean"},
        "allresourcesabsolute": {"type": "boolean"},
        "archivedir": {"type": "string"},
        "archiveevents": {"type": "boolean"},
        "archivejobs": {"type": "boolean"},
        "archiveresvs": {"type": "boolean"},
        "archivescript": {"type": "string"},
        "archivesteps": {"type": "boolean"},
        "archivesuspend": {"type": "boolean"},
        "archivetxn": {"type": "boolean"},
        "archiveusage": {"type": "boolean"},
        "authalttypes": {
            "type": "array",
            "items": {
                "type": "string",
                "enum": ["auth/jwt"],
            },
            "uniqueItems": True,
        },
        "authaltparameters": {
            "type": "object",
            "properties": {
                "jwks": {"type": "string"},
                "jwt_key": {"type": "string"},
            },
            "additionalProperties": False,
        },
        "authinfo": {
            "type": "object",
            "properties": {
                "socket": {"type": "string"},
                "ttl": {"type": "integer", "minimum": 0},
                "use_client_ids": {"type": "boolean"},
            },
            "additionalProperties": False,
        },
        "authtype": {
            "type": "string",
            "enum": ["auth/munge", "auth/slurm"],
        },
        "commitdelay": {"type": "integer", "minimum": 0},
        "communicationparameters": {
            "type": "object",
            "properties": {
                "disableipv4": {"type": "boolean"},
                "enableipv6": {"type": "boolean"},
                "keepaliveinterval": {"type": "integer", "minimum": 0},
                "keepaliveprobes": {"type": "integer", "minimum": 0},
                "keepalivetime": {"type": "integer", "minimum": 0},
            },
            "additionalProperties": False,
        },
        "dbdaddr": {"type": "string"},
        "dbdbackuphost": {"type": "string"},
        "dbdhost": {"type": "string"},
        "dbdport": {"type": "integer", "minimum": 0},
        "debugflags": {
            "type": "array",
            "items": {
                "type": "string",
                "enum": [
                    "auditrpcs",
                    "db_archive",
                    "db_assoc",
                    "db_event",
                    "db_job",
                    "db_qos",
                    "db_query",
                    "db_reservation",
                    "db_resource",
                    "db_step",
                    "db_tres",
                    "db_usage",
                    "db_wckey",
                    "federation",
                    "network",
                    "networkraw",
                    "tls",
                ],
            },
            "uniqueItems": True,
        },
        "debuglevel": {
            "type": "string",
            "enum": [
                "quiet",
                "fatal",
                "error",
                "info",
                "verbose",
                "debug",
                "debug2",
                "debug3",
                "debug4",
                "debug5",
            ],
        },
        "debuglevelsyslog": {
            "type": "string",
            "enum": [
                "quiet",
                "fatal",
                "error",
                "info",
                "verbose",
                "debug",
                "debug2",
                "debug3",
                "debug4",
                "debug5",
            ],
        },
        "defaultqos": {"type": "string"},
        "disablecoorddbd": {"type": "boolean"},
        "hashplugin": {"type": "string", "enum": ["hash/k12", "hash/sha3"]},
        "logfile": {"type": "string"},
        "logtimeformat": {
            "type": "string",
            "enum": [
                "iso8601",
                "iso8601,format_stderr",
                "iso8601_ms",
                "iso8601_ms,format_stderr",
                "rfc5424",
                "rfc5424,format_stderr",
                "rfc5424_ms",
                "rfc5424_ms,format_stderr",
                "rfc3339",
                "rfc3339,format_stderr",
                "clock",
                "clock,format_stderr",
                "short",
                "short,format_stderr",
                "thread_id",
                "thread_id,format_stderr",
            ],
        },
        "maxquerytimerange": {"oneOf": [{"type": "integer", "minimum": 0}, {"type": "string"}]},
        "messagetimeout": {"type": "integer", "minimum": 0},
        "parameters": {
            "type": "object",
            "properties": {"preservecaseuser": {"type": "boolean"}},
            "additionalProperties": False,
        },
        "pidfile": {"type": "string"},
        "plugindir": {
            "type": "array",
            "items": {
                "type": "string",
            },
            "uniqueItems": True,
        },
        "privatedata": {
            "type": "array",
            "items": {
                "type": "string",
                "enum": ["accounts", "events", "jobs", "reservations", "usage", "users"],
            },
            "uniqueItems": True,
        },
        "purgeeventafter": {"type": "string"},
        "purgejobafter": {"type": "string"},
        "purgeresvafter": {"type": "string"},
        "purgestepafter": {"type": "string"},
        "purgesuspendafter": {"type": "string"},
        "purgetxnafter": {"type": "string"},
        "purgeusageafter": {"type": "string"},
        "slurmuser": {"type": "string"},
        "storagebackuphost": {"type": "string"},
        "storagehost": {"type": "string"},
        "storageloc": {"type": "string"},
        "storageparameters": {
            "type": "object",
            "properties": {
                "ssl_cert": {"type": "string"},
                "ssl_ca": {"type": "string"},
                "ssl_capath": {"type": "string"},
                "ssl_key": {"type": "string"},
                "ssl_cipher": {
                    "type": "array",
                    "items": {"type": "string"},
                    "uniqueItems": True,
                },
            },
            "additionalProperties": False,
        },
        "storagepass": {"type": "string"},
        "storageport": {"type": "integer", "minimum": 0},
        "storagetype": {"type": "string", "enum": ["accounting_storage/mysql"]},
        "storageuser": {"type": "string"},
        "tcptimeout": {"type": "integer", "minimum": 0},
        "trackslurmctlddown": {"type": "boolean"},
        "trackwckey": {"type": "boolean"},
    },
    "additionalProperties": False,
}
