
"""
Contributors can be viewed at:
http://svn.secondlife.com/svn/linden/projects/2008/pyogp/lib/base/trunk/CONTRIBUTORS.txt 

$LicenseInfo:firstyear=2008&license=apachev2$

Copyright 2009, Linden Research, Inc.

Licensed under the Apache License, Version 2.0.
You may obtain a copy of the License at:
    http://www.apache.org/licenses/LICENSE-2.0
or in 
    http://svn.secondlife.com/svn/linden/projects/2008/pyogp/lib/base/LICENSE.txt

$/LicenseInfo$
"""

# standard python libs
import unittest
import os

# pyogp
from pyogp.lib.base.message_manager import MessageManager
from pyogp.lib.base.caps import Capability
from pyogp.lib.base.message.message import Message, Block
from pyogp.lib.base.tests.mockup_net import MockupUDPServer, MockupUDPClient
from pyogp.lib.base.message.circuit import Host
from pyogp.lib.base.message.udpdispatcher import UDPDispatcher

# pyogp tests
import pyogp.lib.base.tests.config 

from eventlet import api

class TestMessageManager(unittest.TestCase):

    def setUp(self):
        self.host = Host((MockupUDPServer(), 80))
        self.message_manager = MessageManager(self.host,
                                              capabilities={'EventQueueGet' : Capability('EventQueueGet', 'http://127.0.0.1')})

        
    def tearDown(self):
        pass

    def test_start_stop_monitors(self):
        self.message_manager.start_monitors()
        api.sleep(0)
        self.assertTrue(self.message_manager._is_running)
        self.assertTrue(self.message_manager.event_queue._running)
        self.message_manager.stop_monitors()
        api.sleep(2)
        self.assertFalse(self.message_manager._is_running)
        self.assertTrue(self.message_manager.event_queue.stopped)
        self.assertFalse(self.message_manager.event_queue._running)
        
    def test_enqueue_message(self):
        message = Message('TestMessage1',
                          Block('TestBlock1',
                                Test1 = 0),
                          Block('NeighborBlock',
                                Test0 = 0,
                                Test1 = 1,
                                Test2 = 2))
        self.message_manager.enqueue_message(message,
                                             reliable = True)
        self.assertEqual(self.message_manager.outgoing_queue[0][0].name,
                         'TestMessage1')
        self.assertTrue(self.message_manager.outgoing_queue[0][1])
        message2 = Message('TestMessage2',
                          Block('TestBlock1',
                                Test1 = 0),
                          Block('NeighborBlock',
                                Test0 = 0,
                                Test1 = 1,
                                Test2 = 2))
        self.message_manager.enqueue_message(message2,
                                             reliable = False,
                                             now = True)
        self.assertEqual(self.message_manager.outgoing_queue[0][0].name,
                         'TestMessage2')
        self.assertFalse(self.message_manager.outgoing_queue[0][1])
        

    def test_send_udp_message(self):
        self.message_manager.udp_dispatcher = UDPDispatcher(MockupUDPClient(),
                                                            self.message_manager.settings,
                                                            self.message_manager.message_handler)
        message = Message('PacketAck',
                      Block('Packets', ID=0x00000003))
        buf =  self.message_manager.send_udp_message(message)
        assert buf == \
               '\x00' + '\x00\x00\x00\x01' + '\x00' + '\xff\xff\xff\xfb' + \
               '\x01' + '\x03\x00\x00\x00', \
               'Received: ' + repr(buf) + '  ' + \
               'Expected: ' + repr('\x00' + '\x00\x00\x00\x01' + '\x00' + \
                            '\xff\xff\xff\xfb' + '\x01' + '\x03\x00\x00\x00')

    def test_custom_message_template_count(self):

        message_manager = MessageManager(self.host,
                                            message_template = open(os.path.join(os.path.dirname(__file__), 'mock_message_template.msg')))

        # the custom message_template.msg should pass all the way through, and should have 210 messages
        self.assertEquals(len(message_manager.udp_dispatcher.udp_deserializer.template_dict.template_list), 210)

    def test_embedded_message_template_count(self):

        message_manager = MessageManager(self.host,
                                            message_template = open(os.path.join(os.path.dirname(__file__), 'mock_message_template.msg')))

        # the message_template.msg should default to the embedded instance
        # and should have > 470 messages
        assert len(self.message_manager.udp_dispatcher.udp_deserializer.template_dict.template_list) >= 475

    def test_custom_message_xml(self):

        message_manager = MessageManager(self.host,
                                            message_xml = open(os.path.join(os.path.dirname(__file__), 'mock_message.xml')))

        # the custom message.xml should be used
        self.assertEquals(message_manager.message_xml.__dict__, {'messageBans': {'untrusted': {}, 'trusted': {}}, 'messages': {'PacketAck': {'flavor': 'template', 'trusted-sender': False}}, 'serverDefaults': {'simulator': 'template'}, 'parsed_llsd': {'serverDefaults': {'simulator': 'template'}, 'messages': {'PacketAck': {'flavor': 'template', 'trusted-sender': False}}, 'capBans': {'MapLayer': False, 'UploadBakedTexture': True}, 'maxQueuedEvents': 100, 'messageBans': {'untrusted': {}, 'trusted': {}}}, 'maxQueuedEvents': 100, 'raw_llsd': '<?xml version="1.0"?>\n<llsd>\n    <map>\n\t\t  <key>serverDefaults</key>\n            <!--\n\t\t\t\t    a map of server names to default message transport\n\t\t\t\t-->\n\t\t  <map>\n\t\t\t\t<key>simulator</key>\n\t\t\t\t<string>template</string>\n\n\n\t\t  </map>\n\t\t  <key>messages</key>\n            <!--\n\t\t\t\t    a map of individual message names that override defaults\n\t\t\t\t-->\n\t\t  <map>\n\t\t\t\t<!--\n\t\t\t\t\tCircuit related messages\n\t\t\t\t-->\n\t\t\t\t<key>PacketAck</key>\n\t\t\t\t<map>\n\t\t\t\t\t<key>flavor</key>\n\t\t\t\t\t<string>template</string>\n\t\t\t\t\t<key>trusted-sender</key>\n\t\t\t\t\t<boolean>false</boolean>\n\t\t\t\t</map>\n\n\t\t  </map>\n  \t  \t<key>capBans</key>\n    \t<map>\n\t\t\t<key>MapLayer</key>\n\t\t\t<boolean>false</boolean>\n\n\t\t\t<key>UploadBakedTexture</key>\n\t\t\t<boolean>true</boolean>\n\t\t</map>\n\n\t\t<key>messageBans</key>\n\t\t<map>\n\t\t\t<key>trusted</key>\n\t\t\t<map>\n\t\t\t</map>\n\t\t\n\t\t\t<key>untrusted</key>\n\t\t\t<map>\n\t\t\t</map>\n\t\t</map>\n\n\t\t<key>maxQueuedEvents</key>\n\t\t<integer>100</integer>\n    </map>\n</llsd>\n', 'capBans': {'MapLayer': False, 'UploadBakedTexture': True}})


    def test_embedded_message_xml(self):

        self.assertEquals(self.message_manager.message_xml.__dict__, {'messageBans': {'untrusted': {}, 'trusted': {}}, 'messages': {'ParcelProperties': {'flavor': 'llsd', 'trusted-sender': True}, 'ImprovedTerseObjectUpdate': {'flavor': 'template', 'trusted-sender': False}, 'EstablishAgentCommunication': {'flavor': 'llsd', 'trusted-sender': True}, 'FetchInventoryDescendents': {'flavor': 'template', 'trusted-sender': False}, 'CrossedRegion': {'flavor': 'llsd', 'trusted-sender': True}, 'ForceCloseChatterBoxSession': {'flavor': 'llsd', 'trusted-sender': True}, 'EconomyDataRequest': {'flavor': 'template', 'trusted-sender': False}, 'CoarseLocationUpdate': {'flavor': 'template', 'trusted-sender': True, 'only-send-latest': True}, 'CompletePingCheck': {'flavor': 'template', 'trusted-sender': False}, 'avatarpicksrequest': {'service_name': 'avatar-pick', 'builder': 'template', 'trusted-sender': False}, 'ParcelVoiceInfo': {'flavor': 'llsd', 'trusted-sender': True}, 'ObjectUpdateCached': {'flavor': 'template', 'trusted-sender': False}, 'ObjectUpdateCompressed': {'flavor': 'template', 'trusted-sender': False}, 'RoutedMoneyBalanceReply': {'flavor': 'llsd', 'trusted-sender': False}, 'EdgeDataPacket': {'flavor': 'template', 'trusted-sender': True}, 'LandStatReply': {'flavor': 'llsd', 'trusted-sender': False}, 'AvatarAppearance': {'flavor': 'template', 'trusted-sender': False}, 'EnableSimulator': {'flavor': 'llsd', 'trusted-sender': True}, 'DirLandReply': {'flavor': 'llsd', 'trusted-sender': True}, 'UseCircuitCode': {'flavor': 'template', 'trusted-sender': False}, 'AvatarAnimation': {'flavor': 'template', 'trusted-sender': False}, 'RequiredVoiceVersion': {'flavor': 'llsd', 'trusted-sender': True}, 'ImagePacket': {'flavor': 'template', 'trusted-sender': False}, 'StartGroupProposal': {'flavor': 'llsd', 'trusted-sender': False}, 'StartPingCheck': {'flavor': 'template', 'trusted-sender': False}, 'AgentUpdate': {'flavor': 'template', 'trusted-sender': False}, 'RequestTrustedCircuit': {'flavor': 'template', 'trusted-sender': True}, 'ChatterBoxSessionEventReply': {'flavor': 'llsd', 'trusted-sender': True}, 'GroupProposalBallot': {'flavor': 'llsd', 'trusted-sender': False}, 'RpcScriptRequestInboundForward': {'flavor': 'llsd', 'trusted-sender': False}, 'ParcelObjectOwnersReply': {'flavor': 'llsd', 'trusted-sender': True}, 'OpenCircuit': {'flavor': 'template', 'trusted-sender': False}, 'ParcelVoiceInfoRequest': {'flavor': 'llsd', 'trusted-sender': False}, 'CloseCircuit': {'flavor': 'template', 'trusted-sender': False}, 'PacketAck': {'flavor': 'template', 'trusted-sender': False}, 'ChatterBoxInvitation': {'flavor': 'llsd', 'trusted-sender': True}, 'CompleteAgentMovement': {'flavor': 'template', 'trusted-sender': False}, 'ScriptRunningReply': {'flavor': 'llsd', 'trusted-sender': False}, 'avatarclassifiedsrequest': {'service_name': 'avatar-classifieds', 'builder': 'template', 'trusted-sender': False}, 'LayerData': {'flavor': 'template', 'trusted-sender': False}, 'CreateTrustedCircuit': {'flavor': 'template', 'trusted-sender': False}, 'PlacesReply': {'flavor': 'llsd', 'trusted-sender': True}, 'SetCPURatio': {'flavor': 'template', 'trusted-sender': False}, 'RegionHandshakeReply': {'flavor': 'template', 'trusted-sender': False}, 'pickinforequest': {'service_name': 'pick-info', 'builder': 'template', 'trusted-sender': False}, 'ObjectUpdate': {'flavor': 'template', 'trusted-sender': False}, 'ViewerEffect': {'flavor': 'template', 'trusted-sender': False}, 'ChatterBoxSessionUpdate': {'flavor': 'llsd', 'trusted-sender': True}, 'SimulatorLoad': {'flavor': 'template', 'trusted-sender': True}, 'TeleportFailed': {'flavor': 'llsd', 'trusted-sender': True}, 'avatarnotesrequest': {'service_name': 'avatar-notes', 'builder': 'template', 'trusted-sender': False}, 'AgentGroupDataUpdate': {'flavor': 'llsd', 'trusted-sender': True}, 'GodKickUser': {'flavor': 'llsd', 'trusted-sender': False}, 'AddCircuitCode': {'flavor': 'template', 'trusted-sender': True}, 'ProvisionVoiceAccountRequest': {'flavor': 'llsd', 'trusted-sender': False}, 'AgentDropGroup': {'flavor': 'llsd', 'trusted-sender': True}, 'ChatterBoxSessionStartReply': {'flavor': 'llsd', 'trusted-sender': True}, 'ReportAutosaveCrash': {'flavor': 'template', 'trusted-sender': False}, 'TeleportFinish': {'flavor': 'llsd', 'trusted-sender': True}, 'ChatterBoxSessionAgentListUpdates': {'flavor': 'llsd', 'trusted-sender': True}}, 'serverDefaults': {'simulator': 'template', 'viewer': 'template', 'rpcserver': 'template', 'dataserver': 'template', 'spaceserver': 'template', 'mapserver': 'template', 'inventoryDataserver': 'template', 'logDataserver': 'template'}, 'parsed_llsd': {'serverDefaults': {'simulator': 'template', 'viewer': 'template', 'rpcserver': 'template', 'dataserver': 'template', 'spaceserver': 'template', 'mapserver': 'template', 'inventoryDataserver': 'template', 'logDataserver': 'template'}, 'messages': {'ParcelProperties': {'flavor': 'llsd', 'trusted-sender': True}, 'ImprovedTerseObjectUpdate': {'flavor': 'template', 'trusted-sender': False}, 'EstablishAgentCommunication': {'flavor': 'llsd', 'trusted-sender': True}, 'FetchInventoryDescendents': {'flavor': 'template', 'trusted-sender': False}, 'CrossedRegion': {'flavor': 'llsd', 'trusted-sender': True}, 'ForceCloseChatterBoxSession': {'flavor': 'llsd', 'trusted-sender': True}, 'EconomyDataRequest': {'flavor': 'template', 'trusted-sender': False}, 'CoarseLocationUpdate': {'flavor': 'template', 'trusted-sender': True, 'only-send-latest': True}, 'CompletePingCheck': {'flavor': 'template', 'trusted-sender': False}, 'avatarpicksrequest': {'service_name': 'avatar-pick', 'builder': 'template', 'trusted-sender': False}, 'ParcelVoiceInfo': {'flavor': 'llsd', 'trusted-sender': True}, 'ObjectUpdateCached': {'flavor': 'template', 'trusted-sender': False}, 'ObjectUpdateCompressed': {'flavor': 'template', 'trusted-sender': False}, 'RoutedMoneyBalanceReply': {'flavor': 'llsd', 'trusted-sender': False}, 'EdgeDataPacket': {'flavor': 'template', 'trusted-sender': True}, 'LandStatReply': {'flavor': 'llsd', 'trusted-sender': False}, 'AvatarAppearance': {'flavor': 'template', 'trusted-sender': False}, 'EnableSimulator': {'flavor': 'llsd', 'trusted-sender': True}, 'DirLandReply': {'flavor': 'llsd', 'trusted-sender': True}, 'UseCircuitCode': {'flavor': 'template', 'trusted-sender': False}, 'AvatarAnimation': {'flavor': 'template', 'trusted-sender': False}, 'RequiredVoiceVersion': {'flavor': 'llsd', 'trusted-sender': True}, 'ImagePacket': {'flavor': 'template', 'trusted-sender': False}, 'StartGroupProposal': {'flavor': 'llsd', 'trusted-sender': False}, 'StartPingCheck': {'flavor': 'template', 'trusted-sender': False}, 'AgentUpdate': {'flavor': 'template', 'trusted-sender': False}, 'RequestTrustedCircuit': {'flavor': 'template', 'trusted-sender': True}, 'ChatterBoxSessionEventReply': {'flavor': 'llsd', 'trusted-sender': True}, 'GroupProposalBallot': {'flavor': 'llsd', 'trusted-sender': False}, 'RpcScriptRequestInboundForward': {'flavor': 'llsd', 'trusted-sender': False}, 'ParcelObjectOwnersReply': {'flavor': 'llsd', 'trusted-sender': True}, 'OpenCircuit': {'flavor': 'template', 'trusted-sender': False}, 'ParcelVoiceInfoRequest': {'flavor': 'llsd', 'trusted-sender': False}, 'CloseCircuit': {'flavor': 'template', 'trusted-sender': False}, 'PacketAck': {'flavor': 'template', 'trusted-sender': False}, 'ChatterBoxInvitation': {'flavor': 'llsd', 'trusted-sender': True}, 'CompleteAgentMovement': {'flavor': 'template', 'trusted-sender': False}, 'ScriptRunningReply': {'flavor': 'llsd', 'trusted-sender': False}, 'avatarclassifiedsrequest': {'service_name': 'avatar-classifieds', 'builder': 'template', 'trusted-sender': False}, 'LayerData': {'flavor': 'template', 'trusted-sender': False}, 'CreateTrustedCircuit': {'flavor': 'template', 'trusted-sender': False}, 'PlacesReply': {'flavor': 'llsd', 'trusted-sender': True}, 'SetCPURatio': {'flavor': 'template', 'trusted-sender': False}, 'RegionHandshakeReply': {'flavor': 'template', 'trusted-sender': False}, 'pickinforequest': {'service_name': 'pick-info', 'builder': 'template', 'trusted-sender': False}, 'ObjectUpdate': {'flavor': 'template', 'trusted-sender': False}, 'ViewerEffect': {'flavor': 'template', 'trusted-sender': False}, 'ChatterBoxSessionUpdate': {'flavor': 'llsd', 'trusted-sender': True}, 'SimulatorLoad': {'flavor': 'template', 'trusted-sender': True}, 'TeleportFailed': {'flavor': 'llsd', 'trusted-sender': True}, 'avatarnotesrequest': {'service_name': 'avatar-notes', 'builder': 'template', 'trusted-sender': False}, 'AgentGroupDataUpdate': {'flavor': 'llsd', 'trusted-sender': True}, 'GodKickUser': {'flavor': 'llsd', 'trusted-sender': False}, 'AddCircuitCode': {'flavor': 'template', 'trusted-sender': True}, 'ProvisionVoiceAccountRequest': {'flavor': 'llsd', 'trusted-sender': False}, 'AgentDropGroup': {'flavor': 'llsd', 'trusted-sender': True}, 'ChatterBoxSessionStartReply': {'flavor': 'llsd', 'trusted-sender': True}, 'ReportAutosaveCrash': {'flavor': 'template', 'trusted-sender': False}, 'TeleportFinish': {'flavor': 'llsd', 'trusted-sender': True}, 'ChatterBoxSessionAgentListUpdates': {'flavor': 'llsd', 'trusted-sender': True}}, 'capBans': {'RemoteParcelRequest': False, 'FetchInventoryDescendents': False, 'UploadBakedTexture': True, 'FetchLibDescendents': True, 'ParcelPropertiesUpdate': False, 'SendPostcard': False, 'EstateChangeInfo': True, 'ParcelGodReserveForNewbie': True, 'SearchStatTracking': False, 'EventQueueGet': False, 'UpdateNotecardAgentInventory': False, 'SendUserReport': False, 'NewFileAgentInventory': False, 'WebFetchInventoryDescendents': True, 'ChatSessionRequest': False, 'UpdateNotecardTaskInventory': False, 'UntrustedSimulatorMessage': False, 'ParcelVoiceInfoRequest': False, 'UpdateGestureAgentInventory': False, 'UpdateScriptAgentInventory': False, 'MapLayerGod': False, 'UpdateGestureTaskInventory': False, 'FetchLib': True, 'MapLayer': False, 'SendUserReportWithScreenshot': False, 'RequestTextureDownload': True, 'ViewerStartAuction': True, 'SendPostcard3': True, 'SendPostcard2': True, 'ProvisionVoiceAccountRequest': False, 'UpdateScriptTaskInventory': False, 'FetchInventory': True}, 'maxQueuedEvents': 100, 'messageBans': {'untrusted': {}, 'trusted': {}}}, 'maxQueuedEvents': 100, 'raw_llsd': '<?xml version="1.0"?>\n<llsd>\n    <map>\n\t\t  <key>serverDefaults</key>\n            <!--\n\t\t\t\t    a map of server names to default message transport\n\t\t\t\t-->\n\t\t  <map>\n\t\t\t\t<key>simulator</key>\n\t\t\t\t<string>template</string>\n\t\t\t\t\t\t\t\t\n\t\t\t\t<key>spaceserver</key>\n\t\t\t\t<string>template</string>\n\n\t\t\t\t<key>dataserver</key>\n\t\t\t\t<string>template</string>\n\n\t\t\t\t<key>logDataserver</key>\n\t\t\t\t<string>template</string>\n\n\t\t\t\t<key>inventoryDataserver</key>\n\t\t\t\t<string>template</string>\n\n\t\t\t\t<key>rpcserver</key>\n\t\t\t\t<string>template</string>\n\n\t\t\t\t<key>mapserver</key>\n\t\t\t\t<string>template</string>\n\n\t\t\t\t<key>viewer</key>\n\t\t\t\t<string>template</string>\n\n\t\t  </map>\n\t\t  <key>messages</key>\n            <!--\n\t\t\t\t    a map of individual message names that override defaults\n\t\t\t\t-->\n\t\t  <map>\n\t\t\t\t<!--\n\t\t\t\t\tCircuit related messages\n\t\t\t\t-->\n\t\t\t\t<key>PacketAck</key>\n\t\t\t\t<map>\n\t\t\t\t\t<key>flavor</key>\n\t\t\t\t\t<string>template</string>\n\t\t\t\t\t<key>trusted-sender</key>\n\t\t\t\t\t<boolean>false</boolean>\n\t\t\t\t</map>\n\n\t\t\t\t<key>OpenCircuit</key>\n\t\t\t\t<map>\n\t\t\t\t\t<key>flavor</key>\n\t\t\t\t\t<string>template</string>\n\t\t\t\t\t<key>trusted-sender</key>\n\t\t\t\t\t<boolean>false</boolean>\n\t\t\t\t</map>\n\n\t\t\t\t<key>CloseCircuit</key>\n\t\t\t\t<map>\n\t\t\t\t\t<key>flavor</key>\n\t\t\t\t\t<string>template</string>\n\t\t\t\t\t<key>trusted-sender</key>\n\t\t\t\t\t<boolean>false</boolean>\n\t\t\t\t</map>\n\n\t\t\t\t<key>StartPingCheck</key>\n\t\t\t\t<map>\n\t\t\t\t\t<key>flavor</key>\n\t\t\t\t\t<string>template</string>\n\t\t\t\t\t<key>trusted-sender</key>\n\t\t\t\t\t<boolean>false</boolean>\n\t\t\t\t</map>\n\n\t\t\t\t<key>CompletePingCheck</key>\n\t\t\t\t<map>\n\t\t\t\t\t<key>flavor</key>\n\t\t\t\t\t<string>template</string>\n\t\t\t\t\t<key>trusted-sender</key>\n\t\t\t\t\t<boolean>false</boolean>\n\t\t\t\t</map>\n\n\t\t\t\t<key>AddCircuitCode</key>\n\t\t\t\t<map>\n\t\t\t\t\t<key>flavor</key>\n\t\t\t\t\t<string>template</string>\n\t\t\t\t\t<key>trusted-sender</key>\n\t\t\t\t\t<boolean>true</boolean>\n\t\t\t\t</map>\n\n\t\t\t\t<key>UseCircuitCode</key>\n\t\t\t\t<map>\n\t\t\t\t\t<key>flavor</key>\n\t\t\t\t\t<string>template</string>\n\t\t\t\t\t<key>trusted-sender</key>\n\t\t\t\t\t<boolean>false</boolean>\n\t\t\t\t</map>\n\n\t\t\t\t<key>CreateTrustedCircuit</key>\n\t\t\t\t<map>\n\t\t\t\t\t<key>flavor</key>\n\t\t\t\t\t<string>template</string>\n\t\t\t\t\t<key>trusted-sender</key>\n\t\t\t\t\t<boolean>false</boolean>\n\t\t\t\t</map>\n\n\t\t\t\t<key>RequestTrustedCircuit</key>\n\t\t\t\t<map>\n\t\t\t\t\t<key>flavor</key>\n\t\t\t\t\t<string>template</string>\n\t\t\t\t\t<key>trusted-sender</key>\n\t\t\t\t\t<boolean>true</boolean>\n\t\t\t\t</map>\n\n\t\t\t\t<!--\n\t\t\t\t\tSimulator to Launcher\n\t\t\t\t\tuntil we get a HTTP server in the launcher\n\t\t\t\t-->\n\t\t\t\t<key>ReportAutosaveCrash</key>\n\t\t\t\t<map>\n\t\t\t\t\t<key>flavor</key>\n\t\t\t\t\t<string>template</string>\n\t\t\t\t\t<key>trusted-sender</key>\n\t\t\t\t\t<boolean>false</boolean>\n\t\t\t\t</map>\n\n\t\t\t\t<key>SetCPURatio</key>\n\t\t\t\t<map>\n\t\t\t\t\t<key>flavor</key>\n\t\t\t\t\t<string>template</string>\n\t\t\t\t\t<key>trusted-sender</key>\n\t\t\t\t\t<boolean>false</boolean>\n\t\t\t\t</map>\n\n\t\t\t\t<!--\n\t\t\t\t\tViewer to simulator messages sent before UntrustedSimulatorMessage cap received.\n\t\t\t\t-->\n\t\t\t\t<key>CompleteAgentMovement</key>\n\t\t\t\t<map>\n\t\t\t\t\t<key>flavor</key>\n\t\t\t\t\t<string>template</string>\n\t\t\t\t\t<key>trusted-sender</key>\n\t\t\t\t\t<boolean>false</boolean>\n\t\t\t\t</map>\n\n\t\t\t\t<key>EconomyDataRequest</key>\n\t\t\t\t<map>\n\t\t\t\t\t<key>flavor</key>\n\t\t\t\t\t<string>template</string>\n\t\t\t\t\t<key>trusted-sender</key>\n\t\t\t\t\t<boolean>false</boolean>\n\t\t\t\t</map>\n\n\t\t\t\t<key>ViewerEffect</key>\n\t\t\t\t<map>\n\t\t\t\t\t<key>flavor</key>\n\t\t\t\t\t<string>template</string>\n\t\t\t\t\t<key>trusted-sender</key>\n\t\t\t\t\t<boolean>false</boolean>\n\t\t\t\t</map>\n\n\t\t\t\t<key>RegionHandshakeReply</key>\n\t\t\t\t<map>\n\t\t\t\t\t<key>flavor</key>\n\t\t\t\t\t<string>template</string>\n\t\t\t\t\t<key>trusted-sender</key>\n\t\t\t\t\t<boolean>false</boolean>\n\t\t\t\t</map>\n\n\t\t\t\t<!--\n\t\t\t\t\tViewer to simulator messages sent unreliably.\n\t\t\t\t-->\n\t\t\t\t<key>AgentUpdate</key>\n\t\t\t\t<map>\n\t\t\t\t\t<key>flavor</key>\n\t\t\t\t\t<string>template</string>\n\t\t\t\t\t<key>trusted-sender</key>\n\t\t\t\t\t<boolean>false</boolean>\n\t\t\t\t</map>\n\n\t\t\t\t<!--\n\t\t\t\t\tMessages created by LLThrottleGroup clients\n\t\t\t\t-->\n\t\t\t\t<key>ImagePacket</key>\n\t\t\t\t<map>\n\t\t\t\t\t<key>flavor</key>\n\t\t\t\t\t<string>template</string>\n\t\t\t\t\t<key>trusted-sender</key>\n\t\t\t\t\t<boolean>false</boolean>\n\t\t\t\t</map>\n\n\t\t\t\t<key>LayerData</key>\n\t\t\t\t<map>\n\t\t\t\t\t<key>flavor</key>\n\t\t\t\t\t<string>template</string>\n\t\t\t\t\t<key>trusted-sender</key>\n\t\t\t\t\t<boolean>false</boolean>\n\t\t\t\t</map>\n\n\t\t\t\t<key>ObjectUpdateCached</key>\n\t\t\t\t<map>\n\t\t\t\t\t<key>flavor</key>\n\t\t\t\t\t<string>template</string>\n\t\t\t\t\t<key>trusted-sender</key>\n\t\t\t\t\t<boolean>false</boolean>\n\t\t\t\t</map>\n\n\t\t\t\t<key>ObjectUpdateCompressed</key>\n\t\t\t\t<map>\n\t\t\t\t\t<key>flavor</key>\n\t\t\t\t\t<string>template</string>\n\t\t\t\t\t<key>trusted-sender</key>\n\t\t\t\t\t<boolean>false</boolean>\n\t\t\t\t</map>\n\n\t\t\t\t<key>ObjectUpdate</key>\n\t\t\t\t<map>\n\t\t\t\t\t<key>flavor</key>\n\t\t\t\t\t<string>template</string>\n\t\t\t\t\t<key>trusted-sender</key>\n\t\t\t\t\t<boolean>false</boolean>\n\t\t\t\t</map>\n\n\t\t\t\t<key>ImprovedTerseObjectUpdate</key>\n\t\t\t\t<map>\n\t\t\t\t\t<key>flavor</key>\n\t\t\t\t\t<string>template</string>\n\t\t\t\t\t<key>trusted-sender</key>\n\t\t\t\t\t<boolean>false</boolean>\n\t\t\t\t</map>\n\n\t\t\t\t<key>AvatarAnimation</key>\n\t\t\t\t<map>\n\t\t\t\t\t<key>flavor</key>\n\t\t\t\t\t<string>template</string>\n\t\t\t\t\t<key>trusted-sender</key>\n\t\t\t\t\t<boolean>false</boolean>\n\t\t\t\t</map>\n\n\t\t\t\t<key>AvatarAppearance</key>\n\t\t\t\t<map>\n\t\t\t\t\t<key>flavor</key>\n\t\t\t\t\t<string>template</string>\n\t\t\t\t\t<key>trusted-sender</key>\n\t\t\t\t\t<boolean>false</boolean>\n\t\t\t\t</map>\n\n\t\t\t\t<!-- Simulator to simulator reliable messages -->\n\t\t\t\t<key>GodKickUser</key>\n\t\t\t\t<map>\n\t\t\t\t\t<key>flavor</key>\n\t\t\t\t\t<string>llsd</string>\n\t\t\t\t\t<key>trusted-sender</key>\n\t\t\t\t\t<boolean>false</boolean>\n\t\t\t\t</map>\n\n\t\t\t\t<key>RoutedMoneyBalanceReply</key>\n\t\t\t\t<map>\n\t\t\t\t<key>flavor</key>\n\t\t\t\t<string>llsd</string>\n\t\t\t\t<key>trusted-sender</key>\n\t\t\t\t<boolean>false</boolean>\n\t\t\t\t</map>\n\n\t\t\t\t<!-- Simulator to simulator unreliable messages -->\n\t\t\t\t<key>EdgeDataPacket</key>\n\t\t\t\t<map>\n\t\t\t\t    <key>flavor</key>\n\t\t\t\t\t<string>template</string>\n\t\t\t\t\t<key>trusted-sender</key>\n\t\t\t\t\t<boolean>true</boolean>\n\t\t\t\t</map>\n\n\t\t\t\t<key>CoarseLocationUpdate</key>\n\t\t\t\t<map>\n\t\t\t\t    <key>flavor</key>\n\t\t\t\t\t<string>template</string>\n\t\t\t\t\t<key>trusted-sender</key>\n\t\t\t\t\t<boolean>true</boolean>\n\t\t\t\t\t<key>only-send-latest</key>\n\t\t\t\t\t<boolean>true</boolean>\t\t\t\t\t\n\t\t\t\t</map>\n\n\t\t\t\t<key>SimulatorLoad</key>\n\t\t\t\t<map>\n\t\t\t\t    <key>flavor</key>\n\t\t\t\t\t<string>template</string>\n\t\t\t\t\t<key>trusted-sender</key>\n\t\t\t\t\t<boolean>true</boolean>\n\t\t\t\t</map>\n\n\t\t\t\t<key>EstablishAgentCommunication</key>\n\t\t\t\t<map>\n\t\t\t\t\t<key>flavor</key>\n\t\t\t\t\t<string>llsd</string>\n\t\t\t\t\t<key>trusted-sender</key>\n\t\t\t\t\t<boolean>true</boolean>\n\t\t\t\t</map>\n\n\t\t\t\t<key>AgentGroupDataUpdate</key>\n\t\t\t\t<map>\n\t\t\t\t\t<key>flavor</key>\n\t\t\t\t\t<string>llsd</string>\n\t\t\t\t\t<key>trusted-sender</key>\n\t\t\t\t\t<boolean>true</boolean>\n\t\t\t\t</map>\n\n\t\t\t\t<key>AgentDropGroup</key>\n\t\t\t\t<map>\n\t\t\t\t    <key>flavor</key>\n\t\t\t\t    <string>llsd</string>\n\t\t\t\t    <key>trusted-sender</key>\n\t\t\t\t    <boolean>true</boolean>\n\t\t\t\t</map>\n\n\t\t\t\t<key>ChatterBoxSessionStartReply</key>\n\t\t\t\t<map>\n\t\t\t\t\t<key>flavor</key>\n\t\t\t\t\t<string>llsd</string>\n\t\t\t\t\t<key>trusted-sender</key>\n\t\t\t\t\t<boolean>true</boolean>\n\t\t\t\t</map>\n\n\t\t\t\t<key>ChatterBoxSessionEventReply</key>\n\t\t\t\t<map>\n\t\t\t\t\t<key>flavor</key>\n\t\t\t\t\t<string>llsd</string>\n\t\t\t\t\t<key>trusted-sender</key>\n\t\t\t\t\t<boolean>true</boolean>\n\t\t\t\t</map>\n\n\t\t\t\t<key>ForceCloseChatterBoxSession</key>\n\t\t\t\t<map>\n\t\t\t\t\t<key>flavor</key>\n\t\t\t\t\t<string>llsd</string>\n\t\t\t\t\t<key>trusted-sender</key>\n\t\t\t\t\t<boolean>true</boolean>\n\t\t\t\t</map>\n\n\t\t\t\t<key>ChatterBoxSessionAgentListUpdates</key>\n\t\t\t\t<map>\n\t\t\t\t\t<key>flavor</key>\n\t\t\t\t\t<string>llsd</string>\n\t\t\t\t\t<key>trusted-sender</key>\n\t\t\t\t\t<boolean>true</boolean>\n\t\t\t\t</map>\n\t\t\t\t\n\t\t\t\t<key>ChatterBoxSessionUpdate</key>\n\t\t\t\t<map>\n\t\t\t\t\t<key>flavor</key>\n\t\t\t\t\t<string>llsd</string>\n\t\t\t\t\t<key>trusted-sender</key>\n\t\t\t\t\t<boolean>true</boolean>\n\t\t\t\t</map>\n\t\t\t\t\n\t\t\t\t<key>ChatterBoxInvitation</key>\n\t\t\t\t<map>\n\t\t\t\t\t<key>flavor</key>\n\t\t\t\t\t<string>llsd</string>\n\t\t\t\t\t<key>trusted-sender</key>\n\t\t\t\t\t<boolean>true</boolean>\n\t\t\t\t</map>\n\n\t\t\t\t<!-- Client to server -->\n\t\t\t\t<key>ParcelVoiceInfoRequest</key>\n\t\t\t\t<map>\n\t\t\t\t\t<key>flavor</key>\n\t\t\t\t\t<string>llsd</string>\n\t\t\t\t\t<key>trusted-sender</key>\n\t\t\t\t\t<boolean>false</boolean>\n\t\t\t\t</map>\n\n\t\t\t\t<!-- Server to client -->\n\t\t\t\t<key>ParcelVoiceInfo</key>\n\t\t\t\t<map>\n\t\t\t\t\t<key>flavor</key>\n\t\t\t\t\t<string>llsd</string>\n\t\t\t\t\t<key>trusted-sender</key>\n\t\t\t\t\t<boolean>true</boolean>\n\t\t\t\t</map>\n\t\t\t\t\n\t\t\t\t<key>ParcelObjectOwnersReply</key>\n\t\t\t\t<map>\n\t\t\t\t\t<key>flavor</key>\n\t\t\t\t\t<string>llsd</string>\n\t\t\t\t\t<key>trusted-sender</key>\n\t\t\t\t\t<boolean>true</boolean>\n\t\t\t\t</map>\n\n\t\t\t\t<key>ParcelProperties</key>\n\t\t\t\t<map>\n\t\t\t\t\t<key>flavor</key>\n\t\t\t\t\t<string>llsd</string>\n\t\t\t\t\t<key>trusted-sender</key>\n\t\t\t\t\t<boolean>true</boolean>\n\t\t\t\t</map>\n\n\t\t\t\t<key>LandStatReply</key>\n\t\t\t\t<map>\n\t\t\t\t\t<key>flavor</key>\n\t\t\t\t\t<string>llsd</string>\n\t\t\t\t\t<key>trusted-sender</key>\n\t\t\t\t\t<boolean>true</boolean>\n\t\t\t\t</map>\n\n        <key>PlacesReply</key>\n        <map>\n          <key>flavor</key>\n          <string>llsd</string>\n          <key>trusted-sender</key>\n          <boolean>true</boolean>\n        </map>\n\n        <key>DirLandReply</key>\n        <map>\n          <key>flavor</key>\n          <string>llsd</string>\n          <key>trusted-sender</key>\n          <boolean>true</boolean>\n        </map>\n\n        <key>avatarnotesrequest</key>\n\t\t\t\t<map>\n\t\t\t\t\t<key>service_name</key>\n\t\t\t\t\t<string>avatar-notes</string>\n\t\t\t\t\t<key>builder</key>\n\t\t\t\t\t<string>template</string>\n\t\t\t\t\t<key>trusted-sender</key>\n\t\t\t\t\t<boolean>false</boolean>\n\t\t\t\t</map>\n\n\t\t\t\t<key>avatarclassifiedsrequest</key>\n\t\t\t\t<map>\n\t\t\t\t\t<key>service_name</key>\n\t\t\t\t\t<string>avatar-classifieds</string>\n\t\t\t\t\t<key>builder</key>\n\t\t\t\t\t<string>template</string>\n\t\t\t\t\t<key>trusted-sender</key>\n\t\t\t\t\t<boolean>false</boolean>\n\t\t\t\t</map>\n\n\t\t\t\t<key>avatarpicksrequest</key>\n\t\t\t\t<map>\n\t\t\t\t\t<key>service_name</key>\n\t\t\t\t\t<string>avatar-pick</string>\n\t\t\t\t\t<key>builder</key>\n\t\t\t\t\t<string>template</string>\n\t\t\t\t\t<key>trusted-sender</key>\n\t\t\t\t\t<boolean>false</boolean>\n\t\t\t\t</map>\n\n\t\t\t\t<key>pickinforequest</key>\n\t\t\t\t<map>\n\t\t\t\t\t<key>service_name</key>\n\t\t\t\t\t<string>pick-info</string>\n\t\t\t\t\t<key>builder</key>\n\t\t\t\t\t<string>template</string>\n\t\t\t\t\t<key>trusted-sender</key>\n\t\t\t\t\t<boolean>false</boolean>\n\t\t\t\t</map>\n\t\t\t\t\n\t\t\t\t<key>ProvisionVoiceAccountRequest</key>\n\t\t\t\t<map>\n\t\t\t\t\t<key>flavor</key>\n\t\t\t\t\t<string>llsd</string>\n\t\t\t\t\t<key>trusted-sender</key>\n\t\t\t\t\t<boolean>false</boolean>\n\t\t\t\t</map>\n\n\t\t\t\t<!-- Server to client -->\n\t\t\t\t<key>RequiredVoiceVersion</key>\n\t\t\t\t<map>\n\t\t\t\t\t<key>flavor</key>\n\t\t\t\t\t<string>llsd</string>\n\t\t\t\t\t<key>trusted-sender</key>\n\t\t\t\t\t<boolean>true</boolean>\n\t\t\t\t</map>\n\t\t\t\t\n\t\t\t\t<key>EnableSimulator</key>\n\t\t\t\t<map>\n\t\t\t\t\t<key>flavor</key>\n\t\t\t\t\t<string>llsd</string>\n\t\t\t\t\t<key>trusted-sender</key>\n\t\t\t\t\t<boolean>true</boolean>\n\t\t\t\t</map>\n\t\t\t\t\n\t\t\t\t<key>TeleportFinish</key>\n\t\t\t\t<map>\n\t\t\t\t\t<key>flavor</key>\n\t\t\t\t\t<string>llsd</string>\n\t\t\t\t\t<key>trusted-sender</key>\n\t\t\t\t\t<boolean>true</boolean>\n\t\t\t\t</map>\n\n        <key>TeleportFailed</key>\n        <map>\n          <key>flavor</key>\n          <string>llsd</string>\n          <key>trusted-sender</key>\n          <boolean>true</boolean>\n        </map>\n\n        <key>CrossedRegion</key>\n\t\t\t\t<map>\n\t\t\t\t\t<key>flavor</key>\n\t\t\t\t\t<string>llsd</string>\n\t\t\t\t\t<key>trusted-sender</key>\n\t\t\t\t\t<boolean>true</boolean>\n\t\t\t\t</map>\n\t\t\t\t\n\t\t\t\t<!-- UDPDeprecated Messages -->\n\t\t\t\t<key>ScriptRunningReply</key>\n\t\t\t\t<map>\n\t\t\t\t\t<key>flavor</key>\n\t\t\t\t\t<string>llsd</string>\n\t\t\t\t\t<key>trusted-sender</key>\n\t\t\t\t\t<boolean>false</boolean>\n\t\t\t\t</map>\n\t\t\t\t\n\t\t\t\t<key>LandStatReply</key>\n\t\t\t\t<map>\n\t\t\t\t\t<key>flavor</key>\n\t\t\t\t\t<string>llsd</string>\n\t\t\t\t\t<key>trusted-sender</key>\n\t\t\t\t\t<boolean>false</boolean>\n\t\t\t\t</map>\n\n\t\t\t\t<key>StartGroupProposal</key>\n\t\t\t\t<map>\n\t\t\t\t\t<key>flavor</key>\n\t\t\t\t\t<string>llsd</string>\n\t\t\t\t\t<key>trusted-sender</key>\n\t\t\t\t\t<boolean>false</boolean>\n\t\t\t\t</map>\n\n\t\t\t\t<key>FetchInventoryDescendents</key>\n\t\t\t\t<map>\n\t\t\t\t\t<key>flavor</key>\n\t\t\t\t\t<string>template</string>\n\t\t\t\t\t<key>trusted-sender</key>\n\t\t\t\t\t<boolean>false</boolean>\n\t\t\t\t</map>\n\t\t\t\t\n\t\t\t\t<key>GroupProposalBallot</key>\n\t\t\t\t<map>\n\t\t\t\t\t<key>flavor</key>\n\t\t\t\t\t<string>llsd</string>\n\t\t\t\t\t<key>trusted-sender</key>\n\t\t\t\t\t<boolean>false</boolean>\n\t\t\t\t</map>\n\n\t\t\t\t<key>RpcScriptRequestInboundForward</key>\n\t\t\t\t<map>\n\t\t\t\t\t<key>flavor</key>\n\t\t\t\t\t<string>llsd</string>\n\t\t\t\t\t<key>trusted-sender</key>\n\t\t\t\t\t<boolean>false</boolean>\n\t\t\t\t</map>\n\n\n\t\t  </map>\n  \t  \t<key>capBans</key>\n    \t<map>\n\t\t\t<key>MapLayer</key>\n\t\t\t<boolean>false</boolean>\n\n\t\t\t<key>MapLayerGod</key>\n\t\t\t<boolean>false</boolean>\n\n\t\t\t<key>NewFileAgentInventory</key>\n\t\t\t<boolean>false</boolean>\n\n\t\t\t<key>UpdateGestureAgentInventory</key>\n\t\t\t<boolean>false</boolean>\n\n\t\t\t<key>UpdateNotecardAgentInventory</key>\n\t\t\t<boolean>false</boolean>\n\n\t\t\t<key>UpdateScriptAgentInventory</key>\n\t\t\t<boolean>false</boolean>\n\n\t\t\t<key>UpdateGestureTaskInventory</key>\n\t\t\t<boolean>false</boolean>\n\n\t\t\t<key>UpdateNotecardTaskInventory</key>\n\t\t\t<boolean>false</boolean>\n\n\t\t\t<key>UpdateScriptTaskInventory</key>\n\t\t\t<boolean>false</boolean>\n\n\t\t\t<key>ViewerStartAuction</key>\n\t\t\t<boolean>true</boolean>\n\n\t\t\t<key>ParcelGodReserveForNewbie</key>\n\t\t\t<boolean>true</boolean>\n\n\t\t\t<key>SendUserReport</key>\n\t\t\t<boolean>false</boolean>\n\n\t\t\t<key>SendUserReportWithScreenshot</key>\n\t\t\t<boolean>false</boolean>\n\n\t\t\t<key>RequestTextureDownload</key>\n\t\t\t<boolean>true</boolean>\n\n\t\t\t<key>EventQueueGet</key>\n\t\t\t<boolean>false</boolean>\n\n\t\t\t<key>UntrustedSimulatorMessage</key>\n\t\t\t<boolean>false</boolean>\n\n\t\t\t<key>SendPostcard</key>\n\t\t\t<boolean>false</boolean>\n\n\t\t\t<key>SendPostcard2</key>\n\t\t\t<boolean>true</boolean>\n\n\t\t\t<key>SendPostcard3</key>\n\t\t\t<boolean>true</boolean>\n\n\t\t\t<key>ParcelVoiceInfoRequest</key>\n\t\t\t<boolean>false</boolean>\n\n\t\t\t<key>ChatSessionRequest</key>\n\t\t\t<boolean>false</boolean>\n\n\t\t\t<key>ProvisionVoiceAccountRequest</key>\n\t\t\t<boolean>false</boolean>\n\t\t\t\n\t\t\t<key>RemoteParcelRequest</key>\n\t\t\t<boolean>false</boolean>\n\n\t\t\t<key>SearchStatTracking</key>\n\t\t\t<boolean>false</boolean>\n\t\t\t\n\t\t\t<key>ParcelPropertiesUpdate</key>\n\t\t\t<boolean>false</boolean>\n\t\t\t\n\t\t\t<key>EstateChangeInfo</key>\n\t\t\t<boolean>true</boolean>\n\t\t\n\t\t\t<key>FetchInventoryDescendents</key>\n\t\t\t<boolean>false</boolean>\n\t\t\n\t\t\t<key>WebFetchInventoryDescendents</key>\n\t\t\t<boolean>true</boolean>\n\t\t\n\t\t\t<key>FetchInventory</key>\n\t\t\t<boolean>true</boolean>\n\t\t\n\t\t\t<key>FetchLibDescendents</key>\n\t\t\t<boolean>true</boolean>\n\t\t\n\t\t\t<key>FetchLib</key>\n\t\t\t<boolean>true</boolean>\n\n\t\t\t<key>UploadBakedTexture</key>\n\t\t\t<boolean>true</boolean>\n\t\t</map>\n\n\t\t<key>messageBans</key>\n\t\t<map>\n\t\t\t<key>trusted</key>\n\t\t\t<map>\n\t\t\t</map>\n\t\t\n\t\t\t<key>untrusted</key>\n\t\t\t<map>\n\t\t\t</map>\n\t\t</map>\n\n\t\t<key>maxQueuedEvents</key>\n\t\t<integer>100</integer>\n    </map>\n</llsd>\n', 'capBans': {'RemoteParcelRequest': False, 'FetchInventoryDescendents': False, 'UploadBakedTexture': True, 'FetchLibDescendents': True, 'ParcelPropertiesUpdate': False, 'SendPostcard': False, 'EstateChangeInfo': True, 'ParcelGodReserveForNewbie': True, 'SearchStatTracking': False, 'EventQueueGet': False, 'UpdateNotecardAgentInventory': False, 'SendUserReport': False, 'NewFileAgentInventory': False, 'WebFetchInventoryDescendents': True, 'ChatSessionRequest': False, 'UpdateNotecardTaskInventory': False, 'UntrustedSimulatorMessage': False, 'ParcelVoiceInfoRequest': False, 'UpdateGestureAgentInventory': False, 'UpdateScriptAgentInventory': False, 'MapLayerGod': False, 'UpdateGestureTaskInventory': False, 'FetchLib': True, 'MapLayer': False, 'SendUserReportWithScreenshot': False, 'RequestTextureDownload': True, 'ViewerStartAuction': True, 'SendPostcard3': True, 'SendPostcard2': True, 'ProvisionVoiceAccountRequest': False, 'UpdateScriptTaskInventory': False, 'FetchInventory': True}})

def test_suite():
    from unittest import TestSuite, makeSuite
    suite = TestSuite()
    suite.addTest(makeSuite(TestMessageManager))
    return suite
