"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const s3_assets = require("@aws-cdk/aws-s3-assets");
const cdk = require("@aws-cdk/core");
class Code {
    /**
     * @returns `LambdaS3Code` associated with the specified S3 object.
     * @param bucket The S3 bucket
     * @param key The object key
     * @param objectVersion Optional S3 object version
     */
    static fromBucket(bucket, key, objectVersion) {
        return new S3Code(bucket, key, objectVersion);
    }
    /**
     * @deprecated use `fromBucket`
     */
    static bucket(bucket, key, objectVersion) {
        return this.fromBucket(bucket, key, objectVersion);
    }
    /**
     * @returns `LambdaInlineCode` with inline code.
     * @param code The actual handler code (limited to 4KiB)
     */
    static fromInline(code) {
        return new InlineCode(code);
    }
    /**
     * @deprecated use `fromInline`
     */
    static inline(code) {
        return this.fromInline(code);
    }
    /**
     * Loads the function code from a local disk asset.
     * @param path Either a directory with the Lambda code bundle or a .zip file
     */
    static fromAsset(path) {
        return new AssetCode(path);
    }
    /**
     * @deprecated use `fromAsset`
     */
    static asset(path) {
        return this.fromAsset(path);
    }
    /**
     * Creates a new Lambda source defined using CloudFormation parameters.
     *
     * @returns a new instance of `CfnParametersCode`
     * @param props optional construction properties of {@link CfnParametersCode}
     */
    static fromCfnParameters(props) {
        return new CfnParametersCode(props);
    }
    /**
     * @deprecated use `fromCfnParmaeters`
     */
    static cfnParameters(props) {
        return this.fromCfnParameters(props);
    }
    /**
     * Called after the CFN function resource has been created to allow the code
     * class to bind to it. Specifically it's required to allow assets to add
     * metadata for tooling like SAM CLI to be able to find their origins.
     */
    bindToResource(_resource) {
        return;
    }
}
exports.Code = Code;
/**
 * Lambda code from an S3 archive.
 */
class S3Code extends Code {
    constructor(bucket, key, objectVersion) {
        super();
        this.key = key;
        this.objectVersion = objectVersion;
        this.isInline = false;
        if (!bucket.bucketName) {
            throw new Error('bucketName is undefined for the provided bucket');
        }
        this.bucketName = bucket.bucketName;
    }
    bind(_scope) {
        return {
            s3Location: {
                bucketName: this.bucketName,
                objectKey: this.key,
                objectVersion: this.objectVersion
            }
        };
    }
}
exports.S3Code = S3Code;
/**
 * Lambda code from an inline string (limited to 4KiB).
 */
class InlineCode extends Code {
    constructor(code) {
        super();
        this.code = code;
        this.isInline = true;
        if (code.length === 0) {
            throw new Error(`Lambda inline code cannot be empty`);
        }
        if (code.length > 4096) {
            throw new Error("Lambda source is too large, must be <= 4096 but is " + code.length);
        }
    }
    bind(_scope) {
        return {
            inlineCode: this.code
        };
    }
}
exports.InlineCode = InlineCode;
/**
 * Lambda code from a local directory.
 */
class AssetCode extends Code {
    /**
     * @param path The path to the asset file or directory.
     */
    constructor(path) {
        super();
        this.path = path;
        this.isInline = false;
    }
    bind(scope) {
        // If the same AssetCode is used multiple times, retain only the first instantiation.
        if (!this.asset) {
            this.asset = new s3_assets.Asset(scope, 'Code', { path: this.path });
        }
        if (!this.asset.isZipArchive) {
            throw new Error(`Asset must be a .zip file or a directory (${this.path})`);
        }
        return {
            s3Location: {
                bucketName: this.asset.s3BucketName,
                objectKey: this.asset.s3ObjectKey
            }
        };
    }
    bindToResource(resource) {
        if (!this.asset) {
            throw new Error(`bindToResource() must be called after bind()`);
        }
        // https://github.com/aws/aws-cdk/issues/1432
        this.asset.addResourceMetadata(resource, 'Code');
    }
}
exports.AssetCode = AssetCode;
/**
 * Lambda code defined using 2 CloudFormation parameters.
 * Useful when you don't have access to the code of your Lambda from your CDK code, so you can't use Assets,
 * and you want to deploy the Lambda in a CodePipeline, using CloudFormation Actions -
 * you can fill the parameters using the {@link #assign} method.
 */
class CfnParametersCode extends Code {
    constructor(props = {}) {
        super();
        this.isInline = false;
        this._bucketNameParam = props.bucketNameParam;
        this._objectKeyParam = props.objectKeyParam;
    }
    bind(scope) {
        if (!this._bucketNameParam) {
            this._bucketNameParam = new cdk.CfnParameter(scope, 'LambdaSourceBucketNameParameter', {
                type: 'String',
            });
        }
        if (!this._objectKeyParam) {
            this._objectKeyParam = new cdk.CfnParameter(scope, 'LambdaSourceObjectKeyParameter', {
                type: 'String',
            });
        }
        return {
            s3Location: {
                bucketName: this._bucketNameParam.valueAsString,
                objectKey: this._objectKeyParam.valueAsString,
            }
        };
    }
    /**
     * Create a parameters map from this instance's CloudFormation parameters.
     *
     * It returns a map with 2 keys that correspond to the names of the parameters defined in this Lambda code,
     * and as values it contains the appropriate expressions pointing at the provided S3 location
     * (most likely, obtained from a CodePipeline Artifact by calling the `artifact.s3Location` method).
     * The result should be provided to the CloudFormation Action
     * that is deploying the Stack that the Lambda with this code is part of,
     * in the `parameterOverrides` property.
     *
     * @param location the location of the object in S3 that represents the Lambda code
     */
    assign(location) {
        const ret = {};
        ret[this.bucketNameParam] = location.bucketName;
        ret[this.objectKeyParam] = location.objectKey;
        return ret;
    }
    get bucketNameParam() {
        if (this._bucketNameParam) {
            return this._bucketNameParam.logicalId;
        }
        else {
            throw new Error('Pass CfnParametersCode to a Lambda Function before accessing the bucketNameParam property');
        }
    }
    get objectKeyParam() {
        if (this._objectKeyParam) {
            return this._objectKeyParam.logicalId;
        }
        else {
            throw new Error('Pass CfnParametersCode to a Lambda Function before accessing the objectKeyParam property');
        }
    }
}
exports.CfnParametersCode = CfnParametersCode;
//# sourceMappingURL=data:application/json;base64,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