import cloudwatch = require('@aws-cdk/aws-cloudwatch');
import ec2 = require('@aws-cdk/aws-ec2');
import iam = require('@aws-cdk/aws-iam');
import { ConstructNode, IResource, Resource } from '@aws-cdk/core';
import { IEventSource } from './event-source';
import { EventSourceMapping, EventSourceMappingOptions } from './event-source-mapping';
import { IVersion } from './lambda-version';
import { Permission } from './permission';
export interface IFunction extends IResource, ec2.IConnectable, iam.IGrantable {
    /**
     * The name of the function.
     *
     * @attribute
     */
    readonly functionName: string;
    /**
     * The ARN fo the function.
     *
     * @attribute
     */
    readonly functionArn: string;
    /**
     * The IAM role associated with this function.
     */
    readonly role?: iam.IRole;
    /**
     * Whether or not this Lambda function was bound to a VPC
     *
     * If this is is `false`, trying to access the `connections` object will fail.
     */
    readonly isBoundToVpc: boolean;
    /**
     * The `$LATEST` version of this function.
     */
    readonly latestVersion: IVersion;
    /**
     * The construct node where permissions are attached.
     */
    readonly permissionsNode: ConstructNode;
    /**
     * Adds an event source that maps to this AWS Lambda function.
     * @param id construct ID
     * @param options mapping options
     */
    addEventSourceMapping(id: string, options: EventSourceMappingOptions): EventSourceMapping;
    /**
     * Adds a permission to the Lambda resource policy.
     * @param id The id ƒor the permission construct
     */
    addPermission(id: string, permission: Permission): void;
    addToRolePolicy(statement: iam.PolicyStatement): void;
    /**
     * Grant the given identity permissions to invoke this Lambda
     */
    grantInvoke(identity: iam.IGrantable): iam.Grant;
    /**
     * Return the given named metric for this Lambda
     */
    metric(metricName: string, props?: cloudwatch.MetricOptions): cloudwatch.Metric;
    /**
     * Metric for the Duration of this Lambda
     *
     * @default average over 5 minutes
     */
    metricDuration(props?: cloudwatch.MetricOptions): cloudwatch.Metric;
    /**
     * Metric for the number of invocations of this Lambda
     *
     * @default sum over 5 minutes
     */
    metricInvocations(props?: cloudwatch.MetricOptions): cloudwatch.Metric;
    /**
     * Metric for the number of throttled invocations of this Lambda
     *
     * @default sum over 5 minutes
     */
    metricThrottles(props?: cloudwatch.MetricOptions): cloudwatch.Metric;
    addEventSource(source: IEventSource): void;
}
/**
 * Represents a Lambda function defined outside of this stack.
 */
export interface FunctionAttributes {
    /**
     * The ARN of the Lambda function.
     *
     * Format: arn:<partition>:lambda:<region>:<account-id>:function:<function-name>
     */
    readonly functionArn: string;
    /**
     * The IAM execution role associated with this function.
     *
     * If the role is not specified, any role-related operations will no-op.
     */
    readonly role?: iam.IRole;
    /**
     * Id of the securityGroup for this Lambda, if in a VPC.
     *
     * This needs to be given in order to support allowing connections
     * to this Lambda.
     */
    readonly securityGroupId?: string;
}
export declare abstract class FunctionBase extends Resource implements IFunction {
    /**
     * The principal this Lambda Function is running as
     */
    abstract readonly grantPrincipal: iam.IPrincipal;
    /**
     * The name of the function.
     */
    abstract readonly functionName: string;
    /**
     * The ARN fo the function.
     */
    abstract readonly functionArn: string;
    /**
     * The IAM role associated with this function.
     *
     * Undefined if the function was imported without a role.
     */
    abstract readonly role?: iam.IRole;
    /**
     * The construct node where permissions are attached.
     */
    abstract readonly permissionsNode: ConstructNode;
    /**
     * Whether the addPermission() call adds any permissions
     *
     * True for new Lambdas, false for imported Lambdas (they might live in different accounts).
     */
    protected abstract readonly canCreatePermissions: boolean;
    /**
     * Adds a permission to the Lambda resource policy.
     * @param id The id ƒor the permission construct
     */
    addPermission(id: string, permission: Permission): void;
    addToRolePolicy(statement: iam.PolicyStatement): void;
    /**
     * Access the Connections object
     *
     * Will fail if not a VPC-enabled Lambda Function
     */
    readonly connections: ec2.Connections;
    readonly latestVersion: IVersion;
    /**
     * Whether or not this Lambda function was bound to a VPC
     *
     * If this is is `false`, trying to access the `connections` object will fail.
     */
    readonly isBoundToVpc: boolean;
    addEventSourceMapping(id: string, options: EventSourceMappingOptions): EventSourceMapping;
    /**
     * Grant the given identity permissions to invoke this Lambda
     */
    grantInvoke(grantee: iam.IGrantable): iam.Grant;
    /**
     * Adds an event source to this function.
     *
     * Event sources are implemented in the @aws-cdk/aws-lambda-event-sources module.
     *
     * The following example adds an SQS Queue as an event source:
     *
     *     import { SqsEventSource } from '@aws-cdk/aws-lambda-event-sources';
     *     myFunction.addEventSource(new SqsEventSource(myQueue));
     *
     * @param source The event source to bind to this function
     */
    addEventSource(source: IEventSource): void;
    private parsePermissionPrincipal;
}
export declare abstract class QualifiedFunctionBase extends FunctionBase {
    abstract readonly lambda: IFunction;
    readonly permissionsNode: ConstructNode;
    readonly latestVersion: IVersion;
}
