"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const cloudwatch = require("@aws-cdk/aws-cloudwatch");
const ec2 = require("@aws-cdk/aws-ec2");
const iam = require("@aws-cdk/aws-iam");
const sqs = require("@aws-cdk/aws-sqs");
const core_1 = require("@aws-cdk/core");
const function_base_1 = require("./function-base");
const lambda_version_1 = require("./lambda-version");
const lambda_generated_1 = require("./lambda.generated");
const log_retention_1 = require("./log-retention");
/**
 * X-Ray Tracing Modes (https://docs.aws.amazon.com/lambda/latest/dg/API_TracingConfig.html)
 */
var Tracing;
(function (Tracing) {
    /**
     * Lambda will respect any tracing header it receives from an upstream service.
     * If no tracing header is received, Lambda will call X-Ray for a tracing decision.
     */
    Tracing["ACTIVE"] = "Active";
    /**
     * Lambda will only trace the request from an upstream service
     * if it contains a tracing header with "sampled=1"
     */
    Tracing["PASS_THROUGH"] = "PassThrough";
    /**
     * Lambda will not trace any request.
     */
    Tracing["DISABLED"] = "Disabled";
})(Tracing = exports.Tracing || (exports.Tracing = {}));
/**
 * Deploys a file from from inside the construct library as a function.
 *
 * The supplied file is subject to the 4096 bytes limit of being embedded in a
 * CloudFormation template.
 *
 * The construct includes an associated role with the lambda.
 *
 * This construct does not yet reproduce all features from the underlying resource
 * library.
 */
class Function extends function_base_1.FunctionBase {
    constructor(scope, id, props) {
        super(scope, id, {
            physicalName: props.functionName,
        });
        this.permissionsNode = this.node;
        this.canCreatePermissions = true;
        this.layers = [];
        this.environment = props.environment || {};
        const managedPolicies = new Array();
        // the arn is in the form of - arn:aws:iam::aws:policy/service-role/AWSLambdaBasicExecutionRole
        managedPolicies.push(iam.ManagedPolicy.fromAwsManagedPolicyName("service-role/AWSLambdaBasicExecutionRole"));
        if (props.vpc) {
            // Policy that will have ENI creation permissions
            managedPolicies.push(iam.ManagedPolicy.fromAwsManagedPolicyName("service-role/AWSLambdaVPCAccessExecutionRole"));
        }
        this.role = props.role || new iam.Role(this, 'ServiceRole', {
            assumedBy: new iam.ServicePrincipal('lambda.amazonaws.com'),
            managedPolicies
        });
        this.grantPrincipal = this.role;
        for (const statement of (props.initialPolicy || [])) {
            this.role.addToPolicy(statement);
        }
        const region = core_1.Stack.of(this).region;
        const isChina = !core_1.Token.isUnresolved(region) && region.startsWith('cn-');
        if (isChina && props.environment && Object.keys(props.environment).length > 0) {
            // tslint:disable-next-line:max-line-length
            throw new Error(`Environment variables are not supported in this region (${region}); consider using tags or SSM parameters instead`);
        }
        const code = props.code.bind(this);
        verifyCodeConfig(code, props.runtime);
        const resource = new lambda_generated_1.CfnFunction(this, 'Resource', {
            functionName: this.physicalName,
            description: props.description,
            code: {
                s3Bucket: code.s3Location && code.s3Location.bucketName,
                s3Key: code.s3Location && code.s3Location.objectKey,
                s3ObjectVersion: code.s3Location && code.s3Location.objectVersion,
                zipFile: code.inlineCode
            },
            layers: core_1.Lazy.listValue({ produce: () => this.layers.map(layer => layer.layerVersionArn) }, { omitEmpty: true }),
            handler: props.handler,
            timeout: props.timeout && props.timeout.toSeconds(),
            runtime: props.runtime.name,
            role: this.role.roleArn,
            environment: core_1.Lazy.anyValue({ produce: () => this.renderEnvironment() }),
            memorySize: props.memorySize,
            vpcConfig: this.configureVpc(props),
            deadLetterConfig: this.buildDeadLetterConfig(props),
            tracingConfig: this.buildTracingConfig(props),
            reservedConcurrentExecutions: props.reservedConcurrentExecutions
        });
        resource.node.addDependency(this.role);
        this.functionName = this.getResourceNameAttribute(resource.ref);
        this.functionArn = this.getResourceArnAttribute(resource.attrArn, {
            service: 'lambda',
            resource: 'function',
            resourceName: this.physicalName,
            sep: ':',
        });
        this.runtime = props.runtime;
        if (props.layers) {
            this.addLayers(...props.layers);
        }
        for (const event of props.events || []) {
            this.addEventSource(event);
        }
        // Log retention
        if (props.logRetention) {
            new log_retention_1.LogRetention(this, 'LogRetention', {
                logGroupName: `/aws/lambda/${this.functionName}`,
                retention: props.logRetention
            });
        }
        props.code.bindToResource(resource);
    }
    static fromFunctionArn(scope, id, functionArn) {
        return Function.fromFunctionAttributes(scope, id, { functionArn });
    }
    /**
     * Creates a Lambda function object which represents a function not defined
     * within this stack.
     *
     *    Lambda.import(this, 'MyImportedFunction', { lambdaArn: new LambdaArn('arn:aws:...') });
     *
     * @param scope The parent construct
     * @param id The name of the lambda construct
     * @param attrs the attributes of the function to import
     */
    static fromFunctionAttributes(scope, id, attrs) {
        const functionArn = attrs.functionArn;
        const functionName = extractNameFromArn(attrs.functionArn);
        const role = attrs.role;
        class Import extends function_base_1.FunctionBase {
            constructor(s, i) {
                super(s, i);
                this.functionName = functionName;
                this.functionArn = functionArn;
                this.role = role;
                this.permissionsNode = this.node;
                this.canCreatePermissions = false;
                this.grantPrincipal = role || new iam.UnknownPrincipal({ resource: this });
                if (attrs.securityGroupId) {
                    this._connections = new ec2.Connections({
                        securityGroups: [
                            ec2.SecurityGroup.fromSecurityGroupId(this, 'SecurityGroup', attrs.securityGroupId)
                        ]
                    });
                }
            }
        }
        return new Import(scope, id);
    }
    /**
     * Return the given named metric for this Lambda
     */
    static metricAll(metricName, props) {
        return new cloudwatch.Metric({
            namespace: 'AWS/Lambda',
            metricName,
            ...props
        });
    }
    /**
     * Metric for the number of Errors executing all Lambdas
     *
     * @default sum over 5 minutes
     */
    static metricAllErrors(props) {
        return this.metricAll('Errors', { statistic: 'sum', ...props });
    }
    /**
     * Metric for the Duration executing all Lambdas
     *
     * @default average over 5 minutes
     */
    static metricAllDuration(props) {
        return this.metricAll('Duration', props);
    }
    /**
     * Metric for the number of invocations of all Lambdas
     *
     * @default sum over 5 minutes
     */
    static metricAllInvocations(props) {
        return this.metricAll('Invocations', { statistic: 'sum', ...props });
    }
    /**
     * Metric for the number of throttled invocations of all Lambdas
     *
     * @default sum over 5 minutes
     */
    static metricAllThrottles(props) {
        return this.metricAll('Throttles', { statistic: 'sum', ...props });
    }
    /**
     * Metric for the number of concurrent executions across all Lambdas
     *
     * @default max over 5 minutes
     */
    static metricAllConcurrentExecutions(props) {
        // Mini-FAQ: why max? This metric is a gauge that is emitted every
        // minute, so either max or avg or a percentile make sense (but sum
        // doesn't). Max is more sensitive to spiky load changes which is
        // probably what you're interested in if you're looking at this metric
        // (Load spikes may lead to concurrent execution errors that would
        // otherwise not be visible in the avg)
        return this.metricAll('ConcurrentExecutions', { statistic: 'max', ...props });
    }
    /**
     * Metric for the number of unreserved concurrent executions across all Lambdas
     *
     * @default max over 5 minutes
     */
    static metricAllUnreservedConcurrentExecutions(props) {
        return this.metricAll('UnreservedConcurrentExecutions', { statistic: 'max', ...props });
    }
    /**
     * Adds an environment variable to this Lambda function.
     * If this is a ref to a Lambda function, this operation results in a no-op.
     * @param key The environment variable key.
     * @param value The environment variable's value.
     */
    addEnvironment(key, value) {
        if (!this.environment) {
            // TODO: add metadata
            return this;
        }
        this.environment[key] = value;
        return this;
    }
    /**
     * Adds one or more Lambda Layers to this Lambda function.
     *
     * @param layers the layers to be added.
     *
     * @throws if there are already 5 layers on this function, or the layer is incompatible with this function's runtime.
     */
    addLayers(...layers) {
        for (const layer of layers) {
            if (this.layers.length === 5) {
                throw new Error('Unable to add layer: this lambda function already uses 5 layers.');
            }
            if (layer.compatibleRuntimes && !layer.compatibleRuntimes.find(runtime => runtime.runtimeEquals(this.runtime))) {
                const runtimes = layer.compatibleRuntimes.map(runtime => runtime.name).join(', ');
                throw new Error(`This lambda function uses a runtime that is incompatible with this layer (${this.runtime.name} is not in [${runtimes}])`);
            }
            this.layers.push(layer);
        }
    }
    /**
     * Add a new version for this Lambda
     *
     * If you want to deploy through CloudFormation and use aliases, you need to
     * add a new version (with a new name) to your Lambda every time you want
     * to deploy an update. An alias can then refer to the newly created Version.
     *
     * All versions should have distinct names, and you should not delete versions
     * as long as your Alias needs to refer to them.
     *
     * @param name A unique name for this version
     * @param codeSha256 The SHA-256 hash of the most recently deployed Lambda source code, or
     *  omit to skip validation.
     * @param description A description for this version.
     * @returns A new Version object.
     */
    addVersion(name, codeSha256, description) {
        return new lambda_version_1.Version(this, 'Version' + name, {
            lambda: this,
            codeSha256,
            description,
        });
    }
    renderEnvironment() {
        if (!this.environment || Object.keys(this.environment).length === 0) {
            return undefined;
        }
        return {
            variables: this.environment
        };
    }
    /**
     * If configured, set up the VPC-related properties
     *
     * Returns the VpcConfig that should be added to the
     * Lambda creation properties.
     */
    configureVpc(props) {
        if ((props.securityGroup || props.allowAllOutbound !== undefined) && !props.vpc) {
            throw new Error(`Cannot configure 'securityGroup' or 'allowAllOutbound' without configuring a VPC`);
        }
        if (!props.vpc) {
            return undefined;
        }
        if (props.securityGroup && props.allowAllOutbound !== undefined) {
            throw new Error(`Configure 'allowAllOutbound' directly on the supplied SecurityGroup.`);
        }
        const securityGroup = props.securityGroup || new ec2.SecurityGroup(this, 'SecurityGroup', {
            vpc: props.vpc,
            description: 'Automatic security group for Lambda Function ' + this.node.uniqueId,
            allowAllOutbound: props.allowAllOutbound
        });
        this._connections = new ec2.Connections({ securityGroups: [securityGroup] });
        // Pick subnets, make sure they're not Public. Routing through an IGW
        // won't work because the ENIs don't get a Public IP.
        // Why are we not simply forcing vpcSubnets? Because you might still be choosing
        // Isolated networks or selecting among 2 sets of Private subnets by name.
        const { subnetIds } = props.vpc.selectSubnets(props.vpcSubnets);
        const publicSubnetIds = new Set(props.vpc.publicSubnets.map(s => s.subnetId));
        for (const subnetId of subnetIds) {
            if (publicSubnetIds.has(subnetId)) {
                throw new Error('Not possible to place Lambda Functions in a Public subnet');
            }
        }
        // List can't be empty here, if we got this far you intended to put your Lambda
        // in subnets. We're going to guarantee that we get the nice error message by
        // making VpcNetwork do the selection again.
        return {
            subnetIds,
            securityGroupIds: [securityGroup.securityGroupId]
        };
    }
    buildDeadLetterConfig(props) {
        if (props.deadLetterQueue && props.deadLetterQueueEnabled === false) {
            throw Error('deadLetterQueue defined but deadLetterQueueEnabled explicitly set to false');
        }
        if (!props.deadLetterQueue && !props.deadLetterQueueEnabled) {
            return undefined;
        }
        const deadLetterQueue = props.deadLetterQueue || new sqs.Queue(this, 'DeadLetterQueue', {
            retentionPeriod: core_1.Duration.days(14)
        });
        this.addToRolePolicy(new iam.PolicyStatement({
            actions: ['sqs:SendMessage'],
            resources: [deadLetterQueue.queueArn]
        }));
        return {
            targetArn: deadLetterQueue.queueArn
        };
    }
    buildTracingConfig(props) {
        if (props.tracing === undefined || props.tracing === Tracing.DISABLED) {
            return undefined;
        }
        this.addToRolePolicy(new iam.PolicyStatement({
            actions: ['xray:PutTraceSegments', 'xray:PutTelemetryRecords'],
            resources: ['*']
        }));
        return {
            mode: props.tracing
        };
    }
}
exports.Function = Function;
/**
 * Given an opaque (token) ARN, returns a CloudFormation expression that extracts the function
 * name from the ARN.
 *
 * Function ARNs look like this:
 *
 *   arn:aws:lambda:region:account-id:function:function-name
 *
 * ..which means that in order to extract the `function-name` component from the ARN, we can
 * split the ARN using ":" and select the component in index 6.
 *
 * @returns `FnSelect(6, FnSplit(':', arn))`
 */
function extractNameFromArn(arn) {
    return core_1.Fn.select(6, core_1.Fn.split(':', arn));
}
function verifyCodeConfig(code, runtime) {
    // mutually exclusive
    if ((!code.inlineCode && !code.s3Location) || (code.inlineCode && code.s3Location)) {
        throw new Error(`lambda.Code must specify one of "inlineCode" or "s3Location" but not both`);
    }
    // if this is inline code, check that the runtime supports
    if (code.inlineCode && !runtime.supportsInlineCode) {
        throw new Error(`Inline source not allowed for ${runtime.name}`);
    }
}
exports.verifyCodeConfig = verifyCodeConfig;
//# sourceMappingURL=data:application/json;base64,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