"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const core_1 = require("@aws-cdk/core");
const lambda_generated_1 = require("./lambda.generated");
const runtime_1 = require("./runtime");
/**
 * A reference to a Lambda Layer version.
 */
class LayerVersionBase extends core_1.Resource {
    addPermission(id, permission) {
        if (permission.organizationId != null && permission.accountId !== '*') {
            throw new Error(`OrganizationId can only be specified if AwsAccountId is '*', but it is ${permission.accountId}`);
        }
        new lambda_generated_1.CfnLayerVersionPermission(this, id, {
            action: 'lambda:GetLayerVersion',
            layerVersionArn: this.layerVersionArn,
            principal: permission.accountId,
            organizationId: permission.organizationId,
        });
    }
}
/**
 * Defines a new Lambda Layer version.
 */
class LayerVersion extends LayerVersionBase {
    /**
     * Imports a layer version by ARN. Assumes it is compatible with all Lambda runtimes.
     */
    static fromLayerVersionArn(scope, id, layerVersionArn) {
        return LayerVersion.fromLayerVersionAttributes(scope, id, {
            layerVersionArn,
            compatibleRuntimes: runtime_1.Runtime.ALL
        });
    }
    /**
     * Imports a Layer that has been defined externally.
     *
     * @param scope the parent Construct that will use the imported layer.
     * @param id    the id of the imported layer in the construct tree.
     * @param attrs the properties of the imported layer.
     */
    static fromLayerVersionAttributes(scope, id, attrs) {
        if (attrs.compatibleRuntimes && attrs.compatibleRuntimes.length === 0) {
            throw new Error('Attempted to import a Lambda layer that supports no runtime!');
        }
        class Import extends LayerVersionBase {
            constructor() {
                super(...arguments);
                this.layerVersionArn = attrs.layerVersionArn;
                this.compatibleRuntimes = attrs.compatibleRuntimes;
            }
        }
        return new Import(scope, id);
    }
    constructor(scope, id, props) {
        super(scope, id, {
            physicalName: props.layerVersionName,
        });
        if (props.compatibleRuntimes && props.compatibleRuntimes.length === 0) {
            throw new Error('Attempted to define a Lambda layer that supports no runtime!');
        }
        if (props.code.isInline) {
            throw new Error('Lambda layers cannot be created from inline code');
        }
        // Allow usage of the code in this context...
        const code = props.code.bind(this);
        if (code.inlineCode) {
            throw new Error(`Inline code is not supported for AWS Lambda layers`);
        }
        if (!code.s3Location) {
            throw new Error(`Code must define an S3 location`);
        }
        const resource = new lambda_generated_1.CfnLayerVersion(this, 'Resource', {
            compatibleRuntimes: props.compatibleRuntimes && props.compatibleRuntimes.map(r => r.name),
            content: {
                s3Bucket: code.s3Location.bucketName,
                s3Key: code.s3Location.objectKey,
                s3ObjectVersion: code.s3Location.objectVersion
            },
            description: props.description,
            layerName: this.physicalName,
            licenseInfo: props.license,
        });
        props.code.bindToResource(resource);
        this.layerVersionArn = resource.ref;
        this.compatibleRuntimes = props.compatibleRuntimes;
    }
}
exports.LayerVersion = LayerVersion;
//# sourceMappingURL=data:application/json;base64,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