"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const iam = require("@aws-cdk/aws-iam");
const cdk = require("@aws-cdk/core");
const path = require("path");
const code_1 = require("./code");
const runtime_1 = require("./runtime");
const singleton_lambda_1 = require("./singleton-lambda");
/**
 * Creates a custom resource to control the retention policy of a CloudWatch Logs
 * log group. The log group is created if it doesn't already exist. The policy
 * is removed when `retentionDays` is `undefined` or equal to `Infinity`.
 */
class LogRetention extends cdk.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        // Custom resource provider
        const provider = new singleton_lambda_1.SingletonFunction(this, 'Provider', {
            code: code_1.Code.fromAsset(path.join(__dirname, 'log-retention-provider')),
            runtime: runtime_1.Runtime.NODEJS_8_10,
            handler: 'index.handler',
            uuid: 'aae0aa3c-5b4d-4f87-b02d-85b201efdd8a',
            lambdaPurpose: 'LogRetention',
        });
        // Duplicate statements will be deduplicated by `PolicyDocument`
        provider.addToRolePolicy(new iam.PolicyStatement({
            actions: ['logs:PutRetentionPolicy', 'logs:DeleteRetentionPolicy'],
            // We need '*' here because we will also put a retention policy on
            // the log group of the provider function. Referencing it's name
            // creates a CF circular dependency.
            resources: ['*'],
        }));
        // Need to use a CfnResource here to prevent lerna dependency cycles
        // @aws-cdk/aws-cloudformation -> @aws-cdk/aws-lambda -> @aws-cdk/aws-cloudformation
        new cdk.CfnResource(this, 'Resource', {
            type: 'Custom::LogRetention',
            properties: {
                ServiceToken: provider.functionArn,
                LogGroupName: props.logGroupName,
                RetentionInDays: props.retention === Infinity ? undefined : props.retention
            }
        });
    }
}
exports.LogRetention = LogRetention;
//# sourceMappingURL=data:application/json;base64,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