"use strict";
const assert_1 = require("@aws-cdk/assert");
const iam = require("@aws-cdk/aws-iam");
const logs = require("@aws-cdk/aws-logs");
const sqs = require("@aws-cdk/aws-sqs");
const cdk = require("@aws-cdk/core");
const path = require("path");
const lambda = require("../lib");
function newTestLambda(scope) {
    return new lambda.Function(scope, 'MyLambda', {
        code: new lambda.InlineCode('foo'),
        handler: 'bar',
        runtime: lambda.Runtime.PYTHON_2_7
    });
}
module.exports = {
    'default function'(test) {
        const stack = new cdk.Stack();
        new lambda.Function(stack, 'MyLambda', {
            code: new lambda.InlineCode('foo'),
            handler: 'index.handler',
            runtime: lambda.Runtime.NODEJS_8_10,
        });
        assert_1.expect(stack).toMatch({ Resources: { MyLambdaServiceRole4539ECB6: { Type: 'AWS::IAM::Role',
                    Properties: { AssumeRolePolicyDocument: { Statement: [{ Action: 'sts:AssumeRole',
                                    Effect: 'Allow',
                                    Principal: { Service: { "Fn::Join": ["", ['lambda.', { Ref: "AWS::URLSuffix" }]] } } }],
                            Version: '2012-10-17' },
                        ManagedPolicyArns: 
                        // arn:aws:iam::aws:policy/service-role/AWSLambdaBasicExecutionRole
                        // tslint:disable-next-line:max-line-length
                        [{ 'Fn::Join': ['', ['arn:', { Ref: 'AWS::Partition' }, ':iam::aws:policy/service-role/AWSLambdaBasicExecutionRole']] }],
                    } },
                MyLambdaCCE802FB: { Type: 'AWS::Lambda::Function',
                    Properties: { Code: { ZipFile: 'foo' },
                        Handler: 'index.handler',
                        Role: { 'Fn::GetAtt': ['MyLambdaServiceRole4539ECB6', 'Arn'] },
                        Runtime: 'nodejs8.10' },
                    DependsOn: ['MyLambdaServiceRole4539ECB6'] } } });
        test.done();
    },
    'adds policy permissions'(test) {
        const stack = new cdk.Stack();
        new lambda.Function(stack, 'MyLambda', {
            code: new lambda.InlineCode('foo'),
            handler: 'index.handler',
            runtime: lambda.Runtime.NODEJS_8_10,
            initialPolicy: [new iam.PolicyStatement({ actions: ["*"], resources: ["*"] })]
        });
        assert_1.expect(stack).toMatch({ Resources: { MyLambdaServiceRole4539ECB6: { Type: 'AWS::IAM::Role',
                    Properties: { AssumeRolePolicyDocument: { Statement: [{ Action: 'sts:AssumeRole',
                                    Effect: 'Allow',
                                    Principal: { Service: { "Fn::Join": ["", ['lambda.', { Ref: "AWS::URLSuffix" }]] } } }],
                            Version: '2012-10-17' },
                        ManagedPolicyArns: 
                        // tslint:disable-next-line:max-line-length
                        [{ 'Fn::Join': ['', ['arn:', { Ref: 'AWS::Partition' }, ':iam::aws:policy/service-role/AWSLambdaBasicExecutionRole']] }],
                    } },
                MyLambdaServiceRoleDefaultPolicy5BBC6F68: {
                    Type: "AWS::IAM::Policy",
                    Properties: {
                        PolicyDocument: {
                            Statement: [
                                {
                                    Action: "*",
                                    Effect: "Allow",
                                    Resource: "*"
                                }
                            ],
                            Version: "2012-10-17"
                        },
                        PolicyName: "MyLambdaServiceRoleDefaultPolicy5BBC6F68",
                        Roles: [
                            {
                                Ref: "MyLambdaServiceRole4539ECB6"
                            }
                        ]
                    }
                },
                MyLambdaCCE802FB: { Type: 'AWS::Lambda::Function',
                    Properties: { Code: { ZipFile: 'foo' },
                        Handler: 'index.handler',
                        Role: { 'Fn::GetAtt': ['MyLambdaServiceRole4539ECB6', 'Arn'] },
                        Runtime: 'nodejs8.10' },
                    DependsOn: ['MyLambdaServiceRoleDefaultPolicy5BBC6F68', 'MyLambdaServiceRole4539ECB6'] } } });
        test.done();
    },
    'fails if inline code is used for an invalid runtime'(test) {
        const stack = new cdk.Stack();
        test.throws(() => new lambda.Function(stack, 'MyLambda', {
            code: new lambda.InlineCode('foo'),
            handler: 'bar',
            runtime: lambda.Runtime.DOTNET_CORE_2
        }));
        test.done();
    },
    'addToResourcePolicy': {
        'can be used to add permissions to the Lambda function'(test) {
            const stack = new cdk.Stack();
            const fn = newTestLambda(stack);
            fn.addPermission('S3Permission', {
                action: 'lambda:*',
                principal: new iam.ServicePrincipal('s3.amazonaws.com'),
                sourceAccount: stack.account,
                sourceArn: 'arn:aws:s3:::my_bucket'
            });
            assert_1.expect(stack).toMatch({
                "Resources": {
                    "MyLambdaServiceRole4539ECB6": {
                        "Type": "AWS::IAM::Role",
                        "Properties": {
                            "AssumeRolePolicyDocument": {
                                "Statement": [
                                    {
                                        "Action": "sts:AssumeRole",
                                        "Effect": "Allow",
                                        "Principal": {
                                            "Service": { "Fn::Join": ["", ["lambda.", { Ref: "AWS::URLSuffix" }]] }
                                        }
                                    }
                                ],
                                "Version": "2012-10-17"
                            },
                            "ManagedPolicyArns": 
                            // tslint:disable-next-line:max-line-length
                            [{ 'Fn::Join': ['', ['arn:', { Ref: 'AWS::Partition' }, ':iam::aws:policy/service-role/AWSLambdaBasicExecutionRole']] }],
                        }
                    },
                    "MyLambdaCCE802FB": {
                        "Type": "AWS::Lambda::Function",
                        "Properties": {
                            "Code": {
                                "ZipFile": "foo"
                            },
                            "Handler": "bar",
                            "Role": {
                                "Fn::GetAtt": [
                                    "MyLambdaServiceRole4539ECB6",
                                    "Arn"
                                ]
                            },
                            "Runtime": "python2.7"
                        },
                        "DependsOn": [
                            "MyLambdaServiceRole4539ECB6"
                        ]
                    },
                    "MyLambdaS3Permission99D0EA08": {
                        "Type": "AWS::Lambda::Permission",
                        "Properties": {
                            "Action": "lambda:*",
                            "FunctionName": {
                                "Fn::GetAtt": [
                                    "MyLambdaCCE802FB",
                                    "Arn"
                                ]
                            },
                            "Principal": "s3.amazonaws.com",
                            "SourceAccount": {
                                "Ref": "AWS::AccountId"
                            },
                            "SourceArn": "arn:aws:s3:::my_bucket"
                        }
                    }
                }
            });
            test.done();
        },
        'fails if the principal is not a service, account or arn principal'(test) {
            const stack = new cdk.Stack();
            const fn = newTestLambda(stack);
            test.throws(() => fn.addPermission('F1', { principal: new iam.OrganizationPrincipal('org') }), /Invalid principal type for Lambda permission statement/);
            fn.addPermission('S1', { principal: new iam.ServicePrincipal('my-service') });
            fn.addPermission('S2', { principal: new iam.AccountPrincipal('account') });
            fn.addPermission('S3', { principal: new iam.ArnPrincipal('my:arn') });
            test.done();
        },
        'BYORole'(test) {
            // GIVEN
            const stack = new cdk.Stack();
            const role = new iam.Role(stack, 'SomeRole', {
                assumedBy: new iam.ServicePrincipal('lambda.amazonaws.com'),
            });
            role.addToPolicy(new iam.PolicyStatement({ actions: ['confirm:itsthesame'] }));
            // WHEN
            const fn = new lambda.Function(stack, 'Function', {
                code: new lambda.InlineCode('test'),
                runtime: lambda.Runtime.PYTHON_3_6,
                handler: 'index.test',
                role,
                initialPolicy: [
                    new iam.PolicyStatement({ actions: ['inline:inline'] })
                ]
            });
            fn.addToRolePolicy(new iam.PolicyStatement({ actions: ['explicit:explicit'] }));
            // THEN
            assert_1.expect(stack).to(assert_1.haveResource('AWS::IAM::Policy', {
                "PolicyDocument": {
                    "Version": "2012-10-17",
                    "Statement": [
                        { "Action": "confirm:itsthesame", "Effect": "Allow" },
                        { "Action": "inline:inline", "Effect": "Allow" },
                        { "Action": "explicit:explicit", "Effect": "Allow" }
                    ],
                },
            }));
            test.done();
        }
    },
    'fromFunctionArn'(test) {
        // GIVEN
        const stack2 = new cdk.Stack();
        // WHEN
        const imported = lambda.Function.fromFunctionArn(stack2, 'Imported', 'arn:aws:lambda:us-east-1:123456789012:function:ProcessKinesisRecords');
        // Can call addPermission() but it won't do anything
        imported.addPermission('Hello', {
            principal: new iam.ServicePrincipal('harry')
        });
        // THEN
        test.deepEqual(imported.functionArn, 'arn:aws:lambda:us-east-1:123456789012:function:ProcessKinesisRecords');
        test.deepEqual(imported.functionName, 'ProcessKinesisRecords');
        test.done();
    },
    'Lambda code can be read from a local directory via an asset'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        new lambda.Function(stack, 'MyLambda', {
            code: lambda.Code.fromAsset(path.join(__dirname, 'my-lambda-handler')),
            handler: 'index.handler',
            runtime: lambda.Runtime.PYTHON_3_6
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::Lambda::Function', {
            "Code": {
                "S3Bucket": {
                    "Ref": "MyLambdaCodeS3BucketC82A5870"
                },
                "S3Key": { "Fn::Join": ["", [
                            { "Fn::Select": [0, { "Fn::Split": ["||", { "Ref": "MyLambdaCodeS3VersionKey47762537" }] }] },
                            { "Fn::Select": [1, { "Fn::Split": ["||", { "Ref": "MyLambdaCodeS3VersionKey47762537" }] }] },
                        ]] }
            },
            "Handler": "index.handler",
            "Role": {
                "Fn::GetAtt": [
                    "MyLambdaServiceRole4539ECB6",
                    "Arn"
                ]
            },
            "Runtime": "python3.6"
        }));
        test.done();
    },
    'default function with SQS DLQ when client sets deadLetterQueueEnabled to true and functionName defined by client'(test) {
        const stack = new cdk.Stack();
        new lambda.Function(stack, 'MyLambda', {
            code: new lambda.InlineCode('foo'),
            handler: 'index.handler',
            runtime: lambda.Runtime.NODEJS_8_10,
            functionName: 'OneFunctionToRuleThemAll',
            deadLetterQueueEnabled: true
        });
        assert_1.expect(stack).toMatch({
            "Resources": {
                "MyLambdaServiceRole4539ECB6": {
                    "Type": "AWS::IAM::Role",
                    "Properties": {
                        "AssumeRolePolicyDocument": {
                            "Statement": [
                                {
                                    "Action": "sts:AssumeRole",
                                    "Effect": "Allow",
                                    "Principal": {
                                        "Service": { "Fn::Join": ["", ["lambda.", { Ref: "AWS::URLSuffix" }]] }
                                    }
                                }
                            ],
                            "Version": "2012-10-17"
                        },
                        "ManagedPolicyArns": [
                            {
                                "Fn::Join": [
                                    "",
                                    [
                                        "arn:",
                                        {
                                            "Ref": "AWS::Partition"
                                        },
                                        ":iam::aws:policy/service-role/AWSLambdaBasicExecutionRole"
                                    ]
                                ]
                            }
                        ]
                    }
                },
                "MyLambdaServiceRoleDefaultPolicy5BBC6F68": {
                    "Type": "AWS::IAM::Policy",
                    "Properties": {
                        "PolicyDocument": {
                            "Statement": [
                                {
                                    "Action": "sqs:SendMessage",
                                    "Effect": "Allow",
                                    "Resource": {
                                        "Fn::GetAtt": [
                                            "MyLambdaDeadLetterQueue399EEA2D",
                                            "Arn"
                                        ]
                                    }
                                }
                            ],
                            "Version": "2012-10-17"
                        },
                        "PolicyName": "MyLambdaServiceRoleDefaultPolicy5BBC6F68",
                        "Roles": [
                            {
                                "Ref": "MyLambdaServiceRole4539ECB6"
                            }
                        ]
                    }
                },
                "MyLambdaDeadLetterQueue399EEA2D": {
                    "Type": "AWS::SQS::Queue",
                    "Properties": {
                        "MessageRetentionPeriod": 1209600
                    }
                },
                "MyLambdaCCE802FB": {
                    "Type": "AWS::Lambda::Function",
                    "Properties": {
                        "Code": {
                            "ZipFile": "foo"
                        },
                        "Handler": "index.handler",
                        "Role": {
                            "Fn::GetAtt": [
                                "MyLambdaServiceRole4539ECB6",
                                "Arn"
                            ]
                        },
                        "Runtime": "nodejs8.10",
                        "DeadLetterConfig": {
                            "TargetArn": {
                                "Fn::GetAtt": [
                                    "MyLambdaDeadLetterQueue399EEA2D",
                                    "Arn"
                                ]
                            }
                        },
                        "FunctionName": "OneFunctionToRuleThemAll"
                    },
                    "DependsOn": [
                        "MyLambdaServiceRoleDefaultPolicy5BBC6F68",
                        "MyLambdaServiceRole4539ECB6"
                    ]
                }
            }
        });
        test.done();
    },
    'default function with SQS DLQ when client sets deadLetterQueueEnabled to true and functionName not defined by client'(test) {
        const stack = new cdk.Stack();
        new lambda.Function(stack, 'MyLambda', {
            code: new lambda.InlineCode('foo'),
            handler: 'index.handler',
            runtime: lambda.Runtime.NODEJS_8_10,
            deadLetterQueueEnabled: true,
        });
        assert_1.expect(stack).toMatch({
            "Resources": {
                "MyLambdaServiceRole4539ECB6": {
                    "Type": "AWS::IAM::Role",
                    "Properties": {
                        "AssumeRolePolicyDocument": {
                            "Statement": [
                                {
                                    "Action": "sts:AssumeRole",
                                    "Effect": "Allow",
                                    "Principal": {
                                        "Service": { "Fn::Join": ["", ["lambda.", { Ref: "AWS::URLSuffix" }]] }
                                    }
                                }
                            ],
                            "Version": "2012-10-17"
                        },
                        "ManagedPolicyArns": [
                            {
                                "Fn::Join": [
                                    "",
                                    [
                                        "arn:",
                                        {
                                            "Ref": "AWS::Partition"
                                        },
                                        ":iam::aws:policy/service-role/AWSLambdaBasicExecutionRole"
                                    ]
                                ]
                            }
                        ]
                    }
                },
                "MyLambdaServiceRoleDefaultPolicy5BBC6F68": {
                    "Type": "AWS::IAM::Policy",
                    "Properties": {
                        "PolicyDocument": {
                            "Statement": [
                                {
                                    "Action": "sqs:SendMessage",
                                    "Effect": "Allow",
                                    "Resource": {
                                        "Fn::GetAtt": [
                                            "MyLambdaDeadLetterQueue399EEA2D",
                                            "Arn"
                                        ]
                                    }
                                }
                            ],
                            "Version": "2012-10-17"
                        },
                        "PolicyName": "MyLambdaServiceRoleDefaultPolicy5BBC6F68",
                        "Roles": [
                            {
                                "Ref": "MyLambdaServiceRole4539ECB6"
                            }
                        ]
                    }
                },
                "MyLambdaDeadLetterQueue399EEA2D": {
                    "Type": "AWS::SQS::Queue",
                    "Properties": {
                        "MessageRetentionPeriod": 1209600
                    }
                },
                "MyLambdaCCE802FB": {
                    "Type": "AWS::Lambda::Function",
                    "Properties": {
                        "Code": {
                            "ZipFile": "foo"
                        },
                        "Handler": "index.handler",
                        "Role": {
                            "Fn::GetAtt": [
                                "MyLambdaServiceRole4539ECB6",
                                "Arn"
                            ]
                        },
                        "Runtime": "nodejs8.10",
                        "DeadLetterConfig": {
                            "TargetArn": {
                                "Fn::GetAtt": [
                                    "MyLambdaDeadLetterQueue399EEA2D",
                                    "Arn"
                                ]
                            }
                        }
                    },
                    "DependsOn": [
                        "MyLambdaServiceRoleDefaultPolicy5BBC6F68",
                        "MyLambdaServiceRole4539ECB6"
                    ]
                }
            }
        });
        test.done();
    },
    'default function with SQS DLQ when client sets deadLetterQueueEnabled to false'(test) {
        const stack = new cdk.Stack();
        new lambda.Function(stack, 'MyLambda', {
            code: new lambda.InlineCode('foo'),
            handler: 'index.handler',
            runtime: lambda.Runtime.NODEJS_8_10,
            deadLetterQueueEnabled: false,
        });
        assert_1.expect(stack).toMatch({
            "Resources": {
                "MyLambdaServiceRole4539ECB6": {
                    "Type": "AWS::IAM::Role",
                    "Properties": {
                        "AssumeRolePolicyDocument": {
                            "Statement": [
                                {
                                    "Action": "sts:AssumeRole",
                                    "Effect": "Allow",
                                    "Principal": {
                                        "Service": { "Fn::Join": ["", ["lambda.", { Ref: "AWS::URLSuffix" }]] }
                                    }
                                }
                            ],
                            "Version": "2012-10-17"
                        },
                        "ManagedPolicyArns": [
                            {
                                "Fn::Join": [
                                    "",
                                    [
                                        "arn:",
                                        {
                                            "Ref": "AWS::Partition"
                                        },
                                        ":iam::aws:policy/service-role/AWSLambdaBasicExecutionRole"
                                    ]
                                ]
                            }
                        ]
                    }
                },
                "MyLambdaCCE802FB": {
                    "Type": "AWS::Lambda::Function",
                    "Properties": {
                        "Code": {
                            "ZipFile": "foo"
                        },
                        "Handler": "index.handler",
                        "Role": {
                            "Fn::GetAtt": [
                                "MyLambdaServiceRole4539ECB6",
                                "Arn"
                            ]
                        },
                        "Runtime": "nodejs8.10"
                    },
                    "DependsOn": [
                        "MyLambdaServiceRole4539ECB6"
                    ]
                }
            }
        });
        test.done();
    },
    'default function with SQS DLQ when client provides Queue to be used as DLQ'(test) {
        const stack = new cdk.Stack();
        const dlQueue = new sqs.Queue(stack, 'DeadLetterQueue', {
            queueName: 'MyLambda_DLQ',
            retentionPeriod: cdk.Duration.days(14)
        });
        new lambda.Function(stack, 'MyLambda', {
            code: new lambda.InlineCode('foo'),
            handler: 'index.handler',
            runtime: lambda.Runtime.NODEJS_8_10,
            deadLetterQueue: dlQueue,
        });
        assert_1.expect(stack).toMatch({
            "Resources": {
                "MyLambdaServiceRole4539ECB6": {
                    "Type": "AWS::IAM::Role",
                    "Properties": {
                        "AssumeRolePolicyDocument": {
                            "Statement": [
                                {
                                    "Action": "sts:AssumeRole",
                                    "Effect": "Allow",
                                    "Principal": {
                                        "Service": { "Fn::Join": ["", ["lambda.", { Ref: "AWS::URLSuffix" }]] }
                                    }
                                }
                            ],
                            "Version": "2012-10-17"
                        },
                        "ManagedPolicyArns": [
                            {
                                "Fn::Join": [
                                    "",
                                    [
                                        "arn:",
                                        {
                                            "Ref": "AWS::Partition"
                                        },
                                        ":iam::aws:policy/service-role/AWSLambdaBasicExecutionRole"
                                    ]
                                ]
                            }
                        ]
                    }
                },
                "MyLambdaServiceRoleDefaultPolicy5BBC6F68": {
                    "Type": "AWS::IAM::Policy",
                    "Properties": {
                        "PolicyDocument": {
                            "Statement": [
                                {
                                    "Action": "sqs:SendMessage",
                                    "Effect": "Allow",
                                    "Resource": {
                                        "Fn::GetAtt": [
                                            "DeadLetterQueue9F481546",
                                            "Arn"
                                        ]
                                    }
                                }
                            ],
                            "Version": "2012-10-17"
                        },
                        "PolicyName": "MyLambdaServiceRoleDefaultPolicy5BBC6F68",
                        "Roles": [
                            {
                                "Ref": "MyLambdaServiceRole4539ECB6"
                            }
                        ]
                    }
                },
                "MyLambdaCCE802FB": {
                    "Type": "AWS::Lambda::Function",
                    "Properties": {
                        "Code": {
                            "ZipFile": "foo"
                        },
                        "Handler": "index.handler",
                        "Role": {
                            "Fn::GetAtt": [
                                "MyLambdaServiceRole4539ECB6",
                                "Arn"
                            ]
                        },
                        "Runtime": "nodejs8.10",
                        "DeadLetterConfig": {
                            "TargetArn": {
                                "Fn::GetAtt": [
                                    "DeadLetterQueue9F481546",
                                    "Arn"
                                ]
                            }
                        }
                    },
                    "DependsOn": [
                        "MyLambdaServiceRoleDefaultPolicy5BBC6F68",
                        "MyLambdaServiceRole4539ECB6"
                    ]
                }
            }
        }, assert_1.MatchStyle.SUPERSET);
        test.done();
    },
    'default function with SQS DLQ when client provides Queue to be used as DLQ and deadLetterQueueEnabled set to true'(test) {
        const stack = new cdk.Stack();
        const dlQueue = new sqs.Queue(stack, 'DeadLetterQueue', {
            queueName: 'MyLambda_DLQ',
            retentionPeriod: cdk.Duration.days(14)
        });
        new lambda.Function(stack, 'MyLambda', {
            code: new lambda.InlineCode('foo'),
            handler: 'index.handler',
            runtime: lambda.Runtime.NODEJS_8_10,
            deadLetterQueueEnabled: true,
            deadLetterQueue: dlQueue,
        });
        assert_1.expect(stack).toMatch({
            "Resources": {
                "MyLambdaServiceRole4539ECB6": {
                    "Type": "AWS::IAM::Role",
                    "Properties": {
                        "AssumeRolePolicyDocument": {
                            "Statement": [
                                {
                                    "Action": "sts:AssumeRole",
                                    "Effect": "Allow",
                                    "Principal": {
                                        "Service": { "Fn::Join": ["", ["lambda.", { Ref: "AWS::URLSuffix" }]] }
                                    }
                                }
                            ],
                            "Version": "2012-10-17"
                        },
                        "ManagedPolicyArns": [
                            {
                                "Fn::Join": [
                                    "",
                                    [
                                        "arn:",
                                        {
                                            "Ref": "AWS::Partition"
                                        },
                                        ":iam::aws:policy/service-role/AWSLambdaBasicExecutionRole"
                                    ]
                                ]
                            }
                        ]
                    }
                },
                "MyLambdaServiceRoleDefaultPolicy5BBC6F68": {
                    "Type": "AWS::IAM::Policy",
                    "Properties": {
                        "PolicyDocument": {
                            "Statement": [
                                {
                                    "Action": "sqs:SendMessage",
                                    "Effect": "Allow",
                                    "Resource": {
                                        "Fn::GetAtt": [
                                            "DeadLetterQueue9F481546",
                                            "Arn"
                                        ]
                                    }
                                }
                            ],
                            "Version": "2012-10-17"
                        },
                        "PolicyName": "MyLambdaServiceRoleDefaultPolicy5BBC6F68",
                        "Roles": [
                            {
                                "Ref": "MyLambdaServiceRole4539ECB6"
                            }
                        ]
                    }
                },
                "MyLambdaCCE802FB": {
                    "Type": "AWS::Lambda::Function",
                    "Properties": {
                        "Code": {
                            "ZipFile": "foo"
                        },
                        "Handler": "index.handler",
                        "Role": {
                            "Fn::GetAtt": [
                                "MyLambdaServiceRole4539ECB6",
                                "Arn"
                            ]
                        },
                        "Runtime": "nodejs8.10",
                        "DeadLetterConfig": {
                            "TargetArn": {
                                "Fn::GetAtt": [
                                    "DeadLetterQueue9F481546",
                                    "Arn"
                                ]
                            }
                        }
                    },
                    "DependsOn": [
                        "MyLambdaServiceRoleDefaultPolicy5BBC6F68",
                        "MyLambdaServiceRole4539ECB6",
                    ]
                }
            }
        }, assert_1.MatchStyle.SUPERSET);
        test.done();
    },
    'error when default function with SQS DLQ when client provides Queue to be used as DLQ and deadLetterQueueEnabled set to false'(test) {
        const stack = new cdk.Stack();
        const dlQueue = new sqs.Queue(stack, 'DeadLetterQueue', {
            queueName: 'MyLambda_DLQ',
            retentionPeriod: cdk.Duration.days(14),
        });
        test.throws(() => new lambda.Function(stack, 'MyLambda', {
            code: new lambda.InlineCode('foo'),
            handler: 'index.handler',
            runtime: lambda.Runtime.NODEJS_8_10,
            deadLetterQueueEnabled: false,
            deadLetterQueue: dlQueue,
        }), /deadLetterQueue defined but deadLetterQueueEnabled explicitly set to false/);
        test.done();
    },
    'default function with Active tracing'(test) {
        const stack = new cdk.Stack();
        new lambda.Function(stack, 'MyLambda', {
            code: new lambda.InlineCode('foo'),
            handler: 'index.handler',
            runtime: lambda.Runtime.NODEJS_8_10,
            tracing: lambda.Tracing.ACTIVE
        });
        assert_1.expect(stack).to(assert_1.haveResource('AWS::IAM::Policy', {
            "PolicyDocument": {
                "Statement": [
                    {
                        "Action": [
                            "xray:PutTraceSegments",
                            "xray:PutTelemetryRecords"
                        ],
                        "Effect": "Allow",
                        "Resource": "*"
                    }
                ],
                "Version": "2012-10-17"
            },
            "PolicyName": "MyLambdaServiceRoleDefaultPolicy5BBC6F68",
            "Roles": [
                {
                    "Ref": "MyLambdaServiceRole4539ECB6"
                }
            ]
        }));
        assert_1.expect(stack).to(assert_1.haveResource('AWS::Lambda::Function', {
            "Properties": {
                "Code": {
                    "ZipFile": "foo"
                },
                "Handler": "index.handler",
                "Role": {
                    "Fn::GetAtt": [
                        "MyLambdaServiceRole4539ECB6",
                        "Arn"
                    ]
                },
                "Runtime": "nodejs8.10",
                "TracingConfig": {
                    "Mode": "Active"
                }
            },
            "DependsOn": [
                "MyLambdaServiceRoleDefaultPolicy5BBC6F68",
                "MyLambdaServiceRole4539ECB6",
            ]
        }, assert_1.ResourcePart.CompleteDefinition));
        test.done();
    },
    'default function with PassThrough tracing'(test) {
        const stack = new cdk.Stack();
        new lambda.Function(stack, 'MyLambda', {
            code: new lambda.InlineCode('foo'),
            handler: 'index.handler',
            runtime: lambda.Runtime.NODEJS_8_10,
            tracing: lambda.Tracing.PASS_THROUGH
        });
        assert_1.expect(stack).to(assert_1.haveResource('AWS::IAM::Policy', {
            "PolicyDocument": {
                "Statement": [
                    {
                        "Action": [
                            "xray:PutTraceSegments",
                            "xray:PutTelemetryRecords"
                        ],
                        "Effect": "Allow",
                        "Resource": "*"
                    }
                ],
                "Version": "2012-10-17"
            },
            "PolicyName": "MyLambdaServiceRoleDefaultPolicy5BBC6F68",
            "Roles": [
                {
                    "Ref": "MyLambdaServiceRole4539ECB6"
                }
            ]
        }));
        assert_1.expect(stack).to(assert_1.haveResource('AWS::Lambda::Function', {
            "Properties": {
                "Code": {
                    "ZipFile": "foo"
                },
                "Handler": "index.handler",
                "Role": {
                    "Fn::GetAtt": [
                        "MyLambdaServiceRole4539ECB6",
                        "Arn"
                    ]
                },
                "Runtime": "nodejs8.10",
                "TracingConfig": {
                    "Mode": "PassThrough"
                }
            },
            "DependsOn": [
                "MyLambdaServiceRoleDefaultPolicy5BBC6F68",
                "MyLambdaServiceRole4539ECB6",
            ]
        }, assert_1.ResourcePart.CompleteDefinition));
        test.done();
    },
    'default function with Disabled tracing'(test) {
        const stack = new cdk.Stack();
        new lambda.Function(stack, 'MyLambda', {
            code: new lambda.InlineCode('foo'),
            handler: 'index.handler',
            runtime: lambda.Runtime.NODEJS_8_10,
            tracing: lambda.Tracing.DISABLED
        });
        assert_1.expect(stack).notTo(assert_1.haveResource('AWS::IAM::Policy', {
            "PolicyDocument": {
                "Statement": [
                    {
                        "Action": [
                            "xray:PutTraceSegments",
                            "xray:PutTelemetryRecords"
                        ],
                        "Effect": "Allow",
                        "Resource": "*"
                    }
                ],
                "Version": "2012-10-17"
            },
            "PolicyName": "MyLambdaServiceRoleDefaultPolicy5BBC6F68",
            "Roles": [
                {
                    "Ref": "MyLambdaServiceRole4539ECB6"
                }
            ]
        }));
        assert_1.expect(stack).to(assert_1.haveResource('AWS::Lambda::Function', {
            "Properties": {
                "Code": {
                    "ZipFile": "foo"
                },
                "Handler": "index.handler",
                "Role": {
                    "Fn::GetAtt": [
                        "MyLambdaServiceRole4539ECB6",
                        "Arn"
                    ]
                },
                "Runtime": "nodejs8.10"
            },
            "DependsOn": [
                "MyLambdaServiceRole4539ECB6"
            ]
        }, assert_1.ResourcePart.CompleteDefinition));
        test.done();
    },
    'grantInvoke adds iam:InvokeFunction'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const role = new iam.Role(stack, 'Role', {
            assumedBy: new iam.AccountPrincipal('1234'),
        });
        const fn = new lambda.Function(stack, 'Function', {
            code: lambda.Code.fromInline('xxx'),
            handler: 'index.handler',
            runtime: lambda.Runtime.NODEJS_8_10,
        });
        // WHEN
        fn.grantInvoke(role);
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::IAM::Policy', {
            PolicyDocument: {
                Version: '2012-10-17',
                Statement: [
                    {
                        Action: 'lambda:InvokeFunction',
                        Effect: 'Allow',
                        Resource: { "Fn::GetAtt": ["Function76856677", "Arn"] }
                    }
                ]
            }
        }));
        test.done();
    },
    'grantInvoke with a service principal'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const fn = new lambda.Function(stack, 'Function', {
            code: lambda.Code.fromInline('xxx'),
            handler: 'index.handler',
            runtime: lambda.Runtime.NODEJS_8_10,
        });
        const service = new iam.ServicePrincipal('apigateway.amazonaws.com');
        // WHEN
        fn.grantInvoke(service);
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::Lambda::Permission', {
            Action: 'lambda:InvokeFunction',
            FunctionName: {
                'Fn::GetAtt': [
                    'Function76856677',
                    'Arn'
                ]
            },
            Principal: 'apigateway.amazonaws.com'
        }));
        test.done();
    },
    'grantInvoke with an account principal'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const fn = new lambda.Function(stack, 'Function', {
            code: lambda.Code.fromInline('xxx'),
            handler: 'index.handler',
            runtime: lambda.Runtime.NODEJS_8_10,
        });
        const account = new iam.AccountPrincipal('123456789012');
        // WHEN
        fn.grantInvoke(account);
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::Lambda::Permission', {
            Action: 'lambda:InvokeFunction',
            FunctionName: {
                'Fn::GetAtt': [
                    'Function76856677',
                    'Arn'
                ]
            },
            Principal: '123456789012'
        }));
        test.done();
    },
    'grantInvoke with an arn principal'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const fn = new lambda.Function(stack, 'Function', {
            code: lambda.Code.fromInline('xxx'),
            handler: 'index.handler',
            runtime: lambda.Runtime.NODEJS_8_10,
        });
        const account = new iam.ArnPrincipal('arn:aws:iam::123456789012:role/someRole');
        // WHEN
        fn.grantInvoke(account);
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::Lambda::Permission', {
            Action: 'lambda:InvokeFunction',
            FunctionName: {
                'Fn::GetAtt': [
                    'Function76856677',
                    'Arn'
                ]
            },
            Principal: 'arn:aws:iam::123456789012:role/someRole'
        }));
        test.done();
    },
    'Can use metricErrors on a lambda Function'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const fn = new lambda.Function(stack, 'Function', {
            code: lambda.Code.fromInline('xxx'),
            handler: 'index.handler',
            runtime: lambda.Runtime.NODEJS_8_10,
        });
        // THEN
        test.deepEqual(stack.resolve(fn.metricErrors()), {
            dimensions: { FunctionName: { Ref: 'Function76856677' } },
            namespace: 'AWS/Lambda',
            metricName: 'Errors',
            period: cdk.Duration.minutes(5),
            statistic: 'Sum',
        });
        test.done();
    },
    'addEventSource calls bind'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const fn = new lambda.Function(stack, 'Function', {
            code: lambda.Code.fromInline('xxx'),
            handler: 'index.handler',
            runtime: lambda.Runtime.NODEJS_8_10,
        });
        let bindTarget;
        class EventSourceMock {
            bind(target) {
                bindTarget = target;
            }
        }
        // WHEN
        fn.addEventSource(new EventSourceMock());
        // THEN
        test.same(bindTarget, fn);
        test.done();
    },
    'support inline code for Ruby runtime'(test) {
        const stack = new cdk.Stack();
        new lambda.Function(stack, 'MyLambda', {
            code: new lambda.InlineCode('foo'),
            handler: 'index.handler',
            runtime: lambda.Runtime.RUBY_2_5,
        });
        assert_1.expect(stack).toMatch({ Resources: { MyLambdaServiceRole4539ECB6: { Type: 'AWS::IAM::Role',
                    Properties: { AssumeRolePolicyDocument: { Statement: [{ Action: 'sts:AssumeRole',
                                    Effect: 'Allow',
                                    Principal: { Service: { "Fn::Join": ["", ['lambda.', { Ref: "AWS::URLSuffix" }]] } } }],
                            Version: '2012-10-17' },
                        ManagedPolicyArns: 
                        // arn:aws:iam::aws:policy/service-role/AWSLambdaBasicExecutionRole
                        // tslint:disable-next-line:max-line-length
                        [{ 'Fn::Join': ['', ['arn:', { Ref: 'AWS::Partition' }, ':iam::aws:policy/service-role/AWSLambdaBasicExecutionRole']] }],
                    } },
                MyLambdaCCE802FB: { Type: 'AWS::Lambda::Function',
                    Properties: { Code: { ZipFile: 'foo' },
                        Handler: 'index.handler',
                        Role: { 'Fn::GetAtt': ['MyLambdaServiceRole4539ECB6', 'Arn'] },
                        Runtime: 'ruby2.5' },
                    DependsOn: ['MyLambdaServiceRole4539ECB6'] } } });
        test.done();
    },
    'using an incompatible layer'(test) {
        // GIVEN
        const stack = new cdk.Stack(undefined, 'TestStack');
        const layer = lambda.LayerVersion.fromLayerVersionAttributes(stack, 'TestLayer', {
            layerVersionArn: 'arn:aws:...',
            compatibleRuntimes: [lambda.Runtime.NODEJS_8_10],
        });
        // THEN
        test.throws(() => new lambda.Function(stack, 'Function', {
            layers: [layer],
            runtime: lambda.Runtime.NODEJS_6_10,
            code: lambda.Code.fromInline('exports.main = function() { console.log("DONE"); }'),
            handler: 'index.main'
        }), /nodejs6.10 is not in \[nodejs8.10\]/);
        test.done();
    },
    'using more than 5 layers'(test) {
        // GIVEN
        const stack = new cdk.Stack(undefined, 'TestStack');
        const layers = new Array(6).fill(lambda.LayerVersion.fromLayerVersionAttributes(stack, 'TestLayer', {
            layerVersionArn: 'arn:aws:...',
            compatibleRuntimes: [lambda.Runtime.NODEJS_8_10],
        }));
        // THEN
        test.throws(() => new lambda.Function(stack, 'Function', {
            layers,
            runtime: lambda.Runtime.NODEJS_8_10,
            code: lambda.Code.fromInline('exports.main = function() { console.log("DONE"); }'),
            handler: 'index.main'
        }), /Unable to add layer:/);
        test.done();
    },
    'environment variables are prohibited in China'(test) {
        // GIVEN
        const stack = new cdk.Stack(undefined, undefined, { env: { region: 'cn-north-1' } });
        // WHEN
        test.throws(() => {
            new lambda.Function(stack, 'MyLambda', {
                code: new lambda.InlineCode('foo'),
                handler: 'index.handler',
                runtime: lambda.Runtime.NODEJS,
                environment: {
                    SOME: 'Variable'
                }
            });
        }, /Environment variables are not supported/);
        test.done();
    },
    'environment variables work in an unspecified region'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        // WHEN
        new lambda.Function(stack, 'MyLambda', {
            code: new lambda.InlineCode('foo'),
            handler: 'index.handler',
            runtime: lambda.Runtime.NODEJS,
            environment: {
                SOME: 'Variable'
            }
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::Lambda::Function', {
            Environment: {
                Variables: {
                    SOME: "Variable"
                }
            }
        }));
        test.done();
    },
    'support reserved concurrent executions'(test) {
        const stack = new cdk.Stack();
        new lambda.Function(stack, 'MyLambda', {
            code: new lambda.InlineCode('foo'),
            handler: 'index.handler',
            runtime: lambda.Runtime.NODEJS,
            reservedConcurrentExecutions: 10
        });
        assert_1.expect(stack).toMatch({ Resources: { MyLambdaServiceRole4539ECB6: { Type: 'AWS::IAM::Role',
                    Properties: { AssumeRolePolicyDocument: { Statement: [{ Action: 'sts:AssumeRole',
                                    Effect: 'Allow',
                                    Principal: { Service: { "Fn::Join": ["", ['lambda.', { Ref: "AWS::URLSuffix" }]] } } }],
                            Version: '2012-10-17' },
                        ManagedPolicyArns: 
                        // arn:aws:iam::aws:policy/service-role/AWSLambdaBasicExecutionRole
                        // tslint:disable-next-line:max-line-length
                        [{ 'Fn::Join': ['', ['arn:', { Ref: 'AWS::Partition' }, ':iam::aws:policy/service-role/AWSLambdaBasicExecutionRole']] }],
                    } },
                MyLambdaCCE802FB: { Type: 'AWS::Lambda::Function',
                    Properties: { Code: { ZipFile: 'foo' },
                        Handler: 'index.handler',
                        ReservedConcurrentExecutions: 10,
                        Role: { 'Fn::GetAtt': ['MyLambdaServiceRole4539ECB6', 'Arn'] },
                        Runtime: 'nodejs' },
                    DependsOn: ['MyLambdaServiceRole4539ECB6'] } } });
        test.done();
    },
    'its possible to specify event sources upon creation'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        let bindCount = 0;
        class EventSource {
            bind(_) {
                bindCount++;
            }
        }
        // WHEN
        new lambda.Function(stack, 'fn', {
            code: lambda.Code.fromInline('boom'),
            runtime: lambda.Runtime.NODEJS_8_10,
            handler: 'index.bam',
            events: [
                new EventSource(),
                new EventSource(),
            ]
        });
        // THEN
        test.deepEqual(bindCount, 2);
        test.done();
    },
    'Provided Runtime returns the right values'(test) {
        const rt = lambda.Runtime.PROVIDED;
        test.equal(rt.name, 'provided');
        test.equal(rt.supportsInlineCode, false);
        test.done();
    },
    'specify log retention'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        // WHEN
        new lambda.Function(stack, 'MyLambda', {
            code: new lambda.InlineCode('foo'),
            handler: 'index.handler',
            runtime: lambda.Runtime.NODEJS,
            logRetention: logs.RetentionDays.ONE_MONTH
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('Custom::LogRetention', {
            'LogGroupName': {
                'Fn::Join': [
                    '',
                    [
                        '/aws/lambda/',
                        {
                            Ref: 'MyLambdaCCE802FB'
                        }
                    ]
                ]
            },
            'RetentionInDays': 30
        }));
        test.done();
    }
};
//# sourceMappingURL=data:application/json;base64,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