#!/usr/bin/env python3

"""
Tests for GUI entry Widget with autocompletion features.

This file is part of Ardupilot methodic configurator. https://github.com/ArduPilot/MethodicConfigurator

SPDX-FileCopyrightText: 2024-2025 Amilcar do Carmo Lucas <amilcar.lucas@iav.de>

SPDX-License-Identifier: GPL-3.0-or-later
"""

import tkinter as tk
from tkinter.constants import END
from unittest.mock import MagicMock, patch

import pytest

from ardupilot_methodic_configurator.frontend_tkinter_entry_dynamic import EntryWithDynamicalyFilteredListbox, autoscroll


class TestEntryWithDynamicalyFilteredListbox:  # pylint: disable=too-many-public-methods
    """Test the EntryWithDynamicalyFilteredListbox widget."""

    # pylint: disable=protected-access
    @pytest.fixture
    def setup_widget(self) -> tuple:
        root = tk.Tk()
        list_of_items = ["item1", "item2", "item3", "anotheritem", "test"]
        widget = EntryWithDynamicalyFilteredListbox(root, list_of_items=list_of_items)
        yield root, widget
        root.destroy()

    def test_initialization(self, setup_widget) -> None:
        _, widget = setup_widget
        assert widget._list_of_items == ["item1", "item2", "item3", "anotheritem", "test"]
        assert widget._listbox_height == 12
        assert widget._startswith_match is True
        assert widget._ignorecase_match is False
        assert widget._use_vscrollbar is True
        assert widget._use_hscrollbar is True
        assert widget._listbox is None

    def test_initialization_with_empty_list_match_error(self) -> None:
        root = tk.Tk()
        with pytest.raises(ValueError, match="List_of_items can't be 'None'"):
            EntryWithDynamicalyFilteredListbox(root, list_of_items=None)
        root.destroy()

    def test_initialization_with_custom_parameters(self) -> None:
        """Test initialization with non-default parameters."""
        root = tk.Tk()

        def custom_filter(x) -> list:
            return [i for i in ["a", "b", "c"] if x in i]

        widget = EntryWithDynamicalyFilteredListbox(
            root,
            list_of_items=["test1", "test2"],
            custom_filter_function=custom_filter,
            listbox_width=200,
            listbox_height=5,
            ignorecase_match=True,
            startswith_match=False,
            vscrollbar=False,
            hscrollbar=False,
        )

        assert widget._list_of_items == ["test1", "test2"]
        assert widget.filter_function == custom_filter  # pylint: disable=comparison-with-callable
        assert widget._listbox_width == 200
        assert widget._listbox_height == 5
        assert widget._ignorecase_match is True
        assert widget._startswith_match is False
        assert widget._use_vscrollbar is False
        assert widget._use_hscrollbar is False

        # Test that the custom filter works
        assert widget.filter_function("a") == ["a"]

        root.destroy()

    def test_on_change_entry_var_empty_string(self, setup_widget) -> None:
        _, widget = setup_widget
        # Set up mock for unpost_listbox
        widget.unpost_listbox = MagicMock()
        widget.focus = MagicMock()

        # Call method with empty string
        widget._entry_var.set("")

        # Verify unpost_listbox was called
        widget.unpost_listbox.assert_called_once()
        widget.focus.assert_called_once()

    @patch("tkinter.ttk.Frame")
    def test_on_change_entry_var_with_values(self, mock_frame, setup_widget) -> None:  # pylint: disable=unused-argument
        _, widget = setup_widget
        # Mock build_listbox method
        widget._build_listbox = MagicMock()

        # Call method with a value that will match items
        widget._entry_var.set("item")

        # Verify build_listbox was called with filtered values
        widget._build_listbox.assert_called_once()
        # Check if the call argument is a list with expected items
        args, _ = widget._build_listbox.call_args
        assert args[0] == ["item1", "item2", "item3"]

    @patch("tkinter.ttk.Frame")
    def test_on_change_entry_var_with_existing_listbox(self, mock_frame, setup_widget) -> None:  # pylint: disable=unused-argument
        _, widget = setup_widget
        # Create mock listbox
        widget._listbox = MagicMock()
        widget._listbox.delete = MagicMock()
        widget._listbox.configure = MagicMock()
        widget._listbox.insert = MagicMock()

        # Call method with a value that will match items
        widget._entry_var.set("item")

        # Verify listbox methods were called
        widget._listbox.delete.assert_called_once_with(0, END)
        widget._listbox.configure.assert_called_once()
        # Insert should be called for each matching item
        assert widget._listbox.insert.call_count == 3

    def test_on_change_entry_var_no_matches(self, setup_widget) -> None:
        _, widget = setup_widget
        # Set up mocks
        widget.unpost_listbox = MagicMock()
        widget.focus = MagicMock()
        # Use a custom filter function that returns empty list
        widget.filter_function = lambda _x: []

        # Call method with any string
        widget._entry_var.set("xyz")

        # Verify unpost_listbox was called
        widget.unpost_listbox.assert_called_once()
        widget.focus.assert_called_once()

    def test_post_listbox_no_text(self, setup_widget) -> None:
        _, widget = setup_widget
        # Mock build_listbox
        widget._build_listbox = MagicMock()

        # Empty entry
        widget._entry_var.set("")

        # Call post_listbox
        widget.post_listbox()

        # Verify build_listbox was not called
        widget._build_listbox.assert_not_called()

    def test_post_listbox_with_existing_listbox(self, setup_widget) -> None:
        _, widget = setup_widget
        # Create mock listbox
        widget._listbox = MagicMock()
        # Mock build_listbox
        widget._build_listbox = MagicMock()

        # Set entry text
        widget._entry_var.set("item")

        # Call post_listbox
        widget.post_listbox()

        # Verify build_listbox was not called since listbox already exists
        widget._build_listbox.assert_not_called()

    def test_post_listbox_with_filtered_results(self, setup_widget) -> None:
        _, widget = setup_widget

        # First disable the trace callback to prevent _build_listbox from being called when setting entry_var
        widget._entry_var.trace_remove("write", widget._trace_id)

        # Set entry text without triggering callback
        widget._entry_var.set("item")

        # Replace _build_listbox with mock
        widget._build_listbox = MagicMock()

        # Ensure listbox is None
        widget._listbox = None

        # Call post_listbox directly
        widget.post_listbox()

        # Verify build_listbox was called exactly once with filtered values
        widget._build_listbox.assert_called_once()
        # Check the arguments
        args, _ = widget._build_listbox.call_args
        assert args[0] == ["item1", "item2", "item3"]

        # Restore trace for cleanup
        widget._trace_id = widget._entry_var.trace_add("write", widget._on_change_entry_var)

    def test_set_var(self, setup_widget) -> None:
        _, widget = setup_widget
        # Record initial trace id
        initial_trace_id = widget._trace_id

        # Call _set_var
        widget._set_var("new_text")

        # Verify entry var contains new text
        assert widget._entry_var.get() == "new_text"
        # Verify trace_id has changed
        assert widget._trace_id != initial_trace_id

    def test_update_entry_from_listbox_no_selection(self, setup_widget) -> None:
        _, widget = setup_widget

        # Create a proper mock for listbox with necessary attributes
        widget._listbox = MagicMock()
        widget._listbox.curselection.return_value = ()  # Empty selection

        # Mock the listbox.master to prevent destroy() call from removing the listbox
        listbox_master_mock = MagicMock()
        widget._listbox.master = listbox_master_mock

        # Call method
        result = widget.update_entry_from_listbox(None)

        # Verify behavior with no selection
        assert result == "break"

        # The method should still have attempted to destroy the listbox master
        listbox_master_mock.destroy.assert_called_once()

        # But we would expect that in the actual code, when there's no selection,
        # the destroy() would be skipped and the listbox would remain
        # This is a test limitation since we're mocking the behavior

    def test_next_navigation_no_selection(self, setup_widget) -> None:
        _, widget = setup_widget
        # Set up mock listbox with no selection
        widget._listbox = MagicMock()
        widget._listbox.curselection.return_value = []

        # Call method
        result = widget._next(None)

        # Verify selection was set to first item
        widget._listbox.selection_set.assert_called_with(0)
        widget._listbox.activate.assert_called_with(0)
        assert result == "break"

    def test_next_navigation_last_item(self, setup_widget) -> None:
        _, widget = setup_widget
        # Set up mock listbox with last item selected
        widget._listbox = MagicMock()
        widget._listbox.curselection.return_value = [2]  # Last item index
        widget._listbox.size.return_value = 3

        # Call method
        result = widget._next(None)

        # Verify selection wrapped to first item
        widget._listbox.selection_clear.assert_called_with(2)
        widget._listbox.selection_set.assert_called_with(0)
        widget._listbox.activate.assert_called_with(0)
        assert result == "break"

    def test_previous_navigation_no_selection(self, setup_widget) -> None:
        _, widget = setup_widget
        # Set up mock listbox with no selection
        widget._listbox = MagicMock()
        widget._listbox.curselection.return_value = []

        # Call method
        result = widget._previous(None)

        # Verify selection was set to first item
        widget._listbox.selection_set.assert_called_with(0)
        widget._listbox.activate.assert_called_with(0)
        assert result == "break"

    def test_previous_navigation_first_item(self, setup_widget) -> None:
        _, widget = setup_widget

        # Create listbox mock
        widget._listbox = MagicMock()
        widget._listbox.curselection.return_value = (0,)  # First item is selected
        widget._listbox.size.return_value = 5

        # Test navigating from first item
        result = widget._previous(None)

        # Since we're at the first item, it should select the last one (index = END)
        widget._listbox.selection_clear.assert_called_once_with(0)
        # The function uses index = END which is a tkinter constant, not an integer
        # So we can't directly check for the exact value in the mock call
        assert widget._listbox.selection_set.called
        assert widget._listbox.see.called
        assert widget._listbox.activate.called
        assert result == "break"

    def test_build_listbox(self, setup_widget) -> None:
        root, widget = setup_widget

        # Create values to pass to build_listbox
        values = ["item1", "item2", "item3"]

        # Create a real test frame to contain the listbox
        test_frame = tk.Frame(root)
        with patch("tkinter.ttk.Frame", return_value=test_frame):
            # Call build_listbox
            widget._build_listbox(values)

            # Verify the listbox was created and configured
            assert widget._listbox is not None
            assert widget._listbox.cget("height") == min(widget._listbox_height, len(values))

            # Clean up to avoid test interference
            widget.unpost_listbox()

    def test_unpost_listbox(self, setup_widget) -> None:
        _, widget = setup_widget

        # Create a real test frame and listbox
        test_frame = tk.Frame(widget.master)
        widget._listbox = tk.Listbox(test_frame)
        test_frame.pack()  # Layout the frame

        # Now test unposting
        result = widget.unpost_listbox()

        # Check that listbox is removed
        assert widget._listbox is None
        assert result == ""

        # Clean up
        test_frame.destroy()

    def test_update_entry_from_listbox(self, setup_widget) -> None:
        _, widget = setup_widget

        # Create a real test frame and listbox
        test_frame = tk.Frame(widget.master)
        widget._listbox = tk.Listbox(test_frame)
        test_frame.pack()

        # Add items to listbox
        for item in ["item1", "item2", "item3"]:
            widget._listbox.insert(END, item)

        # Select an item
        widget._listbox.selection_set(1)  # Select "item2"
        widget._listbox.activate(1)

        # Test updating entry from selection
        result = widget.update_entry_from_listbox(None)

        # Verify entry was updated and listbox removed
        assert widget.get_value() == "item2"
        assert widget._listbox is None
        assert result == "break"

        # Clean up
        test_frame.destroy()

    def test_next_previous_navigation(self, setup_widget) -> None:
        _, widget = setup_widget
        # Set up mock listbox
        widget._listbox = MagicMock()
        widget._listbox.curselection.return_value = [1]
        widget._listbox.size.return_value = 3

        # Test _next method
        result = widget._next(None)
        widget._listbox.selection_clear.assert_called_with(1)
        widget._listbox.selection_set.assert_called_with(2)
        assert result == "break"

        # Test _previous method
        widget._listbox.curselection.return_value = [1]
        result = widget._previous(None)
        widget._listbox.selection_clear.assert_called_with(1)
        widget._listbox.selection_set.assert_called_with(first=0)
        assert result == "break"

    def test_default_filter_function_case_sensitive(self, setup_widget) -> None:
        """Test the default filtering with case sensitivity."""
        _, widget = setup_widget
        # Make sure case sensitivity is enabled
        widget._ignorecase_match = False

        # Test with startswith_match=True
        widget._startswith_match = True
        result = widget.default_filter_function("item")
        assert sorted(result) == sorted(["item1", "item2", "item3"])

        # Should not match uppercase when case-sensitive
        result = widget.default_filter_function("ITEM")
        assert result == []

        # Test with startswith_match=False (contains mode)
        widget._startswith_match = False
        result = widget.default_filter_function("tem")
        # In contains mode, "tem" will match items with "tem" anywhere
        # Looking at the actual implementation, it should match "item1", "item2", "item3" and "anotheritem"
        expected = [item for item in widget._list_of_items if "tem" in item]
        assert sorted(result) == sorted(expected)

    def test_default_filter_function_case_insensitive(self, setup_widget) -> None:
        """Test the default filtering with case insensitivity."""
        _, widget = setup_widget
        # Configure for case insensitive matching
        widget._ignorecase_match = True

        # Test with startswith_match=True
        widget._startswith_match = True
        result = widget.default_filter_function("ITEM")
        assert sorted(result) == sorted(["item1", "item2", "item3"])

        # Test with startswith_match=False (contains mode)
        widget._startswith_match = False
        result = widget.default_filter_function("TEM")
        # In case-insensitive contains mode, "TEM" will match items with "tem" anywhere
        expected = [item for item in widget._list_of_items if "tem".lower() in item.lower()]
        assert sorted(result) == sorted(expected)

    def test_get_value_and_set_value(self, setup_widget) -> None:
        """Test get_value and set_value methods."""
        _, widget = setup_widget

        # Test get_value with initial empty value
        assert widget.get_value() == ""

        # Set a value and check
        widget.set_value("test_value")
        assert widget.get_value() == "test_value"

        # Test set_value with close_dialog=True
        widget._listbox = MagicMock()
        widget.unpost_listbox = MagicMock()
        widget.set_value("another_value", close_dialog=True)
        assert widget.get_value() == "another_value"
        widget.unpost_listbox.assert_called_once()

    def test_escape_binding(self, setup_widget) -> None:
        """Test the Escape key binding."""
        _, widget = setup_widget

        # Mock unpost_listbox
        widget.unpost_listbox = MagicMock(return_value="")

        # Get the callback for Escape key
        widget.bind("<Escape>")

        # Create a mock event
        mock_event = MagicMock()

        # Call function directly since we can't easily invoke the binding itself
        # The lambda in the binding forwards to unpost_listbox
        result = widget.unpost_listbox(mock_event)

        assert result == ""
        widget.unpost_listbox.assert_called_once()

    def test_build_listbox_with_scrollbars(self, setup_widget) -> None:
        """Test building listbox with both scrollbars."""
        root, widget = setup_widget

        # Ensure scrollbars are enabled
        widget._use_vscrollbar = True
        widget._use_hscrollbar = True

        # Create values to test
        values = ["item1", "item2", "item3", "very_long_item_that_needs_scrolling"]

        # Create a frame for testing
        test_frame = tk.Frame(root)

        with patch("tkinter.ttk.Frame", return_value=test_frame), patch("tkinter.ttk.Scrollbar") as mock_scrollbar:
            # Set up mock scrollbar
            mock_scrollbar_instance = MagicMock()
            mock_scrollbar.return_value = mock_scrollbar_instance

            # Call build_listbox
            widget._build_listbox(values)

            # Verify scrollbars were created
            assert mock_scrollbar.call_count == 2  # One for vertical, one for horizontal

            # Verify scrollbar configuration
            assert widget._listbox is not None

            # Clean up
            widget.unpost_listbox()

    def test_large_list_handling(self, setup_widget) -> None:
        """Test handling of large lists of items."""
        root, widget = setup_widget

        # Create a large list of items
        large_list = [f"item{i}" for i in range(100)]
        widget._list_of_items = large_list

        # Set up for testing
        test_frame = tk.Frame(root)
        with patch("tkinter.ttk.Frame", return_value=test_frame):
            # Set entry to match several items
            widget._entry_var.set("item")

            # Verify that the height is capped at the configured height
            assert widget._listbox is not None
            height = int(widget._listbox.cget("height"))
            assert height == widget._listbox_height

            # Clean up
            widget.unpost_listbox()

    def test_keyboard_binding_integration(self, setup_widget) -> None:
        """Test that all keyboard bindings are registered."""
        _, widget = setup_widget

        # Check that the key bindings are registered
        assert widget.bind("<Up>")
        assert widget.bind("<Down>")
        assert widget.bind("<Control-n>")
        assert widget.bind("<Control-p>")
        assert widget.bind("<Return>")
        assert widget.bind("<Escape>")

        # When the listbox is created, it should also have bindings
        # Create a test listbox
        test_frame = tk.Frame(widget.master)
        with patch("tkinter.ttk.Frame", return_value=test_frame):
            widget._build_listbox(["item1", "item2"])

            assert widget._listbox.bind("<ButtonRelease-1>")
            assert widget._listbox.bind("<Return>")
            assert widget._listbox.bind("<Escape>")
            assert widget._listbox.bind("<Control-n>")
            assert widget._listbox.bind("<Control-p>")

            # Clean up
            widget.unpost_listbox()

    def test_custom_filter_function(self) -> None:
        """Test with a custom filter function."""
        root = tk.Tk()

        # Define a custom filter function that returns filtered items
        def custom_filter(text) -> list:
            return [item for item in ["apple", "banana", "cherry"] if text in item][::-1]

        widget = EntryWithDynamicalyFilteredListbox(
            root, list_of_items=["item1", "item2"], custom_filter_function=custom_filter
        )

        # Check results directly - for "a" we expect banana and apple (cherry doesn't contain "a")
        # and they should be in reverse order due to the [::-1] in the filter function
        assert widget.filter_function("a") == ["banana", "apple"]

        # Check behavior when typing
        widget._build_listbox = MagicMock()
        widget._entry_var.set("a")

        # Verify that our custom filter function was used
        args, _ = widget._build_listbox.call_args
        assert args[0] == ["banana", "apple"]

        root.destroy()

    def test_empty_string_filter(self, setup_widget) -> None:
        """Test filtering with an empty string."""
        _, widget = setup_widget

        # Empty string should match nothing in the default filter function
        result = widget.default_filter_function("")
        if widget._startswith_match:
            # In startswith mode, empty string matches all items
            assert result == widget._list_of_items
        else:
            # In contains mode, empty string also matches all items
            assert result == widget._list_of_items


# pylint: enable=protected-access


def test_autoscroll() -> None:
    # Test when scrollbar should be hidden
    mock_scrollbar = MagicMock()
    autoscroll(mock_scrollbar, 0, 1)
    mock_scrollbar.grid_remove.assert_called_once()
    mock_scrollbar.set.assert_called_once_with(0.0, 1.0)

    # Test when scrollbar should be shown
    mock_scrollbar.reset_mock()
    autoscroll(mock_scrollbar, 0.1, 0.9)
    mock_scrollbar.grid.assert_called_once()
    mock_scrollbar.set.assert_called_once_with(0.1, 0.9)
