"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.ScheduleTargetBase = void 0;
const jsiiDeprecationWarnings = require("../.warnings.jsii.js");
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const iam = require("aws-cdk-lib/aws-iam");
const helpers_internal_1 = require("aws-cdk-lib/core/lib/helpers-internal");
const util_1 = require("./util");
/**
 * Base class for Schedule Targets
 */
class ScheduleTargetBase {
    constructor(baseProps, targetArn) {
        this.baseProps = baseProps;
        this.targetArn = targetArn;
        try {
            jsiiDeprecationWarnings._aws_cdk_aws_scheduler_targets_alpha_ScheduleTargetBaseProps(baseProps);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, ScheduleTargetBase);
            }
            throw error;
        }
    }
    bindBaseTargetConfig(_schedule) {
        const role = this.baseProps.role ?? this.singletonScheduleRole(_schedule, this.targetArn);
        this.addTargetActionToRole(_schedule, role);
        if (this.baseProps.deadLetterQueue) {
            this.addToDeadLetterQueueResourcePolicy(_schedule, this.baseProps.deadLetterQueue);
        }
        return {
            arn: this.targetArn,
            role: role,
            deadLetterConfig: this.baseProps.deadLetterQueue ? {
                arn: this.baseProps.deadLetterQueue.queueArn,
            } : undefined,
            retryPolicy: this.renderRetryPolicy(this.baseProps.maxEventAge, this.baseProps.retryAttempts),
            input: this.baseProps.input,
        };
    }
    /**
     * Create a return a Schedule Target Configuration for the given schedule
     * @param schedule
     * @returns a Schedule Target Configuration
     */
    bind(schedule) {
        return this.bindBaseTargetConfig(schedule);
    }
    /**
     * Obtain the Role for the EventBridge Scheduler event
     *
     * If a role already exists, it will be returned. This ensures that if multiple
     * events have the same target, they will share a role.
     */
    singletonScheduleRole(schedule, targetArn) {
        const stack = aws_cdk_lib_1.Stack.of(schedule);
        const arn = aws_cdk_lib_1.Token.isUnresolved(targetArn) ? stack.resolve(targetArn).toString() : targetArn;
        const hash = (0, helpers_internal_1.md5hash)(arn).slice(0, 6);
        const id = 'SchedulerRoleForTarget-' + hash;
        const existingRole = stack.node.tryFindChild(id);
        const principal = new iam.PrincipalWithConditions(new iam.ServicePrincipal('scheduler.amazonaws.com'), {
            StringEquals: {
                'aws:SourceAccount': schedule.env.account,
            },
        });
        if (existingRole) {
            existingRole.assumeRolePolicy?.addStatements(new iam.PolicyStatement({
                effect: iam.Effect.ALLOW,
                principals: [principal],
                actions: ['sts:AssumeRole'],
            }));
            return existingRole;
        }
        const role = new iam.Role(stack, id, {
            roleName: aws_cdk_lib_1.PhysicalName.GENERATE_IF_NEEDED,
            assumedBy: principal,
        });
        return role;
    }
    /**
     * Allow a schedule to send events with failed invocation to an Amazon SQS queue.
     * @param schedule schedule to add DLQ to
     * @param queue the DLQ
     */
    addToDeadLetterQueueResourcePolicy(schedule, queue) {
        if (!(0, util_1.sameEnvDimension)(schedule.env.region, queue.env.region)) {
            throw new Error(`Cannot assign Dead Letter Queue in region ${queue.env.region} to the schedule ${aws_cdk_lib_1.Names.nodeUniqueId(schedule.node)} in region ${schedule.env.region}. Both the queue and the schedule must be in the same region.`);
        }
        // Skip Resource Policy creation if the Queue is not in the same account.
        // There is no way to add a target onto an imported schedule, so we can assume we will run the following code only
        // in the account where the schedule is created.
        if ((0, util_1.sameEnvDimension)(schedule.env.account, queue.env.account)) {
            const policyStatementId = `AllowSchedule${aws_cdk_lib_1.Names.nodeUniqueId(schedule.node)}`;
            queue.addToResourcePolicy(new iam.PolicyStatement({
                sid: policyStatementId,
                principals: [new iam.ServicePrincipal('scheduler.amazonaws.com')],
                effect: iam.Effect.ALLOW,
                actions: ['sqs:SendMessage'],
                resources: [queue.queueArn],
            }));
        }
        else {
            aws_cdk_lib_1.Annotations.of(schedule).addWarning(`Cannot add a resource policy to your dead letter queue associated with schedule ${schedule.scheduleName} because the queue is in a different account. You must add the resource policy manually to the dead letter queue in account ${queue.env.account}.`);
        }
    }
    renderRetryPolicy(maximumEventAge, maximumRetryAttempts) {
        const maxMaxAge = aws_cdk_lib_1.Duration.days(1).toSeconds();
        const minMaxAge = aws_cdk_lib_1.Duration.minutes(15).toSeconds();
        let maxAge = maxMaxAge;
        if (maximumEventAge) {
            maxAge = maximumEventAge.toSeconds({ integral: true });
            if (maxAge > maxMaxAge) {
                throw new Error('Maximum event age is 1 day');
            }
            if (maxAge < minMaxAge) {
                throw new Error('Minimum event age is 15 minutes');
            }
        }
        ;
        let maxAttempts = 185;
        if (typeof maximumRetryAttempts != 'undefined') {
            if (maximumRetryAttempts < 0) {
                throw Error('Number of retry attempts should be greater or equal than 0');
            }
            if (maximumRetryAttempts > 185) {
                throw Error('Number of retry attempts should be less or equal than 185');
            }
            maxAttempts = maximumRetryAttempts;
        }
        return {
            maximumEventAgeInSeconds: maxAge,
            maximumRetryAttempts: maxAttempts,
        };
    }
}
exports.ScheduleTargetBase = ScheduleTargetBase;
_a = JSII_RTTI_SYMBOL_1;
ScheduleTargetBase[_a] = { fqn: "@aws-cdk/aws-scheduler-targets-alpha.ScheduleTargetBase", version: "2.113.0-alpha.0" };
//# sourceMappingURL=data:application/json;base64,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