"""
Cache utilities - Generic caching functions that can be used across projects
"""

import hashlib
import json
import logging
import time
import asyncio
from typing import Callable, Any, Optional, Union
from fastapi.encoders import jsonable_encoder
from bson.objectid import ObjectId
from fastapi import Request
import functools
from redis.exceptions import (
    ConnectionError as RedisConnectionError,
    TimeoutError as RedisTimeoutError,
)

# Import core utilities
from ..core.data import normalize
from ..core.serialization import datetime_serializer

logger = logging.getLogger(__name__)


def generate_cache_id(params: dict) -> str:
    """
    Utility to generate a cache key hash from a dictionary.
    Creates a consistent hash based on normalized dictionary content.

    Args:
        params: Dictionary of parameters to hash

    Returns:
        Cache key string
    """
    normalized = normalize(params)
    encoded_data = jsonable_encoder(normalized, custom_encoder={ObjectId: str})
    serialized = json.dumps(encoded_data, separators=(",", ":"), ensure_ascii=True)
    return hashlib.sha256(serialized.encode("utf-8")).hexdigest()


def get_or_set_cache(
    redis_client,
    cache_key: str,
    fetch_func: Callable,
    clear_cache: Union[str, bool] = False,
    ttl: Optional[int] = None,
    serializer: Optional[Callable] = None,
    max_retries: int = 3,
    retry_delay: float = 0.1,
) -> Any:
    """
    Retrieve data from cache if available, otherwise fetch, cache, and return it.
    Optionally clear cache before fetching.

    Args:
        redis_client: Redis client instance
        cache_key: Key to use for caching
        fetch_func: Function to call if cache miss
        clear_cache: Pattern to clear or boolean to clear specific key
        ttl: Time to live in seconds
        serializer: Function to serialize data before caching
        max_retries: Maximum number of retry attempts for Redis operations
        retry_delay: Delay between retry attempts in seconds

    Returns:
        Cached or fetched data
    """
    # Intentar limpiar cache si se solicita
    if clear_cache:
        invalidate_cache_keys(
            redis_client, clear_cache if isinstance(clear_cache, str) else cache_key
        )

    # Intentar obtener del cache
    cached_data = _execute_redis_op(
        lambda: redis_client.get(cache_key) if redis_client.exists(cache_key) else None,
        max_retries=max_retries,
        retry_delay=retry_delay,
    )

    if cached_data is not None:
        try:
            return json.loads(cached_data)
        except (json.JSONDecodeError, TypeError) as e:
            logger.warning(f"Failed to decode cached data for key {cache_key}: {e}")

    # Cache miss o error: obtener datos frescos
    data = fetch_func()
    if serializer:
        data = serializer(data)

    encoded_data = jsonable_encoder(data, custom_encoder={ObjectId: str})

    # Intentar guardar en cache (falla silenciosamente si Redis no está disponible)
    _execute_redis_op(
        lambda: redis_client.set(cache_key, json.dumps(encoded_data), ex=ttl),
        max_retries=max_retries,
        retry_delay=retry_delay,
    )

    return encoded_data


def invalidate_cache_keys(redis_client, pattern: str) -> int:
    """
    Utility to invalidate cache keys matching a pattern.

    Args:
        redis_client: Redis client instance
        pattern: Pattern to match keys for deletion

    Returns:
        Number of keys deleted (0 if Redis is unavailable)
    """
    try:
        cursor = 0
        batch_size = 1000
        total_deleted = 0

        while True:
            cursor, keys = redis_client.scan(
                cursor=cursor, match=pattern, count=batch_size
            )
            if keys:
                redis_client.unlink(*keys)
                total_deleted += len(keys)
            if cursor == 0:
                break

        return total_deleted
    except (RedisConnectionError, RedisTimeoutError) as e:
        logger.warning(f"Redis connection error during cache invalidation: {e}")
        return 0


async def build_cache_key(
    request: Request, api_name: str, collection: str, endpoint_type: str
) -> str:
    """
    Builds a cache key based on the request, api name, collection, and endpoint type.

    Incorporates HTTP method, URL path, path parameters and query parameters to
    avoid collisions between routes that differ only by path params.

    Args:
        request: FastAPI request object
        api_name: Name of the API
        collection: Name of the collection
        endpoint_type: Type of endpoint

    Returns:
        Cache key string
    """
    # Ensure plain dicts for hashing
    query_params_dict = dict(request.query_params) if request.query_params else {}
    path_params_dict = (
        dict(request.path_params) if getattr(request, "path_params", None) else {}
    )

    key_material = {
        "method": request.method,
        "path": request.url.path,
        "path_params": path_params_dict,
        "query": query_params_dict,
    }

    # Include request body for methods where it affects the response identity
    if request.method in {"POST", "PUT"}:
        content_type = (request.headers.get("content-type") or "").split(";")[0].lower()
        body_hash: Optional[str] = None
        if content_type == "application/json":
            try:
                json_body = await request.json()
                normalized_body = normalize(json_body)
                encoded_body = jsonable_encoder(
                    normalized_body, custom_encoder={ObjectId: str}
                )
                key_material["body"] = encoded_body
            except Exception:
                body_bytes = await request.body()
                if body_bytes:
                    body_hash = hashlib.sha256(body_bytes).hexdigest()
        else:
            body_bytes = await request.body()
            if body_bytes:
                body_hash = hashlib.sha256(body_bytes).hexdigest()

        if body_hash:
            key_material["body_hash"] = body_hash

    hashed = generate_cache_id(key_material)
    return f"{api_name}:{collection}:{endpoint_type}:{hashed}"


def _is_async_client(redis_client) -> bool:
    """
    Detecta si el cliente Redis es asíncrono.

    Args:
        redis_client: Cliente Redis a verificar

    Returns:
        True si es asíncrono, False si es síncrono
    """
    return (
        hasattr(redis_client, "__aenter__") or "async" in type(redis_client).__module__
    )


def _execute_redis_op(
    operation: Callable, max_retries: int = 3, retry_delay: float = 0.1
) -> Any:
    """
    Ejecuta operación Redis síncrona con reintentos y degradación elegante.

    Args:
        operation: Función que ejecuta la operación Redis
        max_retries: Número máximo de reintentos
        retry_delay: Tiempo de espera entre reintentos en segundos

    Returns:
        Resultado de la operación o None si falla después de todos los reintentos
    """
    for attempt in range(max_retries):
        try:
            return operation()
        except (RedisConnectionError, RedisTimeoutError) as e:
            if attempt == max_retries - 1:
                logger.warning(f"Redis unavailable after {max_retries} attempts: {e}")
                return None
            time.sleep(retry_delay)
    return None


async def _execute_redis_op_async(
    operation: Callable, max_retries: int = 3, retry_delay: float = 0.1
) -> Any:
    """
    Ejecuta operación Redis asíncrona con reintentos y degradación elegante.

    Args:
        operation: Función async que ejecuta la operación Redis
        max_retries: Número máximo de reintentos
        retry_delay: Tiempo de espera entre reintentos en segundos

    Returns:
        Resultado de la operación o None si falla después de todos los reintentos
    """
    for attempt in range(max_retries):
        try:
            return await operation()
        except (RedisConnectionError, RedisTimeoutError) as e:
            if attempt == max_retries - 1:
                logger.warning(f"Redis unavailable after {max_retries} attempts: {e}")
                return None
            await asyncio.sleep(retry_delay)
    return None


def cache_response(
    redis_client,
    api_name: str,
    collection: str,
    endpoint_type: str,
    tags: list[str],
    ttl: int | None = None,
    use_cache: bool = True,
    max_retries: int = 3,
):
    """
    Decorator to cache the response of an endpoint.

    Args:
        redis_client: Redis client instance (sync or async)
        api_name: Name of the API
        collection: Name of the collection
        endpoint_type: Type of endpoint
        tags: List of tags to use for caching
        ttl: Time to live in seconds
        use_cache: Whether to use cache
        max_retries: Maximum number of retry attempts for Redis operations

    Returns:
        Decorator function
    """
    is_async = _is_async_client(redis_client)

    def decorator(func: Callable):
        @functools.wraps(func)
        async def wrapper(*args, **kwargs):
            if not use_cache:
                return await func(*args, **kwargs)

            # Detectar el Request
            request: Optional[Request] = kwargs.get("request")
            if request is None:
                for arg in args:
                    if isinstance(arg, Request):
                        request = arg
                        break
            if request is None:
                raise ValueError("Request object not found in endpoint parameters.")

            query_params = request.query_params
            clear_cache_value = query_params.get("clear_cache")
            if isinstance(clear_cache_value, str):
                clear_cache_flag = clear_cache_value.lower() in {
                    "1",
                    "true",
                    "yes",
                    "y",
                }
            else:
                clear_cache_flag = bool(clear_cache_value)

            # Intentar limpiar cache si se solicita
            if clear_cache_flag:
                try:
                    invalidate_tag(redis_client, f"{collection}")
                except (RedisConnectionError, RedisTimeoutError) as e:
                    logger.warning(f"Redis connection error during cache clear: {e}")

            # Construir clave
            cache_key = await build_cache_key(
                request, api_name, collection, endpoint_type
            )

            # Intentar obtener del cache
            cached_data = None
            if is_async:

                async def get_cached_data():
                    exists_result = await redis_client.exists(cache_key)
                    if exists_result:
                        return await redis_client.get(cache_key)
                    return None

                cached_data = await _execute_redis_op_async(
                    get_cached_data, max_retries=max_retries
                )
            else:
                cached_data = _execute_redis_op(
                    lambda: (
                        redis_client.get(cache_key)
                        if redis_client.exists(cache_key)
                        else None
                    ),
                    max_retries=max_retries,
                )

            if cached_data is not None:
                try:
                    return json.loads(cached_data)
                except (json.JSONDecodeError, TypeError) as e:
                    logger.warning(
                        f"Failed to decode cached data for key {cache_key}: {e}"
                    )

            # Cache miss o error: ejecutar función y cachear respuesta
            response = await func(*args, **kwargs)
            encoded_data = jsonable_encoder(response, custom_encoder={ObjectId: str})

            # Intentar guardar en cache (falla silenciosamente si Redis no está disponible)
            if is_async:

                async def set_cache_data():
                    await redis_client.set(cache_key, json.dumps(encoded_data), ex=ttl)

                await _execute_redis_op_async(set_cache_data, max_retries=max_retries)
                # Agregar tags
                for tag in tags:
                    # Capturar el valor de tag en la closure usando parámetro por defecto
                    async def add_tag(tag_name=tag):
                        await redis_client.sadd(f"tag:{tag_name}", cache_key)

                    await _execute_redis_op_async(add_tag, max_retries=max_retries)
            else:
                _execute_redis_op(
                    lambda: redis_client.set(
                        cache_key, json.dumps(encoded_data), ex=ttl
                    ),
                    max_retries=max_retries,
                )
                # Agregar tags
                for tag in tags:
                    _execute_redis_op(
                        lambda t=tag: redis_client.sadd(f"tag:{t}", cache_key),
                        max_retries=max_retries,
                    )

            return encoded_data

        return wrapper

    return decorator


def cache_factory(redis_client, api_name: str, collection: str):
    """
    Factory to create cache decorators for different endpoint types.

    Args:
        redis_client: Redis client instance
        api_name: Name of the API
        collection: Name of the collection

    Returns:
        Decorator function
    """

    def wrapper(
        endpoint_type: str,
        tags: list[str],
        ttl: int | None = None,
        use_cache: bool = True,
    ):
        return cache_response(
            redis_client=redis_client,
            api_name=api_name,
            collection=collection,
            endpoint_type=endpoint_type,
            tags=tags,
            ttl=ttl,
            use_cache=use_cache,
        )

    return wrapper


def invalidate_tag(redis_client, tag: str):
    """
    Utility to invalidate cache keys associated with a tag.

    Args:
        redis_client: Redis client instance
        tag: Tag to invalidate
    """
    try:
        keys = redis_client.smembers(f"tag:{tag}")
        for key in keys:
            redis_client.delete(key)
        redis_client.delete(f"tag:{tag}")
    except (RedisConnectionError, RedisTimeoutError) as e:
        logger.warning(f"Redis connection error during tag invalidation: {e}")
