"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.POLICY_ID_ANNOTATION = exports.POLICY_DESCRIPTION_ANNOTATION = void 0;
exports.checkParseSchema = checkParseSchema;
exports.checkParsePolicy = checkParsePolicy;
exports.getPolicyDescription = getPolicyDescription;
exports.getPolicyId = getPolicyId;
exports.splitPolicies = splitPolicies;
exports.checkParseTemplate = checkParseTemplate;
exports.validateMultiplePolicies = validateMultiplePolicies;
exports.cleanUpApiNameForNamespace = cleanUpApiNameForNamespace;
exports.buildSchema = buildSchema;
const cedar = require("@cedar-policy/cedar-wasm/nodejs");
exports.POLICY_DESCRIPTION_ANNOTATION = '@cdkDescription';
exports.POLICY_ID_ANNOTATION = '@cdkId';
function checkParseSchema(schemaStr) {
    const schemaParseResult = cedar.checkParseSchema(JSON.parse(schemaStr));
    if (schemaParseResult.type === 'failure') {
        throw new Error(`Schema is invalid:  ${schemaParseResult.errors.map((error) => error.message).join('; ')}`);
    }
}
function checkParsePolicy(policyStatement) {
    const parsePolicyResult = cedar.checkParsePolicySet({ staticPolicies: policyStatement });
    if (parsePolicyResult.type == 'failure') {
        throw new Error(`Invalid policy statement: ${policyStatement}. Errors: ${parsePolicyResult.errors.map((error) => error.message).join('; ')}`);
    }
}
/**
 * Extracts the Description of the Policy searching for the @see POLICY_DESCRIPTION_ANNOTATION annotation on top of policy contents (before effect)
 * @param policyStatement The policy statement in string format
 * @returns Returns the description if found or null
 */
function getPolicyDescription(policyStatement) {
    const regex = new RegExp(String.raw `${exports.POLICY_DESCRIPTION_ANNOTATION}\(["']([^"']*)["']\)`);
    let matches = policyStatement.match(regex);
    return (matches) ? matches[1] : null;
}
/**
 * Extracts the Description of the Policy searching for the @see POLICY_ID_ANNOTATION annotation on top of policy contents (before effect)
 * @param policyStatement The policy statement in string format
 * @returns Returns the id if found or null
 */
function getPolicyId(policyStatement) {
    const regex = new RegExp(String.raw `${exports.POLICY_ID_ANNOTATION}\(["']([^"']*)["']\)`);
    let matches = policyStatement.match(regex);
    return (matches) ? matches[1] : null;
}
/**
 * Split the Policies in case of multiple Cedar Policies in a string.
 * Please Note: this method doesn't provide a validation of every policy, it just splits
 * them and returns an array.
 * @param policyStatements The policy statements in string format
 * @returns a string array where each element represents a Cedar Policy statement
 */
function splitPolicies(policyStatements) {
    const policySetTextToPartsResult = cedar.policySetTextToParts(policyStatements);
    if (policySetTextToPartsResult.type === 'failure') {
        throw new Error(`Error splitting policies: ${policySetTextToPartsResult.errors.map((error) => error.message).join('; ')}`);
    }
    return policySetTextToPartsResult.policies;
}
function checkParseTemplate(templateStatement) {
    const templateParseResult = cedar.checkParsePolicySet({ templates: { template: templateStatement } });
    if (templateParseResult.type === 'failure') {
        throw new Error(`Error parsing template: ${templateStatement}. Errors: ${templateParseResult.errors.map((error) => error.message).join('; ')}`);
    }
}
/**
 * Method which validates multple Cedar policies in a single string against a Cedar Schema
 *
 * @param policiesStatements a string containing multiple Cedar policies
 * @param schemaStr a Cedar Schema
 */
function validateMultiplePolicies(policiesStatements, schemaStr) {
    const validationResult = cedar.validate({
        schema: JSON.parse(schemaStr),
        policies: { staticPolicies: policiesStatements },
    });
    if (validationResult.type === 'failure') {
        throw new Error(`Policies could not be parsed: ${validationResult.errors.map((error) => error.message).join('; ')}`);
    }
    if (validationResult.type === 'success' && validationResult.validationErrors.length > 0) {
        throw new Error(`Policies could not be validated against the schema:
      ${validationResult.validationErrors.map((error) => error.policyId + ': ' + error.error).join('; ')}`);
    }
}
function cleanUpApiNameForNamespace(apiName) {
    const validCedarName = apiName.replace(/[^a-zA-Z0-9_]/g, '').trim();
    if (validCedarName.length === 0) {
        return 'ImportedApi';
    }
    if (/[0-9_]/.exec(validCedarName[0])) {
        return `Api${validCedarName}`;
    }
    return validCedarName;
}
function buildSchema(namespace, actionNames, principalGroupType) {
    const additionalEntities = {};
    if (principalGroupType) {
        additionalEntities[principalGroupType] = {
            shape: {
                type: 'Record',
                attributes: {},
            },
        };
    }
    const actions = actionNames.reduce((acc, actionName) => {
        return {
            ...acc,
            [actionName]: {
                appliesTo: {
                    context: { type: 'Record', attributes: {} },
                    principalTypes: ['User'],
                    resourceTypes: ['Application'],
                },
            },
        };
    }, {});
    return {
        [namespace]: {
            entityTypes: {
                ...additionalEntities,
                User: {
                    shape: {
                        type: 'Record',
                        attributes: {},
                    },
                    memberOfTypes: principalGroupType ? [principalGroupType] : [],
                },
                Application: {
                    shape: {
                        type: 'Record',
                        attributes: {},
                    },
                },
            },
            actions,
        },
    };
}
//# sourceMappingURL=data:application/json;base64,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