"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.IdentitySource = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const aws_verifiedpermissions_1 = require("aws-cdk-lib/aws-verifiedpermissions");
const core_1 = require("aws-cdk-lib/core");
var ConfigurationMode;
(function (ConfigurationMode) {
    ConfigurationMode["COGNITO"] = "COGNITO";
    ConfigurationMode["OIDC_ACCESS_TOKEN"] = "OIDC_ACCESS_TOKEN";
    ConfigurationMode["OIDC_ID_TOKEN"] = "OIDC_ID_TOKEN";
})(ConfigurationMode || (ConfigurationMode = {}));
class IdentitySourceBase extends core_1.Resource {
}
class IdentitySource extends IdentitySourceBase {
    /**
     * Creates Identity Source from its attributes
     *
     * @param scope The parent creating construct (usually `this`).
     * @param id The construct's name.
     * @param attrs An `IdentitySourceAttributes` object.
     */
    static fromIdentitySourceAttributes(scope, id, attrs) {
        class Import extends IdentitySourceBase {
            constructor(identitySourceId) {
                super(scope, id);
                this.identitySourceId = identitySourceId;
            }
        }
        let identitySourceId;
        identitySourceId = attrs.identitySourceId;
        return new Import(identitySourceId);
    }
    /**
     * Create an Identity Source from its identifier
     *
     * @param scope The parent creating construct (usually `this`).
     * @param id The construct's name.
     * @param identitySourceId The Identity Source identifier.
     */
    static fromIdentitySourceId(scope, id, identitySourceId) {
        return IdentitySource.fromIdentitySourceAttributes(scope, id, {
            identitySourceId,
        });
    }
    constructor(scope, id, props) {
        super(scope, id);
        if (props.configuration.cognitoUserPoolConfiguration && props.configuration.openIdConnectConfiguration) {
            throw new Error('Only one between cognitoUserPoolConfiguration or openIdConnectConfiguration must be defined');
        }
        let cfnConfiguration;
        let issuer;
        if (props.configuration.cognitoUserPoolConfiguration) {
            this.clientIds = props.configuration.cognitoUserPoolConfiguration.clientIds ?? [];
            this.audiencesOIDC = [];
            const cognitoGroupConfiguration = props.configuration.cognitoUserPoolConfiguration.groupConfiguration?.groupEntityType
                ? {
                    groupEntityType: props.configuration.cognitoUserPoolConfiguration.groupConfiguration.groupEntityType,
                }
                : undefined;
            cfnConfiguration = {
                cognitoUserPoolConfiguration: {
                    clientIds: core_1.Lazy.list({ produce: () => this.clientIds }),
                    userPoolArn: props.configuration.cognitoUserPoolConfiguration.userPool.userPoolArn,
                    groupConfiguration: cognitoGroupConfiguration,
                },
            };
            this.cognitoGroupEntityType = cognitoGroupConfiguration?.groupEntityType;
            issuer = 'COGNITO';
            this.configurationMode = ConfigurationMode.COGNITO;
        }
        else if (props.configuration.openIdConnectConfiguration) {
            if (props.configuration.openIdConnectConfiguration.accessTokenOnly &&
                props.configuration.openIdConnectConfiguration.identityTokenOnly) {
                throw new Error('Exactly one token selection method between accessTokenOnly and identityTokenOnly must be defined');
            }
            let tokenSelection;
            if (props.configuration.openIdConnectConfiguration.accessTokenOnly) {
                if (!props.configuration.openIdConnectConfiguration.accessTokenOnly.audiences ||
                    props.configuration.openIdConnectConfiguration.accessTokenOnly.audiences.length == 0) {
                    throw new Error('At least one audience is expected in OIDC Access token selection mode');
                }
                this.clientIds = [];
                this.audiencesOIDC = props.configuration.openIdConnectConfiguration.accessTokenOnly.audiences;
                tokenSelection = {
                    accessTokenOnly: {
                        audiences: core_1.Lazy.list({ produce: () => this.audiencesOIDC }),
                        principalIdClaim: props.configuration.openIdConnectConfiguration.accessTokenOnly.principalIdClaim,
                    },
                };
                this.principalIdClaimOIDC = props.configuration.openIdConnectConfiguration.accessTokenOnly.principalIdClaim;
                this.configurationMode = ConfigurationMode.OIDC_ACCESS_TOKEN;
            }
            else if (props.configuration.openIdConnectConfiguration.identityTokenOnly) {
                this.clientIds = props.configuration.openIdConnectConfiguration.identityTokenOnly.clientIds ?? [];
                this.audiencesOIDC = [];
                tokenSelection = {
                    identityTokenOnly: {
                        clientIds: core_1.Lazy.list({ produce: () => this.clientIds }),
                        principalIdClaim: props.configuration.openIdConnectConfiguration.identityTokenOnly.principalIdClaim,
                    },
                };
                this.principalIdClaimOIDC = props.configuration.openIdConnectConfiguration.identityTokenOnly.principalIdClaim;
                this.configurationMode = ConfigurationMode.OIDC_ID_TOKEN;
            }
            else {
                throw new Error('One token selection method between accessTokenOnly and identityTokenOnly must be defined');
            }
            cfnConfiguration = {
                openIdConnectConfiguration: {
                    issuer: props.configuration.openIdConnectConfiguration.issuer,
                    entityIdPrefix: props.configuration.openIdConnectConfiguration.entityIdPrefix,
                    groupConfiguration: props.configuration.openIdConnectConfiguration.groupConfiguration ? {
                        groupClaim: props.configuration.openIdConnectConfiguration.groupConfiguration.groupClaim,
                        groupEntityType: props.configuration.openIdConnectConfiguration.groupConfiguration.groupEntityType,
                    } : undefined,
                    tokenSelection: tokenSelection,
                },
            };
            this.groupConfigGroupClaimOIDC = props.configuration.openIdConnectConfiguration.groupConfiguration?.groupClaim;
            this.groupConfigGroupEntityTypeOIDC = props.configuration.openIdConnectConfiguration.groupConfiguration?.groupEntityType;
            issuer = props.configuration.openIdConnectConfiguration.issuer;
        }
        else {
            throw new Error('One Identity provider configuration between cognitoUserPoolConfiguration and openIdConnectConfiguration must be defined');
        }
        this.identitySource = new aws_verifiedpermissions_1.CfnIdentitySource(this, id, {
            configuration: cfnConfiguration,
            policyStoreId: props.policyStore.policyStoreId,
            principalEntityType: props.principalEntityType,
        });
        this.userPoolArn = props.configuration.cognitoUserPoolConfiguration?.userPool.userPoolArn || undefined;
        this.identitySourceId = this.identitySource.attrIdentitySourceId;
        this.issuer = issuer;
        this.policyStore = props.policyStore;
    }
    /**
     * Add a User Pool Client
     * The method can be called only when the Identity Source is configured with Cognito auth provider
     *
     * @param userPoolClient The User Pool Client Construct.
     */
    addUserPoolClient(userPoolClient) {
        if (this.configurationMode != ConfigurationMode.COGNITO) {
            throw new Error('Cannot add User Pool Client when IdentitySource auth provider is not Cognito');
        }
        this.addClientId(userPoolClient.userPoolClientId);
    }
    /**
     * Add a clientId to the list
     * The method can be called only when the Identity Source is configured with one of these configs:
     *  - Cognito auth provider
     *  - OIDC auth provider and ID Token Selection mode
     *
     * @param clientId The clientId to be added
     */
    addClientId(clientId) {
        if (this.configurationMode != ConfigurationMode.COGNITO && this.configurationMode != ConfigurationMode.OIDC_ID_TOKEN) {
            throw new Error('Adding a Client ID is only supported for the auth providers Cognito or OIDC with configured with ID Token');
        }
        this.clientIds.push(clientId);
    }
    /**
     * Add an audience to the list.
     * The method can be called only when the Identity Source is configured with OIDC auth provider and Access Token Selection mode
     *
     * @param audience the audience to be added
     */
    addAudience(audience) {
        if (this.configurationMode != ConfigurationMode.OIDC_ACCESS_TOKEN) {
            throw new Error('Cannot add audience when IdentitySource auth provider is not OIDC with Access Token');
        }
        this.audiencesOIDC.push(audience);
    }
}
exports.IdentitySource = IdentitySource;
_a = JSII_RTTI_SYMBOL_1;
IdentitySource[_a] = { fqn: "@cdklabs/cdk-verified-permissions.IdentitySource", version: "0.3.0" };
//# sourceMappingURL=data:application/json;base64,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