"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.PolicyType = exports.Policy = exports.POLICY_DESC_SUFFIX_FROM_FILE = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const fs = require("fs");
const path = require("path");
const aws_verifiedpermissions_1 = require("aws-cdk-lib/aws-verifiedpermissions");
const core_1 = require("aws-cdk-lib/core");
const cedar_helpers_1 = require("./cedar-helpers");
exports.POLICY_DESC_SUFFIX_FROM_FILE = ' - Created by CDK';
class PolicyBase extends core_1.Resource {
}
class Policy extends PolicyBase {
    /**
     *  Import a policy into the CDK using its id.
     *
     * @param scope The parent creating construct (usually `this`).
     * @param id The construct id.
     * @param policyId The policy id.
     */
    static fromPolicyId(scope, id, policyId) {
        return Policy.fromPolicyAttributes(scope, id, { policyId });
    }
    /**
     * Import a Policy construct from attributes.
     *
     * @param scope The parent creating construct (usually `this`).
     * @param id The construct id.
     * @param attrs A `PolicyAttributes` object.
     * @returns
     */
    static fromPolicyAttributes(scope, id, attrs) {
        class Import extends PolicyBase {
            constructor(policyType, policyId) {
                super(scope, id);
                this.policyType = policyType;
                this.policyId = policyId;
            }
        }
        const policyType = attrs.policyType ?? PolicyType.STATIC;
        return new Import(policyType, attrs.policyId);
    }
    /**
     * Create a policy based on a file containing a cedar policy. Best practice would be
     * for the file name to end in `.cedar` but this is not required. Policy is parsed for valid
     * syntax but not validated against schema. In order to validate against schema, use
     * `PolicyStore.addPoliciesFromPath()`
     *
     * @param scope The parent creating construct (usually `this`).
     * @param props A `StaticPolicyFromFileProps` object.
     * @param defaultPolicyId The Policy construct default id. This may be directly passed to the method or defined inside the file.
     *           When you have multiple policies per file it's strongly suggested to define the id directly
     *           inside the file in order to avoid multiple policy constructs with the same id. In case of id passed
     *           directly to the method and also defined in file, the latter will take priority.
     */
    static fromFile(scope, defaultPolicyId, props) {
        const policyFileContents = fs.readFileSync(props.path).toString();
        let relativePath = path.basename(props.path);
        let enablePolicyValidation = (props.enablePolicyValidation == undefined) ? true : props.enablePolicyValidation;
        const policies = (0, cedar_helpers_1.splitPolicies)(policyFileContents);
        const policyDefinitions = policies.map(policyContents => {
            let policyId = (0, cedar_helpers_1.getPolicyId)(policyContents) || defaultPolicyId;
            let policyDescription = (0, cedar_helpers_1.getPolicyDescription)(policyContents) || props.description || `${relativePath}${exports.POLICY_DESC_SUFFIX_FROM_FILE}`;
            return new Policy(scope, policyId, {
                definition: {
                    static: {
                        statement: policyContents,
                        description: policyDescription,
                        enablePolicyValidation: enablePolicyValidation,
                    },
                },
                policyStore: props.policyStore,
            });
        });
        return policyDefinitions;
    }
    constructor(scope, id, props) {
        super(scope, id);
        // validations
        if (props.definition.static && props.definition.templateLinked) {
            throw new Error('Policy can either be static or templateLinked');
        }
        let cfnDefinitionAttr;
        let definitionProperty = props.definition;
        if (props.definition.static) {
            if (props.definition.static.enablePolicyValidation ?? true) {
                (0, cedar_helpers_1.checkParsePolicy)(props.definition.static.statement);
            }
            let description = props.definition.static.description || (0, cedar_helpers_1.getPolicyDescription)(props.definition.static.statement) || undefined;
            definitionProperty = {
                static: {
                    ...props.definition.static,
                    ...{
                        description,
                    },
                },
            };
            cfnDefinitionAttr = {
                static: {
                    description: description,
                    statement: props.definition.static.statement,
                },
            };
        }
        else if (props.definition.templateLinked) {
            cfnDefinitionAttr = {
                templateLinked: {
                    policyTemplateId: props.definition.templateLinked.policyTemplate.policyTemplateId,
                    principal: props.definition.templateLinked.principal,
                    resource: props.definition.templateLinked.resource,
                },
            };
        }
        else {
            throw new Error('Policy must either be static or templateLinked');
        }
        // resource
        this.policy = new aws_verifiedpermissions_1.CfnPolicy(this, id, {
            definition: cfnDefinitionAttr,
            policyStoreId: props.policyStore.policyStoreId,
        });
        // assign construct props
        this.policyId = this.policy.attrPolicyId;
        this.policyType = definitionProperty.static
            ? PolicyType.STATIC
            : PolicyType.TEMPLATELINKED;
        this.definition = definitionProperty;
        this.policyStoreId = props.policyStore.policyStoreId;
    }
}
exports.Policy = Policy;
_a = JSII_RTTI_SYMBOL_1;
Policy[_a] = { fqn: "@cdklabs/cdk-verified-permissions.Policy", version: "0.3.0" };
/**
 * PolicyType options
 */
var PolicyType;
(function (PolicyType) {
    PolicyType["STATIC"] = "Static";
    PolicyType["TEMPLATELINKED"] = "TemplateLinked";
})(PolicyType || (exports.PolicyType = PolicyType = {}));
//# sourceMappingURL=data:application/json;base64,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