#!/usr/bin/env python
# -*- coding: utf-8 -*-
# ---------------------------------------------------------------------
# Copyright (c) Merchise Autrement [~º/~] and Contributors
# All rights reserved.
#
# This is free software; you can do what the LICENCE file allows you to.
#
"""Basic non-Odoo implementations of the types.

"""
import immutables
from datetime import datetime, timedelta
from dataclasses import dataclass, field
from typing import (
    Iterable,
    Iterator,
    Optional,
    Sequence,
    TypeVar,
    Tuple,
    Union,
    TYPE_CHECKING,
)

from xotl.tools.symbols import Unset, boolean as UnsetType

from .i18n import _

from .types import (
    Environment,
    Demand as BaseDemand,
    Commodity as BaseCommodity,
    Request as BaseRequest,
    Procedure,
    Result,
    PriceResultType,
)


S = TypeVar("S")


def _replace(self: S, **kwargs) -> S:
    from copy import copy

    # Ensure all values are hashable.  This will make the culprit more clear
    # in tracebacks.  We're getting some demands being passed a **list** of
    # requests.  But the error happens long after the demand was created.
    #
    # See #188: https://gitlab.merchise.org/mercurio-2018/xhg2/issues/188
    # See also MERCURIO-2019-BM: https://sentry.merchise.org/share/issue/47402a397b2b4e198a417102e818028a/
    for value in kwargs.values():
        hash(value)

    result = copy(self)
    result.__dict__.update(kwargs)
    return result


# NB: Don't put slots in Commodity, Request or Demand.  We use the __dict__ in
# the replace method.


@dataclass(unsafe_hash=True)
class Commodity(BaseCommodity):
    start_date: datetime
    duration: timedelta

    replace = _replace

    @property
    def end_date(self):
        return self.start_date + self.duration


@dataclass(unsafe_hash=True)
class Request(BaseRequest[Commodity]):
    replace = _replace

    commodity: Commodity
    quantity: int

    @classmethod
    def new(cls) -> "Request":
        """Return a new unitary request with a commodity starting now and during a
        day.

        """
        return cls(Commodity(datetime.utcnow(), timedelta(1)), 1)


@dataclass(unsafe_hash=True)
class Demand(BaseDemand[Commodity]):
    date: datetime
    requests: Sequence[Request]

    replace = _replace

    def get_commodities(self) -> Iterable[Commodity]:
        return tuple(r.commodity for r in self.requests)

    @classmethod
    def from_commodities(cls, commodites: Sequence[Commodity], date=None) -> "Demand":
        return cls.from_requests((Request(c, 1) for c in commodites), date=date)

    @classmethod
    def from_requests(cls, requests: Iterable[Request], date=None) -> "Demand":
        return cls(date=date or datetime.utcnow(), requests=tuple(requests))


@dataclass(init=False, unsafe_hash=True)
class PriceResult:
    """The result of price computation.

    The procedure can provide a `title` for the `result`, and also
    sub-results to provide insight about how the price was computed.

    """

    title: str
    procedure: Optional[Procedure]
    result: Result
    # A price result has several 'children' result which contain information
    # about the method used to compute `result` and `title`.
    #
    # There's no a pre-established requirement between the `result` of the
    # children and the parent's result.
    #
    # It's up to the program to create the information.
    subresults: Tuple[PriceResultType[Commodity], ...] = field(compare=False)

    # The same procedure can be used iteratively to compute sub-demands.  Also
    # the environment can change from one call to the other.  The only way to
    # distinguish price results resulting from the same procedure is to know
    # both the demand priced and the environment.
    demand: BaseDemand
    env: Environment

    def __init__(
        self,
        title: str,
        procedure: Optional["Procedure"],
        result: Result,
        demand: BaseDemand,
        env: Environment,
        *subresults: PriceResultType[Commodity],
    ) -> None:
        self.title = title
        self.procedure = procedure
        self.result = result
        self.demand = demand
        self.env = env
        self.subresults = tuple(subresults)

    replace = _replace

    def __str__(self):
        import textwrap

        subresults = [textwrap.indent(str(r), " - ") for r in self.subresults]
        if isinstance(self.result, (int, float)):
            result = f"{self.title!s} = {self.result:.2f}"
        else:
            result = f"{self.title!s} = {self.result!s}"
        if not subresults:
            return f"{result}."
        else:
            return "\n".join([f"{result}:"] + subresults)

    def to_html(self):
        subresults = [r.to_html() for r in self.subresults]
        if isinstance(self.result, (int, float)):
            result = f"{self.title!s} = {self.result:.2f}"
        else:
            result = f"{self.title!s} = {self.result!s}"
        if not subresults:
            return f"<p>{result}</p>"
        else:
            return "\n".join(
                [f"<details><summary>{result}:</summary>"] + subresults + ["</details>"]
            )


@dataclass
class LazyCall:
    """A lazy call to a procedure.

    """

    proc: Procedure
    demand: BaseDemand
    env: Environment

    def __call__(self) -> PriceResultType:
        return self.proc(self.demand, self.env)


@dataclass(init=False)
class LazyResults(Sequence[PriceResultType]):
    """Represents many results which are computed on demand.

    I assume that `partials` are memoized so that calling twice only performs
    the first computation.

    """

    partials: Sequence[LazyCall]

    def __init__(self, partials: Iterable[LazyCall]) -> None:
        self.partials = partials = tuple(partials)
        self.results = [Unset] * len(partials)

    @property
    def _pairs(self) -> Iterable[Tuple[Union[Result, UnsetType], LazyCall]]:
        return zip(self.results, self.partials)

    def __len__(self):
        return len(self.results)

    def __getitem__(self, index):
        result = self.results[index]
        if result is Unset:
            partial = self.partials[index]
            self.results[index] = result = partial()
        return result

    def __iter__(self) -> Iterator[PriceResultType]:
        for index, (result, partial) in enumerate(zip(self.results, self.partials)):
            if result is Unset:
                self.results[index] = result = partial()
            yield result


class LazyPriceResult(PriceResult):
    def __init__(
        self,
        title: str,
        procedure: Optional[Procedure],
        result: Result,
        demand: BaseDemand,
        env: Environment,
        subresults: Sequence[PriceResultType],
    ) -> None:
        super().__init__(title, procedure, result, demand, env)
        self.subresults = subresults  # type: ignore

    def __str__(self):
        import textwrap

        def _str(r, partial):
            if r is Unset:
                name = str(partial.proc)
                title = getattr(
                    partial.proc, "title", _("Procedure {name}").format(name=name)
                )
                return _("{title} was optimized away").format(title=title)
            else:
                return str(r)

        subresults = [
            textwrap.indent(_str(r, partial), " - ")
            for r, partial in self.subresults._pairs
        ]
        result = f"{self.title} = {self.result}"
        if not subresults:
            return f"{result}."
        else:
            return "\n".join([f"{result}:"] + subresults)

    def to_html(self):
        def _to_html(r, partial):
            if r is Unset:
                name = str(partial.proc)
                title = getattr(
                    partial.proc, "title", _("Procedure {name}").format(name=name)
                )
                return _("<p>{title} was optimized away<p>").format(title=title)
            else:
                return r.to_html()

        subresults = [_to_html(r, partial) for r, partial in self.subresults._pairs]
        result = f"{self.title} = {self.result}"
        if not subresults:
            return f"<p>{result}</p>"
        else:
            return "\n".join(
                [f"<details><summary>{result}:</summary>"] + subresults + ["</details>"]
            )


del _replace


NULL_DEMAND: Demand = Demand.from_requests(())
EMPTY_ENV: Environment = immutables.Map()


if TYPE_CHECKING:

    def check_result(r: PriceResultType):
        pass

    check_result(
        PriceResult(title="", procedure=None, result=0, demand=NULL_DEMAND, env=EMPTY_ENV)
    )
