"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.ServerDeploymentConfig = exports.MinimumHealthyHosts = void 0;
const cdk = require("@aws-cdk/core");
const codedeploy_generated_1 = require("../codedeploy.generated");
const utils_1 = require("../utils");
/**
 * Minimum number of healthy hosts for a server deployment.
 */
class MinimumHealthyHosts {
    constructor(json) {
        this.json = json;
    }
    /**
     * The minimum healhty hosts threshold expressed as an absolute number.
     */
    static count(value) {
        return new MinimumHealthyHosts({
            type: 'HOST_COUNT',
            value,
        });
    }
    /**
     * The minmum healhty hosts threshold expressed as a percentage of the fleet.
     */
    static percentage(value) {
        return new MinimumHealthyHosts({
            type: 'FLEET_PERCENT',
            value,
        });
    }
    /**
     * @internal
     */
    get _json() {
        return this.json;
    }
}
exports.MinimumHealthyHosts = MinimumHealthyHosts;
/**
 * A custom Deployment Configuration for an EC2/on-premise Deployment Group.
 *
 * @resource AWS::CodeDeploy::DeploymentConfig
 */
class ServerDeploymentConfig extends cdk.Resource {
    /**
     *
     */
    constructor(scope, id, props) {
        super(scope, id, {
            physicalName: props.deploymentConfigName,
        });
        const resource = new codedeploy_generated_1.CfnDeploymentConfig(this, 'Resource', {
            deploymentConfigName: this.physicalName,
            minimumHealthyHosts: props.minimumHealthyHosts._json,
        });
        this.deploymentConfigName = resource.ref;
        this.deploymentConfigArn = utils_1.arnForDeploymentConfig(this.deploymentConfigName);
    }
    /**
     * Import a custom Deployment Configuration for an EC2/on-premise Deployment Group defined either outside the CDK app, or in a different region.
     *
     * @param scope the parent Construct for this new Construct.
     * @param id the logical ID of this new Construct.
     * @param serverDeploymentConfigName the properties of the referenced custom Deployment Configuration.
     * @returns a Construct representing a reference to an existing custom Deployment Configuration
     */
    static fromServerDeploymentConfigName(scope, id, serverDeploymentConfigName) {
        ignore(scope);
        ignore(id);
        return deploymentConfig(serverDeploymentConfigName);
    }
}
exports.ServerDeploymentConfig = ServerDeploymentConfig;
/**
 *
 */
ServerDeploymentConfig.ONE_AT_A_TIME = deploymentConfig('CodeDeployDefault.OneAtATime');
/**
 *
 */
ServerDeploymentConfig.HALF_AT_A_TIME = deploymentConfig('CodeDeployDefault.HalfAtATime');
/**
 *
 */
ServerDeploymentConfig.ALL_AT_ONCE = deploymentConfig('CodeDeployDefault.AllAtOnce');
function deploymentConfig(name) {
    return {
        deploymentConfigName: name,
        deploymentConfigArn: utils_1.arnForDeploymentConfig(name),
    };
}
function ignore(_x) { return; }
//# sourceMappingURL=data:application/json;base64,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