#!/usr/bin/env python3
"""
DevCore CLI with Template Engine (Python)

Enhancement: Added support for dynamic template generation using Jinja2.

Usage examples:
  python devcore_cli.py new project --type wordpress --client "Evadne Beauty" --stack "wordpress+woo" --init-git
  python devcore_cli.py list

This version supports:
- Jinja2-based templating for files (README.md, .env, docker-compose.yml, etc.)
- Custom template directory (defaults to ./templates)
- Auto-create SQLite project record
- Git init (optional)

"""
from core.github_integration import github_login, github_init, github_connect
from core.wp_init import init_wp_project
from core.wp_dropdb import drop_wp_database
from core.command_config import cmd_reset_config
from core.command_wp_setup import cmd_wp_setup, generate_project_config
from core.env_manager import rebuild_env_config

import argparse
import argparse
import os
import sqlite3
import subprocess
import sys
import shutil
from datetime import datetime
from jinja2 import Environment, FileSystemLoader, select_autoescape
from core import env_manager

DB_PATH = os.path.join(os.getcwd(), "devcore_projects.db")
TEMPLATE_DIR = os.path.join(os.getcwd(), "templates")

def ensure_db():
    conn = sqlite3.connect(DB_PATH)
    cur = conn.cursor()
    cur.execute("""
        CREATE TABLE IF NOT EXISTS projects (
            id INTEGER PRIMARY KEY AUTOINCREMENT,
            name TEXT,
            client_name TEXT,
            project_type TEXT,
            stack TEXT,
            path TEXT,
            repo_url TEXT,
            status TEXT,
            created_at TEXT
        )
    """)
    conn.commit()
    cur.execute("SELECT id, name, client_name, project_type, stack, path, repo_url, status, created_at FROM projects ORDER BY id DESC")
    rows = cur.fetchall()
    for r in rows:
        print(f"[{r[0]}] {r[1]} ({r[2]}) — {r[3]} — {r[6]} — {r[7]} — {r[8]}")
    conn.close()

def add_project_to_db(name, client_name, project_type, stack, path, repo_url=None, status="created"):
    conn = sqlite3.connect(DB_PATH)
    cur = conn.cursor()
    cur.execute(
        "INSERT INTO projects (name, client_name, project_type, stack, path, repo_url, status, created_at) VALUES (?,?,?,?,?,?,?,?)",
        (name, client_name, project_type, stack, path, repo_url, status, datetime.utcnow().isoformat()),
    )
    conn.commit()
    conn.close()

def shutil_which(cmd):
    from shutil import which
    return which(cmd)

def init_git_repo(path):
    if not shutil_which("git"):
        print("Git not available, skipping.")
        return None
    try:
        subprocess.check_call(["git", "init"], cwd=path)
        subprocess.check_call(["git", "add", "--all"], cwd=path)
        subprocess.check_call(["git", "commit", "-m", "chore: initial scaffold by devcore"], cwd=path)
        print("Initialized local git repository.")
        return "local"
    except subprocess.CalledProcessError:
        print("Git init failed.")
        return None

def render_template(template_name, context):
    env = Environment(
        loader=FileSystemLoader(TEMPLATE_DIR),
        autoescape=select_autoescape()
    )
    template = env.get_template(template_name)
    return template.render(context)

def write_file(path, content):
    os.makedirs(os.path.dirname(path), exist_ok=True)
    with open(path, "w", encoding="utf-8") as f:
        f.write(content)

def create_default_templates():
    os.makedirs(TEMPLATE_DIR, exist_ok=True)
    defaults = {
        "README.md.j2": "# {{ name }}\n\nClient: {{ client_name }}\nType: {{ project_type }}\nStack: {{ stack }}\n\nGenerated by DevCore CLI.\n",
        ".env.example.j2": "APP_NAME={{ name }}\nCLIENT={{ client_name }}\nSTACK={{ stack }}\n",
        "docker-compose.yml.j2": "version: '3.7'\nservices:\n  app:\n    image: alpine:3.17\n    command: sleep infinity\n    volumes:\n      - ./:/workspace\n    working_dir: /workspace\n"
    }
    for filename, content in defaults.items():
        path = os.path.join(TEMPLATE_DIR, filename)
        if not os.path.exists(path):
            with open(path, "w", encoding="utf-8") as f:
                f.write(content)

def create_project_structure(base_path, context):
    os.makedirs(base_path, exist_ok=True)
    for file in ["README.md.j2", ".env.example.j2", "docker-compose.yml.j2"]:
        output_name = file.replace(".j2", "")
        rendered = render_template(file, context)
        write_file(os.path.join(base_path, output_name), rendered)
    write_file(os.path.join(base_path, "src", "hello.txt"), f"This is {context['name']} scaffolded by devcore.\n")

def cmd_new_project(args):
    name = args.name or f"{args.client}-{args.type}"
    project_folder = os.path.join(os.getcwd(), name.replace(" ", "-"))

    if os.path.exists(project_folder) and not args.force:
        print(f"Error: {project_folder} exists. Use --force to overwrite.")
        sys.exit(1)
    if os.path.exists(project_folder) and args.force:
        shutil.rmtree(project_folder)

    create_default_templates()
    context = {
        "name": name,
        "client_name": args.client,
        "project_type": args.type,
        "stack": args.stack or "default",
    }

    print(f"Creating project '{name}'...")
    create_project_structure(project_folder, context)

    repo_marker = None
    if args.init_git:
        repo_marker = init_git_repo(project_folder)

    ensure_db()
    add_project_to_db(name, args.client, args.type, args.stack or "default", project_folder, repo_url=None)
    print(f"Project created at {project_folder}\nDatabase updated.")

def cmd_list_projects(_args):
    ensure_db()
    conn = sqlite3.connect(DB_PATH)
    cur = conn.cursor()
    cur.execute("SELECT id, name, client_name, project_type, stack, path, status, created_at FROM projects ORDER BY id DESC")
    rows = cur.fetchall()
    if not rows:
        print("No projects found.")
        return
    for r in rows:
        print(f"[{r[0]}] {r[1]} ({r[2]}) — {r[3]} — {r[6]} — {r[7]}")

def build_parser():
    parser = argparse.ArgumentParser(prog="devcore", description="DevCore CLI with Template Engine")
    subparsers = parser.add_subparsers(dest="command")

    new_parser = subparsers.add_parser("new", help="Create new resource")
    new_sub = new_parser.add_subparsers(dest="subcommand")

    proj = new_sub.add_parser("project", help="Create project scaffold")
    proj.add_argument("--type", required=True, choices=["wordpress", "laravel", "nextjs", "api"], help="Project type")
    proj.add_argument("--client", required=True, help="Client name")
    proj.add_argument("--name", required=False, help="Project name")
    proj.add_argument("--stack", required=False, help="Stack info")
    proj.add_argument("--init-git", action="store_true", help="Initialize git repo")
    proj.add_argument("--force", action="store_true", help="Overwrite existing folder")
    proj.set_defaults(func=cmd_new_project)

    list_parser = subparsers.add_parser("list", help="List projects")
    list_parser.set_defaults(func=cmd_list_projects)
    
    # ---------------------------
    # GitHub Integration Commands
    # ---------------------------
    login_parser = subparsers.add_parser("login", help="Login ke layanan eksternal")
    login_sub = login_parser.add_subparsers(dest="service")

    github_login_parser = login_sub.add_parser("github", help="Login ke GitHub dengan Personal Access Token")
    github_login_parser.set_defaults(func=lambda args: github_login())

    github_parser = subparsers.add_parser("github", help="GitHub management commands")
    github_sub = github_parser.add_subparsers(dest="action")

    github_init_parser = github_sub.add_parser("init", help="Buat repo GitHub baru dan hubungkan ke project lokal")
    github_init_parser.add_argument("project_name", help="Nama project (harus sudah dibuat sebelumnya)")
    github_init_parser.set_defaults(func=lambda args: github_init(args.project_name, os.path.join(os.getcwd(), args.project_name)))

    github_connect_parser = github_sub.add_parser("connect", help="Hubungkan project lokal ke repo GitHub yang sudah ada")
    github_connect_parser.add_argument("project_name", help="Nama project lokal")
    github_connect_parser.add_argument("repo_url", help="URL repo GitHub yang sudah ada")
    github_connect_parser.set_defaults(func=lambda args: github_connect(os.path.join(os.getcwd(), args.project_name), args.repo_url))

    # ---------------------------
    # WordPress Project Generator
    # ---------------------------
    wp_parser = subparsers.add_parser("wp", help="WordPress utilities")
    wp_sub = wp_parser.add_subparsers(dest="action")

    # wp_init_parser = wp_sub.add_parser("init", help="Inisialisasi project WordPress baru")
    # wp_init_parser.add_argument("project_name", help="Nama project WordPress")
    # wp_init_parser.set_defaults(func=lambda args: init_wp_project(args.project_name))
    wp_init_parser = wp_sub.add_parser("init", help="Inisialisasi project WordPress baru")
    wp_init_parser.add_argument("project_name", help="Nama project WordPress")
    wp_init_parser.add_argument(
        "--include-setup",
        action="store_true",
        help="Jalankan setup plugin & theme otomatis setelah init selesai"
    )
    wp_init_parser.set_defaults(func=lambda args: init_wp_project(args.project_name, include_setup=args.include_setup))

    
    
    wp_drop_parser = wp_sub.add_parser("dropdb", help="Hapus database, folder project, dan record dari DevCore")
    wp_drop_parser.add_argument("project_name", help="Nama project WordPress yang ingin dihapus")
    wp_drop_parser.set_defaults(func=lambda args: drop_wp_database(args.project_name))
    
    wp_setup_parser = wp_sub.add_parser("setup", help="Setup plugin & theme WordPress sesuai devcore_project.json")
    wp_setup_parser.add_argument("--generate", action="store_true", help="Buat ulang file devcore_project.json")
    wp_setup_parser.add_argument("project_dir", nargs="?", default=".", help="Path direktori proyek WordPress")
    wp_setup_parser.set_defaults(func=lambda args: generate_project_config(args.project_dir) if args.generate else cmd_wp_setup(args.project_dir))

    
    # ---------------------------
    # Environment Config Commands
    # ---------------------------
    config_parser = subparsers.add_parser("config", help="Kelola konfigurasi environment DevCore")
    config_sub = config_parser.add_subparsers(dest="action")

    config_reset_parser = config_sub.add_parser("reset", help="Hapus konfigurasi environment dan kembalikan ke default")
    config_reset_parser.set_defaults(func=lambda args: cmd_reset_config())
    
    config_rebuild_parser = config_sub.add_parser("rebuild", help="Hapus dan buat ulang file konfigurasi environment")
    config_rebuild_parser.set_defaults(func=lambda args: rebuild_env_config())

    return parser

def main():
    global args
    parser = build_parser()
    args = parser.parse_args()
    if not args.command:
        parser.print_help()
        sys.exit(0)

    if hasattr(args, "func"):
        args.func(args)
    else:
        print("Invalid command.")

if __name__ == "__main__":
    main()
    
    