"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.ViolationCheck = void 0;
class ViolationCheck {
    constructor(ruleCheck, templatePath) {
        this.ruleCheck = ruleCheck;
        this.templatePath = templatePath;
        this.violatingResources = new Map();
        this.violation = {};
    }
    /**
     * A single guard rule can contain multiple "checks" that are run against a resource
     * or against multiple resources. So for example you might get something like:
     * {
     *   name: 's3-public-buckets-prohibited',
     *   messages: {
     *     custom_message: 'Buckets should not be public',
     *   },
     *   checks: [
     *     {
     *       traversed: {
     *         to: {
     *           path: '/Resources/MyCustomL3ConstructBucket8C61BCA7/Properties/PublicAccessBlockConfiguration/BlockPublicPolicy'
     *         }
     *       }
     *     },
     *     {
     *       traversed: {
     *         to: {
     *           path: '/Resources/MyCustomL3ConstructBucket8C61BCA7/Properties/PublicAccessBlockConfiguration/BlockPublicAcls'
     *         }
     *       }
     *     }
     *   ]
     * }
     *
     * We want to display this to the user as a single violation since there is a single
     * custom_message. This method sets up some inheritance and constructs a single violation per
     * resource+message.
     */
    setViolatingResources(check) {
        // pull the description from the custom message or from the error message if available
        this.violation.description = this.violation.description || check.messages?.custom_message || check.messages?.error_message;
        // The fix will only appear in a custom_message because it would be a user
        // generated message
        this.violation.fix = this.violation.fix ?? check.messages?.custom_message ?? 'N/A';
        const location = check.traversed.to.path;
        const resourceName = location.split('/')[2];
        const violatingResource = this.violatingResources.get(this.violation.fix);
        const result = {
            locations: [location],
        };
        if (violatingResource) {
            const resource = violatingResource.resource.get(resourceName);
            if (resource) {
                resource.locations.push(location);
            }
            else {
                violatingResource.resource.set(resourceName, result);
            }
        }
        else {
            this.violatingResources.set(this.violation.fix, {
                resource: new Map([[resourceName, result]]),
                violation: {
                    description: this.violation.description ?? '',
                    fix: this.violation.fix,
                },
            });
        }
    }
    /**
     * Process a Guard result check and return a plugin violation
     * We are establishing a bit of a convention with the messages where we expect
     * the custom_message field to contain a string formatted like this
     * (based on the Control Tower rules)
     *
     *     [FIX]: Do something...
     *     [XXX]: description of the rule
     *
     * If it does contain a structure like that then we try and parse out the
     * fix and description fields, otherwise we just take the custom_message as
     * is and use it for both.
     */
    processCheck() {
        this.ruleCheck.checks.forEach(check => {
            if (check.messages?.custom_message) {
                const message = check.messages.custom_message.split('\n').filter(m => m.trim() !== '');
                message.forEach(m => {
                    const mes = m.trim();
                    if (mes.startsWith('[FIX]')) {
                        this.violation.fix = mes;
                    }
                    else {
                        this.violation.description = mes;
                    }
                });
            }
            this.setViolatingResources(check);
        });
        return Array.from(this.violatingResources.values()).map(violation => {
            return {
                description: violation.violation.description,
                fix: violation.violation.fix,
                ruleMetadata: {
                    DocumentationUrl: this.generateRuleDocUrl(),
                },
                ruleName: this.ruleCheck.name,
                violatingResources: Array.from(violation.resource.entries()).map(([key, value]) => {
                    return {
                        locations: value.locations,
                        resourceLogicalId: key,
                        templatePath: this.templatePath,
                    };
                }),
            };
        });
    }
    generateRuleDocUrl() {
        return 'https://github.com/cdklabs/cdk-validator-cfnguard#bundled-control-tower-rules';
    }
}
exports.ViolationCheck = ViolationCheck;
//# sourceMappingURL=data:application/json;base64,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