"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.CfnGuardValidator = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const fs = require("fs");
const os = require("os");
const path = require("path");
const check_1 = require("./check");
const utils_1 = require("./utils");
// eslint-disable-next-line @typescript-eslint/no-require-imports
const packageJson = require('../package.json');
/**
 * A validation plugin using CFN Guard
 */
class CfnGuardValidator {
    constructor(props = {}) {
        this.rulesPaths = [];
        this.name = 'cdk-validator-cfnguard';
        this.disabledRules = props.disabledRules ?? [];
        const defaultRulesPath = path.join(__dirname, '..', 'rules/control-tower/cfn-guard');
        if (props.controlTowerRulesEnabled ?? true) {
            this.rulesPaths.push(defaultRulesPath);
        }
        this.rulesPaths.push(...props.rules ?? []);
        const osPlatform = os.platform();
        // guard calls it ubuntu but seems to apply to all linux
        // https://github.com/aws-cloudformation/cloudformation-guard/blob/184002cdfc0ae9e29c61995aae41b7d1f1d3b26c/install-guard.sh#L43-L46
        const platform = osPlatform === 'linux'
            ? 'ubuntu'
            : osPlatform === 'darwin' ? 'macos' : undefined;
        if (!platform) {
            throw new Error(`${os.platform()} not supported, must be either 'darwin' or 'linux'`);
        }
        this.guard = path.join(__dirname, '..', 'bin', platform, 'cfn-guard');
        this.ruleIds = this.rulesPaths.flatMap(rule => {
            const parsed = path.parse(rule);
            if (rule === defaultRulesPath || parsed.dir.startsWith(defaultRulesPath)) {
                return this.getRuleIds(rule);
            }
            return [];
        });
        // eslint-disable-next-line @typescript-eslint/no-require-imports
        this.version = packageJson.version;
    }
    getRuleIds(rulesPath) {
        const stat = fs.statSync(rulesPath);
        if (stat.isFile()) {
            const parsed = path.parse(rulesPath);
            if (!this.disabledRules.includes(parsed.name)) {
                return [path.parse(rulesPath).name.replace('ct-', '').replace(/-/g, '')];
            }
            return [];
        }
        else {
            return fs.readdirSync(rulesPath).flatMap(rule => this.getRuleIds(path.join(rulesPath, rule)));
        }
    }
    validate(context) {
        const report = this.execGuard({
            templatePaths: context.templatePaths,
            rulePaths: this.rulesPaths,
        });
        return report;
    }
    /**
     * Get the rules to execute. We can return directories as long as none of the rules in the
     * directory have been disabled
     */
    getRules(filePaths) {
        return filePaths.flatMap(file => {
            const stat = fs.statSync(file);
            if (stat.isDirectory()) {
                const dir = fs.readdirSync(file);
                const rules = dir.flatMap(d => this.getRules([path.join(file, d)]));
                if (rules.length === dir.length)
                    return [file];
                return rules;
            }
            else {
                if (!this.disabledRules.includes(path.parse(file).name)) {
                    return [file];
                }
                else {
                    return [];
                }
            }
        });
    }
    execGuard(config) {
        const flags = [
            'validate',
            ...this.getRules(config.rulePaths).flatMap(rule => ['--rules', rule]),
            ...config.templatePaths.flatMap(template => ['--data', template]),
            '--output-format', 'json',
            '--show-summary', 'none',
            '--structured',
        ];
        const violations = [];
        let success = true;
        try {
            const result = (0, utils_1.exec)([this.guard, ...flags], {
                json: true,
            });
            const guardResults = JSON.parse(JSON.stringify(result), reviver);
            for (const guardResult of guardResults) {
                if (!guardResult.not_compliant || guardResult.not_compliant.length === 0) {
                    continue;
                }
                success = false;
                guardResult.not_compliant.forEach((check) => {
                    const violationCheck = new check_1.ViolationCheck(check, guardResult.name);
                    const violation = violationCheck.processCheck();
                    violations.push(...violation);
                });
            }
        }
        catch (e) {
            success = false;
            throw new Error(`
        CfnGuardValidator plugin failed processing cfn-guard results.
        Please create an issue https://github.com/cdklabs/cdk-validator-cfnguard/issues/new
        Error: ${e}`);
        }
        return {
            success,
            violations: violations,
        };
    }
}
_a = JSII_RTTI_SYMBOL_1;
CfnGuardValidator[_a] = { fqn: "@cdklabs/cdk-validator-cfnguard.CfnGuardValidator", version: "0.0.58" };
exports.CfnGuardValidator = CfnGuardValidator;
/**
 * Guard does not have a standard JSON schema and the schema
 * that is returned can be dependent on the type of rule or type
 * of check that was executed. The results are very much an attempt to
 * display the internals of guard to the user. Trying to make sense of that
 * can be difficult.
 *
 * The result structure can depend on the way that the rule was written. For example
 * I could write a rule like this:
 *
 *     rule MY_RULE {
 *       # This is a "check" and is a `Clause` type check
 *       Properties.SomeProp == true
 *     }
 *
 * Or I could write a rule like this:
 *
 *     rule MY_RULE {
 *       #  This is a "check" and is a `Rule` type check
 *       check(Properties)
 *     }
 *     rule check(properties) {
 *       properties.SomeProp == true
 *     }
 *
 * Both of the above examples are checking the same thing
 * but the schema that is returned is different because the
 * way the rule was written is different
 *
 * This reviver function is executed bottom up and is essentially
 * creating a new object with a well known schema that the rest of the
 * plugin can work with. It looks for certain fields that always appear in
 * the guard results, but appear in different locations. It finds those fields
 * and then pulls them up the object, dropping any fields that we don't
 * care about. For example guard may return
 *
 * {
 *   Clause: {
 *     Unary: {
 *       check: {
 *         UnResolved: {
 *           value: {
 *             traversed_to: {...} // we only care about this!!!
 *           }
 *         }
 *       }
 *     }
 *   }
 * }
 *
 * Or it may return
 *
 * {
 *   Rule: {
 *     checks: [{
 *       Block: {
 *         unresolved: {
 *           traversed_to: {...} // we only care about this!!!
 *         }
 *       }
 *     }]
 *   }
 * }
 *
 * In the above example we only care about the 'traversed_to' field,
 * so this reviver function will grab that field and pull it up the object, dropping
 * the fields we don't care about, ending with something like
 * {
 *   checks: [{
 *     resolved: false,
 *     traversed: {...}
 *   }]
 * }
 *
 */
function reviver(key, value) {
    if (key === 'not_compliant') {
        // not_compliant can sometimes be an empty object (but not an Array), so we
        // process this value before diving into other object values to ensure this
        // one is always made into an Array
        return Object.values(value).map((v) => v.Rule);
    }
    else if (value !== null && typeof value === 'object' && !Array.isArray(value)) {
        return extractNestedObject(value);
    }
    else if (key === 'checks' && Array.isArray(value)) {
        return extractNestedChecks(value.flatMap(v => v));
    }
    return value;
}
/**
 * Extract a nested 'checks' object. This also handles checks
 * nested within checks. It will grab the checks at the level below
 * and pull it up to the next level.
 */
function extractNestedChecks(checks) {
    const containsNestedChecks = checks.some(check => Object.values(check).some((value) => {
        return typeof value === 'object' && value.hasOwnProperty('checks');
    }));
    if (containsNestedChecks) {
        return checks.flatMap(check => {
            if (Object.keys(check).includes('traversed')) {
                return check;
            }
            return Object.values(check).flatMap((checkValue) => {
                return Object.values(checkValue.checks ?? checkValue).flatMap((nestedCheckValue) => {
                    return {
                        ...nestedCheckValue,
                        name: checkValue.name,
                        messages: checkValue.messages ?? nestedCheckValue.messages,
                    };
                });
            });
        });
    }
    return checks.flatMap(check => {
        if (Object.keys(check).includes('traversed')) {
            return check;
        }
        return Object.values(check);
    });
}
/**
 * Extract a nested object and pull it up a level
 */
function extractNestedObject(object) {
    let newObject = object;
    Object.entries(object).forEach(([level1NestedKey, level1NestedValue]) => {
        const nestedValue = level1NestedValue;
        switch (level1NestedKey.toLowerCase()) {
            // this should always be found earlier than the rest since it appears
            // within the 'unresolved' and 'resolved' objects. The object
            // is slightly different for each case so here we create
            // a new object with the key 'traversed' with a consistent value
            case 'traversed_to':
                newObject = {
                    traversed: {
                        to: {
                            path: nestedValue.path,
                            value: nestedValue.value,
                        },
                        from: nestedValue.from ? {
                            path: nestedValue.from.path,
                            value: undefined,
                        } : undefined,
                    },
                    messages: nestedValue.messages,
                };
                break;
            // This should be found in the "second" pass after the above
            // 'traversed_to' case has been executed. We take the new
            // object that was created in the `traversed_to` case and
            // a couple other fields, dropping the rest that we don't care about
            case 'unresolved':
                newObject = {
                    resolved: false,
                    traversed: nestedValue.traversed,
                    messages: nestedValue.messages ?? object.messages,
                };
                break;
            // This should be found in the "second" pass after the above
            // 'traversed_to' case has been executed. We take the new
            // object that was created in the `traversed_to` case and
            // a couple other fields, dropping the rest that we don't care about
            // A check can either be resolved or unresolved
            case 'resolved':
            case 'inresolved':
                if ('from' in nestedValue) {
                    newObject = {
                        resolved: true,
                        traversed: {
                            from: nestedValue.from,
                            to: {
                                path: nestedValue.from.path,
                                value: nestedValue.to.value,
                            },
                        },
                        messages: nestedValue.messages,
                    };
                }
                else if ('value' in nestedValue) {
                    newObject = {
                        resolved: true,
                        traversed: {
                            from: nestedValue.value,
                            to: {
                                path: nestedValue.value.path,
                                value: nestedValue.value.value,
                            },
                        },
                        messages: nestedValue.messages,
                    };
                }
                break;
        }
        // this check will be evaluated _after_ the 'traversed_to' check and _before_ the 'resolved'
        // and 'unresolved' checks above. There may be a case where 'traversed' is nested 2 (or 3 or 4) below
        // 'unresolved' or 'resolved' and this will keep pulling it up until it is just one below
        // and the above checks can work
        if (level1NestedValue !== null && typeof level1NestedValue === 'object' && !Array.isArray(level1NestedValue)) {
            Object.entries(level1NestedValue).forEach(([level2NestedKey, level2NestedValue]) => {
                switch (level2NestedKey.toLowerCase()) {
                    case 'traversed':
                        newObject = {
                            traversed: nestedValue.traversed,
                            resolved: nestedValue.resolved,
                            messages: nestedValue.messages ?? level2NestedValue.messages ?? object.messages,
                        };
                        break;
                }
            });
        }
    });
    return newObject;
}
//# sourceMappingURL=data:application/json;base64,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