"""
File injection tool to allow reuse of ConstraintSet/KeyFrameSet/etc definitions in multiple MotionScenes.

Requirements:
- Create a directory in your project's `res` directory called `inject` e.g. `app/src/main/res/xml/`.
- All files in this directory must be prepended with `MERGE_FILE_PREFIX` which is set to `_` by default.
 - e.g. res/xml/_your_motionscene_filename.xml
 - e.g. res/xml/_your_constraintset_filename.xml

- To inject the content of one file into another, add a line with the signature
`<inject src="_your_constraintset_filename"/>` where you want that content to appear.

When you run the script, you should see the merged file `app/src/main/res/xml/your_motionscene_filename.xml`.

Warning: This is a somewhat naive pattern-based find/replace - we do not actually parse the XML tree.
         We recognise <MotionScene>...<MotionScene/> tags and <inject/> tags - everything else is just
         treated as plain text, warts and all.

"""


import argparse
import glob
import logging
import os
import re
import shutil
from typing import (
    Dict,
    List,
    Optional,
)

log = logging.getLogger(__name__)
log.setLevel(logging.INFO)
log.addHandler(logging.StreamHandler())


MERGE_FILE_PREFIX = '_'

TEMP_DIR = 'temp-scenemerge/'
DEFAULT_SOURCE_RES_DIR = 'xml'  # Name of the directory in /src/../res/ for storing source files

# <inject arg1="" arg2="" />
MERGE_TAG_PATTERN = re.compile(rf'^(<!--)?([ ]*)<inject (.*?)/>', flags=re.DOTALL | re.MULTILINE)
MERGE_ARGS = [
    'src',
]

UNWRAP_TAGS = [
    'MotionScene',
    'merge',
    'injected',
]
UNWRAP_PATTERN = r'.*?<{tag}.*?>(.*?)</{tag}>.*'
XML_FILE_HEADER = '<?xml version="1.0" encoding="utf-8"?>'
INJECTION_FILE_HEADER = '<!--\nWARNING: This file was generated by scenemerge - any changes may be overwritten!\nYou should edit the source file \'{filename}\' instead.\n-->\n'
INJECTION_MESSAGE_START = '<!-- Start injected content from \'{filename}\' -->\n'
INJECTION_MESSAGE_END = '<!-- End injected content from \'{filename}\' -->\n'
IGNORED_LINES = [
    '<?xml version="1.0" encoding="utf-8"?>',
    'xmlns:android="http://schemas.android.com/apk/res/android"',
    'xmlns:app="http://schemas.android.com/apk/res-auto"',
    'xmlns:motion="http://schemas.android.com/apk/res-auto"',
    'xmlns:tools="http://schemas.android.com/tools"',
]


class SourceFile:
    def __init__(self, filepath):
        self.filepath = filepath
        self.filename = os.path.basename(filepath)
        self.resolved = False  # Set True when all inject tags have been processed.
        self.dependencies = []
        self.is_injected = False  # Set True when this file has been injected into another.

    def __str__(self):
        return f'{self.filename}: {self.resolved}'

    def resolve_injections(self, sources: Dict[str, 'SourceFile']) -> int:
        if self.resolved:
            return 0

        changes = 0

        with open(self.filepath, 'r') as f:
            text = f.read()

        tags = _find_merge_tags(text)

        if len(tags) == 0:
            # No <inject/> tags found - text is final
            self.resolved = True
            changes = changes + 1
            return changes

        for t in tags:
            if t.src in sources:
                src = sources[t.src]
            else:
                raise KeyError(f'Cannot find source for tag referencing \'{t.src}\': {sources.keys()}')

            if src.resolved:
                # If source file is resolved, copy its content here.
                # Otherwise we wait for the next pass to check again
                self._add_depencency(src)

                with open(src.filepath, 'r') as f:
                    content = _get_wrapped_content(f)
                    if content is None:
                        content = _get_generic_content(f, t.indent)

                content = self._wrap_tag_content(content, src)
                text = text.replace(t.tag, content)
                with open(self.filepath, 'w') as f:
                    f.write(text)

                changes = changes + 1

        return changes

    def _add_depencency(self, dep: 'SourceFile'):
        dep.is_injected = True
        self.dependencies.append(dep.filename)
        self.dependencies += dep.dependencies

    def _wrap_tag_content(self, text: str, src: 'SourceFile') -> str:
        before = INJECTION_MESSAGE_START.format(filename=src.filename)
        after = INJECTION_MESSAGE_END.format(filename=src.filename)
        return f'{before}{text}{after}'


class MergeTag:
    def __init__(self, tag: str,  src: str, indent: int):
        if not src.endswith('.xml'):
            src = f'{src}.xml'
        self.tag = tag  # Original text of the <inject .../> tag this represents
        self.src = src
        self.indent = indent

    def __str__(self):
        return f'{self.tag}'


def _parse_mergetag(match) -> Optional['MergeTag']:
    if match.group(1):
        """Comment tag `<!--` found at start of line - tag should be ignored."""
        return
    indent = len(match.group(2))
    args_src = match.group(3)

    args = {}
    for a in MERGE_ARGS:
        args[a] = re.search(f'{a}="(.*?)"', args_src)[1]

    return MergeTag(tag=match.group(0), indent=indent, **args)


def _find_merge_tags(src_text: str) -> List[MergeTag]:
    matches = [x for x in MERGE_TAG_PATTERN.finditer(src_text)]
    return list(filter(None, [_parse_mergetag(m) for m in matches]))


def _get_source_filepaths(rootdir: str, sourceset: str, res_dir: str) -> List[str]:
    glb = f'{rootdir}/**/{sourceset}/res/{res_dir}/{MERGE_FILE_PREFIX}*.xml'.replace('//', '/')
    print(glb)
    result = glob.glob(glb, recursive=True)
    log.info(f'Found {len(result)} files in xml resource directory (root={rootdir})...')
    return result


def _merge_sources_for_directory(
        root: str,
        sourceset: str = 'main',
        res_dir: str = DEFAULT_SOURCE_RES_DIR,
        keep_transitive=False
):
    source_filepaths = _get_source_filepaths(root, sourceset, res_dir)
    working_filepaths = _clone_to_working_directory(source_filepaths)
    working_files = [SourceFile(x) for x in working_filepaths]

    sourceset_res_dir = glob.glob(f'{root}/**/{sourceset}/res/', recursive=True)[0]
    output_dir = os.path.join(sourceset_res_dir, 'xml')
    _merge_sources(working_files, output_dir, keep_transitive)

    _clean_up()


def _merge_sources(source_files: List['SourceFile'], output_dir: str, keep_transitive=False):
    """Resolve inject tags until all have been handled."""
    sources = _build_sourcemap(source_files)

    loop_count = 0
    while True:
        changes = 0
        for f in source_files:
            changes = changes + f.resolve_injections(sources)

        if changes == 0:
            break

        loop_count = loop_count + 1
        if loop_count > 10:
            raise Exception("Excessive looping")

    unresolved = [x for x in source_files if not x.resolved]
    if unresolved:
        log.warning(f'Process finished with {len(unresolved)} unresolved <inject/> tags:')
        for x in unresolved:
            log.warning(f'{x.filename} with dependencies={x.dependencies}')
        log.warning(f'Available sources: {sources.keys()}')
    else:
        log.debug(f'Finished in {loop_count} passes')

    # Merging complete - now copy the resulting files to output directory
    if not os.path.exists(output_dir):
        os.makedirs(output_dir)

    if keep_transitive:
        files_to_be_copied = source_files
    else:
        files_to_be_copied = [src for src in source_files if not src.is_injected]

    for src in files_to_be_copied:
        with open(src.filepath, 'r') as f:
            content = f.read()

        output_filename = src.filename.replace(MERGE_FILE_PREFIX, '', 1)
        content = content.replace(
            XML_FILE_HEADER,
            f'{XML_FILE_HEADER}\n{INJECTION_FILE_HEADER.format(filename=src.filename)}')
        with open(os.path.join(output_dir, output_filename), 'w') as f:
            f.write(content)


def _build_sourcemap(source_files: List['SourceFile']) -> Dict['str', 'SourceFile']:
    return {
        sourcefile.filename: sourcefile for sourcefile in source_files
    }


def _clone_to_working_directory(xml_files: List[str]) -> List[str]:
    """Copy files to a temporary directory for processing."""
    if not os.path.exists(TEMP_DIR):
        os.makedirs(TEMP_DIR)

    working_filepaths = []

    for f in xml_files:
        filename = os.path.basename(f)
        newpath = shutil.copy(f, os.path.join(TEMP_DIR, filename))
        working_filepaths.append(newpath)

    return working_filepaths


def _clean_up():
    if os.path.exists(TEMP_DIR):
        for f in os.listdir(TEMP_DIR):
            os.remove(os.path.join(TEMP_DIR, f))
        os.rmdir(TEMP_DIR)


def _parse_args():
    parser = argparse.ArgumentParser()

    parser.add_argument(
        'root',
        default='.',
    )

    parser.add_argument(
        '--source',
        type=str,
        default='main',
        choices=['main', 'debug', 'test', 'androidTest'],
        help=(
            'Which Android sourceset to use.'
        )
    )

    parser.add_argument(
        '-keep_transitive',
        action='store_true',
        default=False,
        help=(
            'Keep files that were injected into some parent file. '
            'If not set, these files will be skipped when copying files '
            'to output res/xml directory.'
        ),
    )

    parser.add_argument(
        '--resdir',
        type=str,
        default='xml',
        help=(
            'Name of the directory in which you store your source template files'
        )
    )

    return parser.parse_args()


def main():
    _args = _parse_args()
    _merge_sources_for_directory(_args.root, _args.source, _args.resdir, _args.keep_transitive)


if __name__ == '__main__':
    main()


def _get_wrapped_content(file) -> Optional[str]:
    """
    Return any content that lies within any of the tags defined in UNWRAP_TAGS.
    e.g. <MotionScene>...</MotionScene>
    """
    file.seek(0)
    content = file.read()
    for tag in UNWRAP_TAGS:
        match = re.match(
            pattern=UNWRAP_PATTERN.format(tag=tag),
            string=content,
            flags=re.DOTALL
        )
        if match:
            return match.group(1)


def _get_generic_content(file, indent) -> str:
    file.seek(0)
    content = ''
    for line in file.readlines():
        stripped = _stripped(line)
        if stripped:
            content = content + _get_indented_line(stripped, indent)

    return content


def _get_indented_line(content, indent):
    indent = ' ' * indent
    return f'{indent}{content}'


def _stripped(line):
    for ignored in IGNORED_LINES:
        if ignored in line:
            content = line.replace(ignored, '').strip()
            if content:
                log.info(f'{line.strip()} -> {content}')
                return content
            else:
                return None

    return line
