"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.collectRuntimeInformation = void 0;
const path_1 = require("path");
const node_version_1 = require("./node-version");
// list of NPM scopes included in version reporting e.g. @aws-cdk and @aws-solutions-konstruk
const WHITELIST_SCOPES = ['@aws-cdk', '@aws-solutions-konstruk', '@aws-solutions-constructs'];
/**
 * Returns a list of loaded modules and their versions.
 */
function collectRuntimeInformation() {
    const libraries = {};
    for (const fileName of Object.keys(require.cache)) {
        const pkg = findNpmPackage(fileName);
        if (pkg && !pkg.private) {
            libraries[pkg.name] = pkg.version;
        }
    }
    // include only libraries that are in the whitelistLibraries list
    for (const name of Object.keys(libraries)) {
        let foundMatch = false;
        for (const scope of WHITELIST_SCOPES) {
            if (name.startsWith(`${scope}/`)) {
                foundMatch = true;
            }
        }
        if (!foundMatch) {
            delete libraries[name];
        }
    }
    // add jsii runtime version
    libraries['jsii-runtime'] = getJsiiAgentVersion();
    return { libraries };
}
exports.collectRuntimeInformation = collectRuntimeInformation;
/**
 * Determines which NPM module a given loaded javascript file is from.
 *
 * The only infromation that is available locally is a list of Javascript files,
 * and every source file is associated with a search path to resolve the further
 * ``require`` calls made from there, which includes its own directory on disk,
 * and parent directories - for example:
 *
 * [ '...repo/packages/aws-cdk-resources/lib/cfn/node_modules',
 *   '...repo/packages/aws-cdk-resources/lib/node_modules',
 *   '...repo/packages/aws-cdk-resources/node_modules',
 *   '...repo/packages/node_modules',
 *   // etc...
 * ]
 *
 * We are looking for ``package.json`` that is anywhere in the tree, except it's
 * in the parent directory, not in the ``node_modules`` directory. For this
 * reason, we strip the ``/node_modules`` suffix off each path and use regular
 * module resolution to obtain a reference to ``package.json``.
 *
 * @param fileName a javascript file name.
 * @returns the NPM module infos (aka ``package.json`` contents), or
 *      ``undefined`` if the lookup was unsuccessful.
 */
function findNpmPackage(fileName) {
    const mod = require.cache[fileName];
    if (!(mod === null || mod === void 0 ? void 0 : mod.paths)) {
        // sometimes this can be undefined. for example when querying for .json modules
        // inside a jest runtime environment.
        // see https://github.com/aws/aws-cdk/issues/7657
        // potentially we can remove this if it turns out to be a bug in how jest implemented the 'require' module.
        return undefined;
    }
    // For any path in ``mod.paths`` that is a node_modules folder, use its parent directory instead.
    const paths = mod === null || mod === void 0 ? void 0 : mod.paths.map((path) => path_1.basename(path) === 'node_modules' ? path_1.dirname(path) : path);
    try {
        const packagePath = require.resolve(
        // Resolution behavior changed in node 12.0.0 - https://github.com/nodejs/node/issues/27583
        node_version_1.major >= 12 ? './package.json' : 'package.json', { paths });
        // eslint-disable-next-line @typescript-eslint/no-require-imports
        return require(packagePath);
    }
    catch (e) {
        return undefined;
    }
}
function getJsiiAgentVersion() {
    let jsiiAgent = process.env.JSII_AGENT;
    // if JSII_AGENT is not specified, we will assume this is a node.js runtime
    // and plug in our node.js version
    if (!jsiiAgent) {
        jsiiAgent = `node.js/${process.version}`;
    }
    return jsiiAgent;
}
//# sourceMappingURL=data:application/json;base64,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