"use strict";
const child_process = require("child_process");
const sinon = require("sinon");
const lib_1 = require("../lib");
module.exports = {
    'tearDown'(callback) {
        sinon.restore();
        callback();
    },
    'bundling with image from registry'(test) {
        const spawnSyncStub = sinon.stub(child_process, 'spawnSync').returns({
            status: 0,
            stderr: Buffer.from('stderr'),
            stdout: Buffer.from('stdout'),
            pid: 123,
            output: ['stdout', 'stderr'],
            signal: null,
        });
        const image = lib_1.BundlingDockerImage.fromRegistry('alpine');
        image._run({
            command: ['cool', 'command'],
            environment: {
                VAR1: 'value1',
                VAR2: 'value2',
            },
            volumes: [{ hostPath: '/host-path', containerPath: '/container-path' }],
            workingDirectory: '/working-directory',
            user: 'user:group',
        });
        test.ok(spawnSyncStub.calledWith('docker', [
            'run', '--rm',
            '-u', 'user:group',
            '-v', '/host-path:/container-path:delegated',
            '--env', 'VAR1=value1',
            '--env', 'VAR2=value2',
            '-w', '/working-directory',
            'alpine',
            'cool', 'command',
        ], { stdio: ['ignore', process.stderr, 'inherit'] }));
        test.done();
    },
    'bundling with image from asset'(test) {
        const imageId = 'abcdef123456';
        const spawnSyncStub = sinon.stub(child_process, 'spawnSync').returns({
            status: 0,
            stderr: Buffer.from('stderr'),
            stdout: Buffer.from(`sha256:${imageId}`),
            pid: 123,
            output: ['stdout', 'stderr'],
            signal: null,
        });
        const image = lib_1.BundlingDockerImage.fromAsset('docker-path', {
            buildArgs: {
                TEST_ARG: 'cdk-test',
            },
        });
        image._run();
        test.ok(spawnSyncStub.firstCall.calledWith('docker', [
            'build', '-q',
            '--build-arg', 'TEST_ARG=cdk-test',
            'docker-path',
        ]));
        test.ok(spawnSyncStub.secondCall.calledWith('docker', [
            'run', '--rm',
            imageId,
        ]));
        test.done();
    },
    'throws if image id cannot be extracted from build output'(test) {
        sinon.stub(child_process, 'spawnSync').returns({
            status: 0,
            stderr: Buffer.from('stderr'),
            stdout: Buffer.from('stdout'),
            pid: 123,
            output: ['stdout', 'stderr'],
            signal: null,
        });
        test.throws(() => lib_1.BundlingDockerImage.fromAsset('docker-path'), /Failed to extract image ID from Docker build output/);
        test.done();
    },
    'throws in case of spawnSync error'(test) {
        sinon.stub(child_process, 'spawnSync').returns({
            status: 0,
            stderr: Buffer.from('stderr'),
            stdout: Buffer.from('stdout'),
            pid: 123,
            output: ['stdout', 'stderr'],
            signal: null,
            error: new Error('UnknownError'),
        });
        const image = lib_1.BundlingDockerImage.fromRegistry('alpine');
        test.throws(() => image._run(), /UnknownError/);
        test.done();
    },
    'throws if status is not 0'(test) {
        sinon.stub(child_process, 'spawnSync').returns({
            status: -1,
            stderr: Buffer.from('stderr'),
            stdout: Buffer.from('stdout'),
            pid: 123,
            output: ['stdout', 'stderr'],
            signal: null,
        });
        const image = lib_1.BundlingDockerImage.fromRegistry('alpine');
        test.throws(() => image._run(), /\[Status -1\]/);
        test.done();
    },
};
//# sourceMappingURL=data:application/json;base64,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