"use strict";
const nodeunit_1 = require("nodeunit");
const lib_1 = require("../lib");
const util_1 = require("../lib/util");
class SomeToken {
    constructor() {
        this.foo = 60;
        this.goo = 40;
    }
    resolve() {
        return this.foo + this.goo;
    }
}
class Nested extends lib_1.Stack {
    constructor(scope, id) {
        const resource = new lib_1.CfnResource(scope, `${id}+NestedStackResource`, { type: 'AWS::CloudFormation::Stack' });
        super(scope, id);
        this.nestedStackResource = resource;
    }
}
module.exports = nodeunit_1.testCase({
    'capitalizeResourceProperties capitalizes all keys of an object (recursively) from camelCase to PascalCase'(test) {
        const c = new lib_1.Stack();
        test.equal(util_1.capitalizePropertyNames(c, undefined), undefined);
        test.equal(util_1.capitalizePropertyNames(c, 12), 12);
        test.equal(util_1.capitalizePropertyNames(c, 'hello'), 'hello');
        test.deepEqual(util_1.capitalizePropertyNames(c, ['hello', 88]), ['hello', 88]);
        test.deepEqual(util_1.capitalizePropertyNames(c, { Hello: 'world', hey: 'dude' }), { Hello: 'world', Hey: 'dude' });
        test.deepEqual(util_1.capitalizePropertyNames(c, [1, 2, { three: 3 }]), [1, 2, { Three: 3 }]);
        test.deepEqual(util_1.capitalizePropertyNames(c, { Hello: 'world', recursive: { foo: 123, there: { another: ['hello', { world: 123 }] } } }), { Hello: 'world', Recursive: { Foo: 123, There: { Another: ['hello', { World: 123 }] } } });
        // make sure tokens are resolved and result is also capitalized
        test.deepEqual(util_1.capitalizePropertyNames(c, { hello: { resolve: () => ({ foo: 'bar' }) }, world: new SomeToken() }), { Hello: { Foo: 'bar' }, World: 100 });
        test.done();
    },
    ignoreEmpty: {
        '[]'(test) {
            const stack = new lib_1.Stack();
            test.strictEqual(stack.resolve(util_1.ignoreEmpty([])), undefined);
            test.done();
        },
        '{}'(test) {
            const stack = new lib_1.Stack();
            test.strictEqual(stack.resolve(util_1.ignoreEmpty({})), undefined);
            test.done();
        },
        'undefined/null'(test) {
            const stack = new lib_1.Stack();
            test.strictEqual(stack.resolve(util_1.ignoreEmpty(undefined)), undefined);
            test.strictEqual(stack.resolve(util_1.ignoreEmpty(null)), null);
            test.done();
        },
        'primitives'(test) {
            const stack = new lib_1.Stack();
            test.strictEqual(stack.resolve(util_1.ignoreEmpty(12)), 12);
            test.strictEqual(stack.resolve(util_1.ignoreEmpty('12')), '12');
            test.done();
        },
        'non-empty arrays/objects'(test) {
            const stack = new lib_1.Stack();
            test.deepEqual(stack.resolve(util_1.ignoreEmpty([1, 2, 3, undefined])), [1, 2, 3]); // undefined array values is cleaned up by "resolve"
            test.deepEqual(stack.resolve(util_1.ignoreEmpty({ o: 1, b: 2, j: 3 })), { o: 1, b: 2, j: 3 });
            test.done();
        },
        'resolve first'(test) {
            const stack = new lib_1.Stack();
            test.deepEqual(stack.resolve(util_1.ignoreEmpty({ xoo: { resolve: () => 123 } })), { xoo: 123 });
            test.strictEqual(stack.resolve(util_1.ignoreEmpty({ xoo: { resolve: () => undefined } })), undefined);
            test.deepEqual(stack.resolve(util_1.ignoreEmpty({ xoo: { resolve: () => [] } })), { xoo: [] });
            test.deepEqual(stack.resolve(util_1.ignoreEmpty({ xoo: { resolve: () => [undefined, undefined] } })), { xoo: [] });
            test.done();
        },
    },
    filterUnderined: {
        'is null-safe (aka treats null and undefined the same)'(test) {
            test.deepEqual(util_1.filterUndefined({ 'a null': null, 'a not null': true }), { 'a not null': true });
            test.done();
        },
        'removes undefined, but leaves the rest'(test) {
            test.deepEqual(util_1.filterUndefined({ 'an undefined': undefined, 'yes': true }), { yes: true });
            test.done();
        },
    },
    'pathToTopLevelStack returns the array of stacks that lead to a stack'(test) {
        const a = new lib_1.Stack(undefined, 'a');
        const aa = new Nested(a, 'aa');
        const aaa = new Nested(aa, 'aaa');
        test.deepEqual(path(aaa), ['a', 'aa', 'aaa']);
        test.deepEqual(path(aa), ['a', 'aa']);
        test.deepEqual(path(a), ['a']);
        test.done();
        function path(s) {
            return util_1.pathToTopLevelStack(s).map(x => x.node.id);
        }
    },
    'findCommonStack returns the lowest common stack between two stacks or undefined'(test) {
        const a = new lib_1.Stack(undefined, 'a');
        const aa = new Nested(a, 'aa');
        const ab = new Nested(a, 'ab');
        const aaa = new Nested(aa, 'aaa');
        const aab = new Nested(aa, 'aab');
        const aba = new Nested(ab, 'aba');
        const b = new lib_1.Stack(undefined, 'b');
        const ba = new Nested(b, 'ba');
        const baa = new Nested(ba, 'baa');
        test.equal(lca(a, b), undefined);
        test.equal(lca(aa, ab), 'a');
        test.equal(lca(ab, aa), 'a');
        test.equal(lca(aa, aba), 'a');
        test.equal(lca(aba, aa), 'a');
        test.equal(lca(ab, aba), 'ab');
        test.equal(lca(aba, ab), 'ab');
        test.equal(lca(aba, aba), 'aba');
        test.equal(lca(aa, aa), 'aa');
        test.equal(lca(a, aaa), 'a');
        test.equal(lca(aaa, aab), 'aa');
        test.equal(lca(aaa, b), undefined);
        test.equal(lca(aaa, ba), undefined);
        test.equal(lca(baa, ba), 'ba');
        test.done();
        function lca(s1, s2) {
            const res = util_1.findLastCommonElement(util_1.pathToTopLevelStack(s1), util_1.pathToTopLevelStack(s2));
            if (!res) {
                return undefined;
            }
            return res.node.id;
        }
    },
});
//# sourceMappingURL=data:application/json;base64,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