"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Annotations = void 0;
const cxschema = require("@aws-cdk/cloud-assembly-schema");
/**
 * Includes API for attaching annotations such as warning messages to constructs.
 */
class Annotations {
    constructor(scope) {
        this.scope = scope;
    }
    /**
     * Returns the annotations API for a construct scope.
     * @param scope The scope
     */
    static of(scope) {
        return new Annotations(scope);
    }
    /**
     * Adds a warning metadata entry to this construct.
     *
     * The CLI will display the warning when an app is synthesized, or fail if run
     * in --strict mode.
     *
     * @param message The warning message.
     */
    addWarning(message) {
        this.addMessage(cxschema.ArtifactMetadataEntryType.WARN, message);
    }
    /**
     * Adds an info metadata entry to this construct.
     *
     * The CLI will display the info message when apps are synthesized.
     *
     * @param message The info message.
     */
    addInfo(message) {
        this.addMessage(cxschema.ArtifactMetadataEntryType.INFO, message);
    }
    /**
     * Adds an { "error": <message> } metadata entry to this construct.
     * The toolkit will fail synthesis when errors are reported.
     * @param message The error message.
     */
    addError(message) {
        this.addMessage(cxschema.ArtifactMetadataEntryType.ERROR, message);
    }
    /**
     * Adds a message metadata entry to the construct node, to be displayed by the CDK CLI.
     * @param level The message level
     * @param message The message itself
     */
    addMessage(level, message) {
        this.scope.node.addMetadata(level, message);
    }
}
exports.Annotations = Annotations;
//# sourceMappingURL=data:application/json;base64,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