"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.rootPathTo = exports.Stack = void 0;
const fs = require("fs");
const path = require("path");
const cxschema = require("@aws-cdk/cloud-assembly-schema");
const cxapi = require("@aws-cdk/cx-api");
const annotations_1 = require("./annotations");
const app_1 = require("./app");
const arn_1 = require("./arn");
const cfn_element_1 = require("./cfn-element");
const cfn_fn_1 = require("./cfn-fn");
const cfn_pseudo_1 = require("./cfn-pseudo");
const cfn_resource_1 = require("./cfn-resource");
const construct_compat_1 = require("./construct-compat");
const context_provider_1 = require("./context-provider");
const cloudformation_lang_1 = require("./private/cloudformation-lang");
const logical_id_1 = require("./private/logical-id");
const resolve_1 = require("./private/resolve");
const uniqueid_1 = require("./private/uniqueid");
const STACK_SYMBOL = Symbol.for('@aws-cdk/core.Stack');
const MY_STACK_CACHE = Symbol.for('@aws-cdk/core.Stack.myStack');
const VALID_STACK_NAME_REGEX = /^[A-Za-z][A-Za-z0-9-]*$/;
/**
 * A root construct which represents a single CloudFormation stack.
 */
class Stack extends construct_compat_1.Construct {
    /**
     * Creates a new stack.
     *
     * @param scope Parent of this stack, usually an `App` or a `Stage`, but could be any construct.
     * @param id The construct ID of this stack. If `stackName` is not explicitly
     * defined, this id (and any parent IDs) will be used to determine the
     * physical ID of the stack.
     * @param props Stack properties.
     */
    constructor(scope, id, props = {}) {
        var _a;
        // For unit test scope and id are optional for stacks, but we still want an App
        // as the parent because apps implement much of the synthesis logic.
        scope = scope !== null && scope !== void 0 ? scope : new app_1.App({
            autoSynth: false,
            outdir: fs_1.FileSystem.mkdtemp('cdk-test-app-'),
        });
        // "Default" is a "hidden id" from a `node.uniqueId` perspective
        id = id !== null && id !== void 0 ? id : 'Default';
        super(scope, id);
        this._missingContext = new Array();
        this._stackDependencies = {};
        this.templateOptions = {};
        Object.defineProperty(this, STACK_SYMBOL, { value: true });
        this._logicalIds = new logical_id_1.LogicalIDs();
        const { account, region, environment } = this.parseEnvironment(props.env);
        this.account = account;
        this.region = region;
        this.environment = environment;
        this.terminationProtection = props.terminationProtection;
        if (props.description !== undefined) {
            // Max length 1024 bytes
            // Typically 2 bytes per character, may be more for more exotic characters
            if (props.description.length > 512) {
                throw new Error(`Stack description must be <= 1024 bytes. Received description: '${props.description}'`);
            }
            this.templateOptions.description = props.description;
        }
        this._stackName = props.stackName !== undefined ? props.stackName : this.generateStackName();
        this.tags = new tag_manager_1.TagManager(cfn_resource_1.TagType.KEY_VALUE, 'aws:cdk:stack', props.tags);
        if (!VALID_STACK_NAME_REGEX.test(this.stackName)) {
            throw new Error(`Stack name must match the regular expression: ${VALID_STACK_NAME_REGEX.toString()}, got '${this.stackName}'`);
        }
        // the preferred behavior is to generate a unique id for this stack and use
        // it as the artifact ID in the assembly. this allows multiple stacks to use
        // the same name. however, this behavior is breaking for 1.x so it's only
        // applied under a feature flag which is applied automatically for new
        // projects created using `cdk init`.
        //
        // Also use the new behavior if we are using the new CI/CD-ready synthesizer; that way
        // people only have to flip one flag.
        // eslint-disable-next-line max-len
        this.artifactId = this.node.tryGetContext(cxapi.ENABLE_STACK_NAME_DUPLICATES_CONTEXT) || this.node.tryGetContext(cxapi.NEW_STYLE_STACK_SYNTHESIS_CONTEXT)
            ? this.generateStackArtifactId()
            : this.stackName;
        this.templateFile = `${this.artifactId}.template.json`;
        this.synthesizer = (_a = props.synthesizer) !== null && _a !== void 0 ? _a : (this.node.tryGetContext(cxapi.NEW_STYLE_STACK_SYNTHESIS_CONTEXT)
            ? new stack_synthesizers_1.DefaultStackSynthesizer()
            : new stack_synthesizers_1.LegacyStackSynthesizer());
        this.synthesizer.bind(this);
    }
    /**
     * Return whether the given object is a Stack.
     *
     * We do attribute detection since we can't reliably use 'instanceof'.
     */
    static isStack(x) {
        return x !== null && typeof (x) === 'object' && STACK_SYMBOL in x;
    }
    /**
     * Looks up the first stack scope in which `construct` is defined. Fails if there is no stack up the tree.
     * @param construct The construct to start the search from.
     */
    static of(construct) {
        // we want this to be as cheap as possible. cache this result by mutating
        // the object. anecdotally, at the time of this writing, @aws-cdk/core unit
        // tests hit this cache 1,112 times, @aws-cdk/aws-cloudformation unit tests
        // hit this 2,435 times).
        const cache = construct[MY_STACK_CACHE];
        if (cache) {
            return cache;
        }
        else {
            const value = _lookup(construct);
            Object.defineProperty(construct, MY_STACK_CACHE, {
                enumerable: false,
                writable: false,
                configurable: false,
                value,
            });
            return value;
        }
        function _lookup(c) {
            if (Stack.isStack(c)) {
                return c;
            }
            if (!c.node.scope) {
                throw new Error(`No stack could be identified for the construct at path ${construct.node.path}`);
            }
            return _lookup(c.node.scope);
        }
    }
    /**
     * Resolve a tokenized value in the context of the current stack.
     */
    resolve(obj) {
        return resolve_1.resolve(obj, {
            scope: this,
            prefix: [],
            resolver: cloudformation_lang_1.CLOUDFORMATION_TOKEN_RESOLVER,
            preparing: false,
        });
    }
    /**
     * Convert an object, potentially containing tokens, to a JSON string
     */
    toJsonString(obj, space) {
        return cloudformation_lang_1.CloudFormationLang.toJSON(obj, space).toString();
    }
    /**
     * Indicate that a context key was expected
     *
     * Contains instructions which will be emitted into the cloud assembly on how
     * the key should be supplied.
     *
     * @param report The set of parameters needed to obtain the context
     */
    reportMissingContext(report) {
        if (!Object.values(cxschema.ContextProvider).includes(report.provider)) {
            throw new Error(`Unknown context provider requested in: ${JSON.stringify(report)}`);
        }
        this._missingContext.push(report);
    }
    /**
     * Rename a generated logical identities
     *
     * To modify the naming scheme strategy, extend the `Stack` class and
     * override the `allocateLogicalId` method.
     */
    renameLogicalId(oldId, newId) {
        this._logicalIds.addRename(oldId, newId);
    }
    /**
     * Allocates a stack-unique CloudFormation-compatible logical identity for a
     * specific resource.
     *
     * This method is called when a `CfnElement` is created and used to render the
     * initial logical identity of resources. Logical ID renames are applied at
     * this stage.
     *
     * This method uses the protected method `allocateLogicalId` to render the
     * logical ID for an element. To modify the naming scheme, extend the `Stack`
     * class and override this method.
     *
     * @param element The CloudFormation element for which a logical identity is
     * needed.
     */
    getLogicalId(element) {
        const logicalId = this.allocateLogicalId(element);
        return this._logicalIds.applyRename(logicalId);
    }
    /**
     * Add a dependency between this stack and another stack.
     *
     * This can be used to define dependencies between any two stacks within an
     * app, and also supports nested stacks.
     */
    addDependency(target, reason) {
        deps_1.addDependency(this, target, reason);
    }
    /**
     * Return the stacks this stack depends on
     */
    get dependencies() {
        return Object.values(this._stackDependencies).map(x => x.stack);
    }
    /**
     * The concrete CloudFormation physical stack name.
     *
     * This is either the name defined explicitly in the `stackName` prop or
     * allocated based on the stack's location in the construct tree. Stacks that
     * are directly defined under the app use their construct `id` as their stack
     * name. Stacks that are defined deeper within the tree will use a hashed naming
     * scheme based on the construct path to ensure uniqueness.
     *
     * If you wish to obtain the deploy-time AWS::StackName intrinsic,
     * you can use `Aws.stackName` directly.
     */
    get stackName() {
        return this._stackName;
    }
    /**
     * The partition in which this stack is defined
     */
    get partition() {
        // Always return a non-scoped partition intrinsic. These will usually
        // be used to construct an ARN, but there are no cross-partition
        // calls anyway.
        return cfn_pseudo_1.Aws.PARTITION;
    }
    /**
     * The Amazon domain suffix for the region in which this stack is defined
     */
    get urlSuffix() {
        // Since URL Suffix always follows partition, it is unscoped like partition is.
        return cfn_pseudo_1.Aws.URL_SUFFIX;
    }
    /**
     * The ID of the stack
     *
     * @example After resolving, looks like arn:aws:cloudformation:us-west-2:123456789012:stack/teststack/51af3dc0-da77-11e4-872e-1234567db123
     */
    get stackId() {
        return new cfn_pseudo_1.ScopedAws(this).stackId;
    }
    /**
     * Returns the list of notification Amazon Resource Names (ARNs) for the current stack.
     */
    get notificationArns() {
        return new cfn_pseudo_1.ScopedAws(this).notificationArns;
    }
    /**
     * Indicates if this is a nested stack, in which case `parentStack` will include a reference to it's parent.
     */
    get nested() {
        return this.nestedStackResource !== undefined;
    }
    /**
     * Creates an ARN from components.
     *
     * If `partition`, `region` or `account` are not specified, the stack's
     * partition, region and account will be used.
     *
     * If any component is the empty string, an empty string will be inserted
     * into the generated ARN at the location that component corresponds to.
     *
     * The ARN will be formatted as follows:
     *
     *   arn:{partition}:{service}:{region}:{account}:{resource}{sep}}{resource-name}
     *
     * The required ARN pieces that are omitted will be taken from the stack that
     * the 'scope' is attached to. If all ARN pieces are supplied, the supplied scope
     * can be 'undefined'.
     */
    formatArn(components) {
        return arn_1.Arn.format(components, this);
    }
    /**
     * Given an ARN, parses it and returns components.
     *
     * If the ARN is a concrete string, it will be parsed and validated. The
     * separator (`sep`) will be set to '/' if the 6th component includes a '/',
     * in which case, `resource` will be set to the value before the '/' and
     * `resourceName` will be the rest. In case there is no '/', `resource` will
     * be set to the 6th components and `resourceName` will be set to the rest
     * of the string.
     *
     * If the ARN includes tokens (or is a token), the ARN cannot be validated,
     * since we don't have the actual value yet at the time of this function
     * call. You will have to know the separator and the type of ARN. The
     * resulting `ArnComponents` object will contain tokens for the
     * subexpressions of the ARN, not string literals. In this case this
     * function cannot properly parse the complete final resourceName (path) out
     * of ARNs that use '/' to both separate the 'resource' from the
     * 'resourceName' AND to subdivide the resourceName further. For example, in
     * S3 ARNs:
     *
     *    arn:aws:s3:::my_corporate_bucket/path/to/exampleobject.png
     *
     * After parsing the resourceName will not contain
     * 'path/to/exampleobject.png' but simply 'path'. This is a limitation
     * because there is no slicing functionality in CloudFormation templates.
     *
     * @param arn The ARN string to parse
     * @param sepIfToken The separator used to separate resource from resourceName
     * @param hasName Whether there is a name component in the ARN at all. For
     * example, SNS Topics ARNs have the 'resource' component contain the topic
     * name, and no 'resourceName' component.
     *
     * @returns an ArnComponents object which allows access to the various
     * components of the ARN.
     *
     * @returns an ArnComponents object which allows access to the various
     *      components of the ARN.
     */
    parseArn(arn, sepIfToken = '/', hasName = true) {
        return arn_1.Arn.parse(arn, sepIfToken, hasName);
    }
    /**
     * Returns the list of AZs that are available in the AWS environment
     * (account/region) associated with this stack.
     *
     * If the stack is environment-agnostic (either account and/or region are
     * tokens), this property will return an array with 2 tokens that will resolve
     * at deploy-time to the first two availability zones returned from CloudFormation's
     * `Fn::GetAZs` intrinsic function.
     *
     * If they are not available in the context, returns a set of dummy values and
     * reports them as missing, and let the CLI resolve them by calling EC2
     * `DescribeAvailabilityZones` on the target environment.
     *
     * To specify a different strategy for selecting availability zones override this method.
     */
    get availabilityZones() {
        // if account/region are tokens, we can't obtain AZs through the context
        // provider, so we fallback to use Fn::GetAZs. the current lowest common
        // denominator is 2 AZs across all AWS regions.
        const agnostic = token_1.Token.isUnresolved(this.account) || token_1.Token.isUnresolved(this.region);
        if (agnostic) {
            return this.node.tryGetContext(cxapi.AVAILABILITY_ZONE_FALLBACK_CONTEXT_KEY) || [
                cfn_fn_1.Fn.select(0, cfn_fn_1.Fn.getAzs()),
                cfn_fn_1.Fn.select(1, cfn_fn_1.Fn.getAzs()),
            ];
        }
        const value = context_provider_1.ContextProvider.getValue(this, {
            provider: cxschema.ContextProvider.AVAILABILITY_ZONE_PROVIDER,
            dummyValue: ['dummy1a', 'dummy1b', 'dummy1c'],
        }).value;
        if (!Array.isArray(value)) {
            throw new Error(`Provider ${cxschema.ContextProvider.AVAILABILITY_ZONE_PROVIDER} expects a list`);
        }
        return value;
    }
    /**
     * Register a file asset on this Stack
     *
     * @deprecated Use `stack.synthesizer.addFileAsset()` if you are calling,
     * and a different IDeploymentEnvironment class if you are implementing.
     */
    addFileAsset(asset) {
        return this.synthesizer.addFileAsset(asset);
    }
    /**
     * Register a docker image asset on this Stack
     *
     * @deprecated Use `stack.synthesizer.addDockerImageAsset()` if you are calling,
     * and a different `IDeploymentEnvironment` class if you are implementing.
     */
    addDockerImageAsset(asset) {
        return this.synthesizer.addDockerImageAsset(asset);
    }
    /**
     * If this is a nested stack, returns it's parent stack.
     */
    get nestedStackParent() {
        return this.nestedStackResource && Stack.of(this.nestedStackResource);
    }
    /**
     * Returns the parent of a nested stack.
     *
     * @deprecated use `nestedStackParent`
     */
    get parentStack() {
        return this.nestedStackParent;
    }
    /**
     * Add a Transform to this stack. A Transform is a macro that AWS
     * CloudFormation uses to process your template.
     *
     * Duplicate values are removed when stack is synthesized.
     *
     * @example addTransform('AWS::Serverless-2016-10-31')
     *
     * @see https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/transform-section-structure.html
     *
     * @param transform The transform to add
     */
    addTransform(transform) {
        if (!this.templateOptions.transforms) {
            this.templateOptions.transforms = [];
        }
        this.templateOptions.transforms.push(transform);
    }
    /**
     * Called implicitly by the `addDependency` helper function in order to
     * realize a dependency between two top-level stacks at the assembly level.
     *
     * Use `stack.addDependency` to define the dependency between any two stacks,
     * and take into account nested stack relationships.
     *
     * @internal
     */
    _addAssemblyDependency(target, reason) {
        // defensive: we should never get here for nested stacks
        if (this.nested || target.nested) {
            throw new Error('Cannot add assembly-level dependencies for nested stacks');
        }
        reason = reason || 'dependency added using stack.addDependency()';
        const cycle = target.stackDependencyReasons(this);
        if (cycle !== undefined) {
            // eslint-disable-next-line max-len
            throw new Error(`'${target.node.path}' depends on '${this.node.path}' (${cycle.join(', ')}). Adding this dependency (${reason}) would create a cyclic reference.`);
        }
        let dep = this._stackDependencies[target.node.uniqueId];
        if (!dep) {
            dep = this._stackDependencies[target.node.uniqueId] = {
                stack: target,
                reasons: [],
            };
        }
        dep.reasons.push(reason);
        if (process.env.CDK_DEBUG_DEPS) {
            // eslint-disable-next-line no-console
            console.error(`[CDK_DEBUG_DEPS] stack "${this.node.path}" depends on "${target.node.path}" because: ${reason}`);
        }
    }
    /**
     * Synthesizes the cloudformation template into a cloud assembly.
     * @internal
     */
    _synthesizeTemplate(session) {
        // In principle, stack synthesis is delegated to the
        // StackSynthesis object.
        //
        // However, some parts of synthesis currently use some private
        // methods on Stack, and I don't really see the value in refactoring
        // this right now, so some parts still happen here.
        const builder = session.assembly;
        // write the CloudFormation template as a JSON file
        const outPath = path.join(builder.outdir, this.templateFile);
        const text = JSON.stringify(this._toCloudFormation(), undefined, 2);
        fs.writeFileSync(outPath, text);
        for (const ctx of this._missingContext) {
            builder.addMissing(ctx);
        }
        // Delegate adding artifacts to the Synthesizer
        this.synthesizer.synthesizeStackArtifacts(session);
    }
    /**
     * Returns the naming scheme used to allocate logical IDs. By default, uses
     * the `HashedAddressingScheme` but this method can be overridden to customize
     * this behavior.
     *
     * In order to make sure logical IDs are unique and stable, we hash the resource
     * construct tree path (i.e. toplevel/secondlevel/.../myresource) and add it as
     * a suffix to the path components joined without a separator (CloudFormation
     * IDs only allow alphanumeric characters).
     *
     * The result will be:
     *
     *   <path.join('')><md5(path.join('/')>
     *     "human"      "hash"
     *
     * If the "human" part of the ID exceeds 240 characters, we simply trim it so
     * the total ID doesn't exceed CloudFormation's 255 character limit.
     *
     * We only take 8 characters from the md5 hash (0.000005 chance of collision).
     *
     * Special cases:
     *
     * - If the path only contains a single component (i.e. it's a top-level
     *   resource), we won't add the hash to it. The hash is not needed for
     *   disamiguation and also, it allows for a more straightforward migration an
     *   existing CloudFormation template to a CDK stack without logical ID changes
     *   (or renames).
     * - For aesthetic reasons, if the last components of the path are the same
     *   (i.e. `L1/L2/Pipeline/Pipeline`), they will be de-duplicated to make the
     *   resulting human portion of the ID more pleasing: `L1L2Pipeline<HASH>`
     *   instead of `L1L2PipelinePipeline<HASH>`
     * - If a component is named "Default" it will be omitted from the path. This
     *   allows refactoring higher level abstractions around constructs without affecting
     *   the IDs of already deployed resources.
     * - If a component is named "Resource" it will be omitted from the user-visible
     *   path, but included in the hash. This reduces visual noise in the human readable
     *   part of the identifier.
     *
     * @param cfnElement The element for which the logical ID is allocated.
     */
    allocateLogicalId(cfnElement) {
        const scopes = cfnElement.node.scopes;
        const stackIndex = scopes.indexOf(cfnElement.stack);
        const pathComponents = scopes.slice(stackIndex + 1).map(x => x.node.id);
        return uniqueid_1.makeUniqueId(pathComponents);
    }
    /**
     * Validate stack name
     *
     * CloudFormation stack names can include dashes in addition to the regular identifier
     * character classes, and we don't allow one of the magic markers.
     *
     * @internal
     */
    _validateId(name) {
        if (name && !VALID_STACK_NAME_REGEX.test(name)) {
            throw new Error(`Stack name must match the regular expression: ${VALID_STACK_NAME_REGEX.toString()}, got '${name}'`);
        }
    }
    /**
     * Returns the CloudFormation template for this stack by traversing
     * the tree and invoking _toCloudFormation() on all Entity objects.
     *
     * @internal
     */
    _toCloudFormation() {
        let transform;
        if (this.templateOptions.transform) {
            // eslint-disable-next-line max-len
            annotations_1.Annotations.of(this).addWarning('This stack is using the deprecated `templateOptions.transform` property. Consider switching to `addTransform()`.');
            this.addTransform(this.templateOptions.transform);
        }
        if (this.templateOptions.transforms) {
            if (this.templateOptions.transforms.length === 1) { // Extract single value
                transform = this.templateOptions.transforms[0];
            }
            else { // Remove duplicate values
                transform = Array.from(new Set(this.templateOptions.transforms));
            }
        }
        const template = {
            Description: this.templateOptions.description,
            Transform: transform,
            AWSTemplateFormatVersion: this.templateOptions.templateFormatVersion,
            Metadata: this.templateOptions.metadata,
        };
        const elements = cfnElements(this);
        const fragments = elements.map(e => this.resolve(e._toCloudFormation()));
        // merge in all CloudFormation fragments collected from the tree
        for (const fragment of fragments) {
            merge(template, fragment);
        }
        // resolve all tokens and remove all empties
        const ret = this.resolve(template) || {};
        this._logicalIds.assertAllRenamesApplied();
        return ret;
    }
    /**
     * Deprecated.
     *
     * @see https://github.com/aws/aws-cdk/pull/7187
     * @returns reference itself without any change
     * @deprecated cross reference handling has been moved to `App.prepare()`.
     */
    prepareCrossReference(_sourceStack, reference) {
        return reference;
    }
    /**
     * Determine the various stack environment attributes.
     *
     */
    parseEnvironment(env = {}) {
        var _a, _b, _c, _d;
        // if an environment property is explicitly specified when the stack is
        // created, it will be used. if not, use tokens for account and region.
        //
        // (They do not need to be anchored to any construct like resource attributes
        // are, because we'll never Export/Fn::ImportValue them -- the only situation
        // in which Export/Fn::ImportValue would work is if the value are the same
        // between producer and consumer anyway, so we can just assume that they are).
        const containingAssembly = stage_1.Stage.of(this);
        const account = (_b = (_a = env.account) !== null && _a !== void 0 ? _a : containingAssembly === null || containingAssembly === void 0 ? void 0 : containingAssembly.account) !== null && _b !== void 0 ? _b : cfn_pseudo_1.Aws.ACCOUNT_ID;
        const region = (_d = (_c = env.region) !== null && _c !== void 0 ? _c : containingAssembly === null || containingAssembly === void 0 ? void 0 : containingAssembly.region) !== null && _d !== void 0 ? _d : cfn_pseudo_1.Aws.REGION;
        // this is the "aws://" env specification that will be written to the cloud assembly
        // manifest. it will use "unknown-account" and "unknown-region" to indicate
        // environment-agnosticness.
        const envAccount = !token_1.Token.isUnresolved(account) ? account : cxapi.UNKNOWN_ACCOUNT;
        const envRegion = !token_1.Token.isUnresolved(region) ? region : cxapi.UNKNOWN_REGION;
        return {
            account, region,
            environment: cxapi.EnvironmentUtils.format(envAccount, envRegion),
        };
    }
    /**
     * Check whether this stack has a (transitive) dependency on another stack
     *
     * Returns the list of reasons on the dependency path, or undefined
     * if there is no dependency.
     */
    stackDependencyReasons(other) {
        if (this === other) {
            return [];
        }
        for (const dep of Object.values(this._stackDependencies)) {
            const ret = dep.stack.stackDependencyReasons(other);
            if (ret !== undefined) {
                return [...dep.reasons, ...ret];
            }
        }
        return undefined;
    }
    /**
     * Calculate the stack name based on the construct path
     *
     * The stack name is the name under which we'll deploy the stack,
     * and incorporates containing Stage names by default.
     *
     * Generally this looks a lot like how logical IDs are calculated.
     * The stack name is calculated based on the construct root path,
     * as follows:
     *
     * - Path is calculated with respect to containing App or Stage (if any)
     * - If the path is one component long just use that component, otherwise
     *   combine them with a hash.
     *
     * Since the hash is quite ugly and we'd like to avoid it if possible -- but
     * we can't anymore in the general case since it has been written into legacy
     * stacks. The introduction of Stages makes it possible to make this nicer however.
     * When a Stack is nested inside a Stage, we use the path components below the
     * Stage, and prefix the path components of the Stage before it.
     */
    generateStackName() {
        const assembly = stage_1.Stage.of(this);
        const prefix = (assembly && assembly.stageName) ? `${assembly.stageName}-` : '';
        return `${prefix}${this.generateStackId(assembly)}`;
    }
    /**
     * The artifact ID for this stack
     *
     * Stack artifact ID is unique within the App's Cloud Assembly.
     */
    generateStackArtifactId() {
        return this.generateStackId(this.node.root);
    }
    /**
     * Generate an ID with respect to the given container construct.
     */
    generateStackId(container) {
        const rootPath = rootPathTo(this, container);
        const ids = rootPath.map(c => c.node.id);
        // In unit tests our Stack (which is the only component) may not have an
        // id, so in that case just pretend it's "Stack".
        if (ids.length === 1 && !ids[0]) {
            throw new Error('unexpected: stack id must always be defined');
        }
        return makeStackName(ids);
    }
}
exports.Stack = Stack;
function merge(template, fragment) {
    for (const section of Object.keys(fragment)) {
        const src = fragment[section];
        // create top-level section if it doesn't exist
        const dest = template[section];
        if (!dest) {
            template[section] = src;
        }
        else {
            template[section] = mergeSection(section, dest, src);
        }
    }
}
function mergeSection(section, val1, val2) {
    switch (section) {
        case 'Description':
            return `${val1}\n${val2}`;
        case 'AWSTemplateFormatVersion':
            if (val1 != null && val2 != null && val1 !== val2) {
                throw new Error(`Conflicting CloudFormation template versions provided: '${val1}' and '${val2}`);
            }
            return val1 !== null && val1 !== void 0 ? val1 : val2;
        case 'Transform':
            return mergeSets(val1, val2);
        default:
            return mergeObjectsWithoutDuplicates(section, val1, val2);
    }
}
function mergeSets(val1, val2) {
    const array1 = val1 == null ? [] : (Array.isArray(val1) ? val1 : [val1]);
    const array2 = val2 == null ? [] : (Array.isArray(val2) ? val2 : [val2]);
    for (const value of array2) {
        if (!array1.includes(value)) {
            array1.push(value);
        }
    }
    return array1.length === 1 ? array1[0] : array1;
}
function mergeObjectsWithoutDuplicates(section, dest, src) {
    if (typeof dest !== 'object') {
        throw new Error(`Expecting ${JSON.stringify(dest)} to be an object`);
    }
    if (typeof src !== 'object') {
        throw new Error(`Expecting ${JSON.stringify(src)} to be an object`);
    }
    // add all entities from source section to destination section
    for (const id of Object.keys(src)) {
        if (id in dest) {
            throw new Error(`section '${section}' already contains '${id}'`);
        }
        dest[id] = src[id];
    }
    return dest;
}
/**
 * Collect all CfnElements from a Stack.
 *
 * @param node Root node to collect all CfnElements from
 * @param into Array to append CfnElements to
 * @returns The same array as is being collected into
 */
function cfnElements(node, into = []) {
    if (cfn_element_1.CfnElement.isCfnElement(node)) {
        into.push(node);
    }
    for (const child of node.node.children) {
        // Don't recurse into a substack
        if (Stack.isStack(child)) {
            continue;
        }
        cfnElements(child, into);
    }
    return into;
}
/**
 * Return the construct root path of the given construct relative to the given ancestor
 *
 * If no ancestor is given or the ancestor is not found, return the entire root path.
 */
function rootPathTo(construct, ancestor) {
    const scopes = construct.node.scopes;
    for (let i = scopes.length - 2; i >= 0; i--) {
        if (scopes[i] === ancestor) {
            return scopes.slice(i + 1);
        }
    }
    return scopes;
}
exports.rootPathTo = rootPathTo;
/**
 * makeUniqueId, specialized for Stack names
 *
 * Stack names may contain '-', so we allow that character if the stack name
 * has only one component. Otherwise we fall back to the regular "makeUniqueId"
 * behavior.
 */
function makeStackName(components) {
    if (components.length === 1) {
        return components[0];
    }
    return uniqueid_1.makeUniqueId(components);
}
// These imports have to be at the end to prevent circular imports
const deps_1 = require("./deps");
const stack_synthesizers_1 = require("./stack-synthesizers");
const stage_1 = require("./stage");
const tag_manager_1 = require("./tag-manager");
const token_1 = require("./token");
const fs_1 = require("./fs");
//# sourceMappingURL=data:application/json;base64,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