"use strict";
const cxapi = require("@aws-cdk/cx-api");
const lib_1 = require("../lib");
class TouchingAspect {
    constructor() {
        this.visits = new Array();
    }
    visit(node) {
        this.visits.push(node);
    }
}
class BogusStack extends lib_1.Stack {
    constructor(scope, id) {
        super(scope, id);
        new lib_1.CfnResource(this, 'Resource', {
            type: 'CDK::Test::Resource',
        });
    }
}
function acctRegion(s) {
    return [s.account, s.region];
}
module.exports = {
    'Stack inherits unspecified part of the env from Stage'(test) {
        // GIVEN
        const app = new lib_1.App();
        const stage = new lib_1.Stage(app, 'Stage', {
            env: { account: 'account', region: 'region' },
        });
        // WHEN
        const stack1 = new lib_1.Stack(stage, 'Stack1', { env: { region: 'elsewhere' } });
        const stack2 = new lib_1.Stack(stage, 'Stack2', { env: { account: 'tnuocca' } });
        // THEN
        test.deepEqual(acctRegion(stack1), ['account', 'elsewhere']);
        test.deepEqual(acctRegion(stack2), ['tnuocca', 'region']);
        test.done();
    },
    'envs are inherited deeply'(test) {
        // GIVEN
        const app = new lib_1.App();
        const outer = new lib_1.Stage(app, 'Stage', {
            env: { account: 'account', region: 'region' },
        });
        // WHEN
        const innerAcct = new lib_1.Stage(outer, 'Acct', { env: { account: 'tnuocca' } });
        const innerRegion = new lib_1.Stage(outer, 'Rgn', { env: { region: 'elsewhere' } });
        const innerNeither = new lib_1.Stage(outer, 'Neither');
        // THEN
        test.deepEqual(acctRegion(new lib_1.Stack(innerAcct, 'Stack')), ['tnuocca', 'region']);
        test.deepEqual(acctRegion(new lib_1.Stack(innerRegion, 'Stack')), ['account', 'elsewhere']);
        test.deepEqual(acctRegion(new lib_1.Stack(innerNeither, 'Stack')), ['account', 'region']);
        test.done();
    },
    'The Stage Assembly is in the app Assembly\'s manifest'(test) {
        // WHEN
        const app = new lib_1.App();
        const stage = new lib_1.Stage(app, 'Stage');
        new BogusStack(stage, 'Stack2');
        // THEN -- app manifest contains a nested cloud assembly
        const appAsm = app.synth();
        const artifact = appAsm.artifacts.find(x => x instanceof cxapi.NestedCloudAssemblyArtifact);
        test.ok(artifact);
        test.done();
    },
    'Stacks in Stage are in a different cxasm than Stacks in App'(test) {
        // WHEN
        const app = new lib_1.App();
        const stack1 = new BogusStack(app, 'Stack1');
        const stage = new lib_1.Stage(app, 'Stage');
        const stack2 = new BogusStack(stage, 'Stack2');
        // THEN
        const stageAsm = stage.synth();
        test.deepEqual(stageAsm.stacks.map(s => s.stackName), [stack2.stackName]);
        const appAsm = app.synth();
        test.deepEqual(appAsm.stacks.map(s => s.stackName), [stack1.stackName]);
        test.done();
    },
    'Can nest Stages inside other Stages'(test) {
        // WHEN
        const app = new lib_1.App();
        const outer = new lib_1.Stage(app, 'Outer');
        const inner = new lib_1.Stage(outer, 'Inner');
        const stack = new BogusStack(inner, 'Stack');
        // WHEN
        const appAsm = app.synth();
        const outerAsm = appAsm.getNestedAssembly(outer.artifactId);
        const innerAsm = outerAsm.getNestedAssembly(inner.artifactId);
        test.ok(innerAsm.tryGetArtifact(stack.artifactId));
        test.done();
    },
    'Default stack name in Stage objects incorporates the Stage name and no hash'(test) {
        // WHEN
        const app = new lib_1.App();
        const stage = new lib_1.Stage(app, 'MyStage');
        const stack = new BogusStack(stage, 'MyStack');
        // THEN
        test.equal(stage.stageName, 'MyStage');
        test.equal(stack.stackName, 'MyStage-MyStack');
        test.done();
    },
    'Can not have dependencies to stacks outside the nested asm'(test) {
        // GIVEN
        const app = new lib_1.App();
        const stack1 = new BogusStack(app, 'Stack1');
        const stage = new lib_1.Stage(app, 'MyStage');
        const stack2 = new BogusStack(stage, 'Stack2');
        // WHEN
        test.throws(() => {
            stack2.addDependency(stack1);
        }, /dependency cannot cross stage boundaries/);
        test.done();
    },
    'When we synth() a stage, prepare must be called on constructs in the stage'(test) {
        // GIVEN
        const app = new lib_1.App();
        let prepared = false;
        const stage = new lib_1.Stage(app, 'MyStage');
        const stack = new BogusStack(stage, 'Stack');
        class HazPrepare extends lib_1.Construct {
            prepare() {
                prepared = true;
            }
        }
        new HazPrepare(stack, 'Preparable');
        // WHEN
        stage.synth();
        // THEN
        test.equals(prepared, true);
        test.done();
    },
    'When we synth() a stage, aspects inside it must have been applied'(test) {
        // GIVEN
        const app = new lib_1.App();
        const stage = new lib_1.Stage(app, 'MyStage');
        const stack = new BogusStack(stage, 'Stack');
        // WHEN
        const aspect = new TouchingAspect();
        lib_1.Aspects.of(stack).add(aspect);
        // THEN
        app.synth();
        test.deepEqual(aspect.visits.map(c => c.node.path), [
            'MyStage/Stack',
            'MyStage/Stack/Resource',
        ]);
        test.done();
    },
    'Aspects do not apply inside a Stage'(test) {
        // GIVEN
        const app = new lib_1.App();
        const stage = new lib_1.Stage(app, 'MyStage');
        new BogusStack(stage, 'Stack');
        // WHEN
        const aspect = new TouchingAspect();
        lib_1.Aspects.of(app).add(aspect);
        // THEN
        app.synth();
        test.deepEqual(aspect.visits.map(c => c.node.path), [
            '',
            'Tree',
        ]);
        test.done();
    },
    'Automatic dependencies inside a stage are available immediately after synth'(test) {
        // GIVEN
        const app = new lib_1.App();
        const stage = new lib_1.Stage(app, 'MyStage');
        const stack1 = new lib_1.Stack(stage, 'Stack1');
        const stack2 = new lib_1.Stack(stage, 'Stack2');
        // WHEN
        const resource1 = new lib_1.CfnResource(stack1, 'Resource', {
            type: 'CDK::Test::Resource',
        });
        new lib_1.CfnResource(stack2, 'Resource', {
            type: 'CDK::Test::Resource',
            properties: {
                OtherThing: resource1.ref,
            },
        });
        const asm = stage.synth();
        // THEN
        test.deepEqual(asm.getStackArtifact(stack2.artifactId).dependencies.map(d => d.id), [stack1.artifactId]);
        test.done();
    },
    'Assemblies can be deeply nested'(test) {
        // GIVEN
        const app = new lib_1.App({ runtimeInfo: false, treeMetadata: false });
        const level1 = new lib_1.Stage(app, 'StageLevel1');
        const level2 = new lib_1.Stage(level1, 'StageLevel2');
        new lib_1.Stage(level2, 'StageLevel3');
        // WHEN
        const rootAssembly = app.synth();
        // THEN
        test.deepEqual(rootAssembly.manifest.artifacts, {
            'assembly-StageLevel1': {
                type: 'cdk:cloud-assembly',
                properties: {
                    directoryName: 'assembly-StageLevel1',
                    displayName: 'StageLevel1',
                },
            },
        });
        const assemblyLevel1 = rootAssembly.getNestedAssembly('assembly-StageLevel1');
        test.deepEqual(assemblyLevel1.manifest.artifacts, {
            'assembly-StageLevel1-StageLevel2': {
                type: 'cdk:cloud-assembly',
                properties: {
                    directoryName: 'assembly-StageLevel1-StageLevel2',
                    displayName: 'StageLevel1/StageLevel2',
                },
            },
        });
        const assemblyLevel2 = assemblyLevel1.getNestedAssembly('assembly-StageLevel1-StageLevel2');
        test.deepEqual(assemblyLevel2.manifest.artifacts, {
            'assembly-StageLevel1-StageLevel2-StageLevel3': {
                type: 'cdk:cloud-assembly',
                properties: {
                    directoryName: 'assembly-StageLevel1-StageLevel2-StageLevel3',
                    displayName: 'StageLevel1/StageLevel2/StageLevel3',
                },
            },
        });
        test.done();
    },
    'stage name validation'(test) {
        const app = new lib_1.App();
        new lib_1.Stage(app, 'abcd');
        new lib_1.Stage(app, 'abcd123');
        new lib_1.Stage(app, 'abcd123-588dfjjk');
        new lib_1.Stage(app, 'abcd123-588dfjjk.sss');
        new lib_1.Stage(app, 'abcd123-588dfjjk.sss_ajsid');
        test.throws(() => new lib_1.Stage(app, 'abcd123-588dfjjk.sss_ajsid '), /invalid stage name "abcd123-588dfjjk.sss_ajsid "/);
        test.throws(() => new lib_1.Stage(app, 'abcd123-588dfjjk.sss_ajsid/dfo'), /invalid stage name "abcd123-588dfjjk.sss_ajsid\/dfo"/);
        test.throws(() => new lib_1.Stage(app, '&'), /invalid stage name "&"/);
        test.throws(() => new lib_1.Stage(app, '45hello'), /invalid stage name "45hello"/);
        test.throws(() => new lib_1.Stage(app, 'f'), /invalid stage name "f"/);
        test.done();
    },
    'outdir cannot be specified for nested stages'(test) {
        // WHEN
        const app = new lib_1.App();
        // THEN
        test.throws(() => new lib_1.Stage(app, 'mystage', { outdir: '/tmp/foo/bar' }), /"outdir" cannot be specified for nested stages/);
        test.done();
    },
};
//# sourceMappingURL=data:application/json;base64,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