from veroviz._common import *

from veroviz._getSnapLoc import privGetSnapLocBatch

from veroviz._internal import locs2Dict
from veroviz._internal import loc2Dict
from veroviz._internal import stripCesiumColor

from veroviz._utilities import privInitDataframe

def privCreateNodesFromLocs(locs=None, initNodes=None, nodeType=None, nodeName=None, startNode=1, incrementName=False, incrementStart=1, snapToRoad=False, dataProvider=None, dataProviderArgs=None, popupText=None, leafletIconPrefix=config['VRV_DEFAULT_LEAFLETICONPREFIX'], leafletIconType=config['VRV_DEFAULT_LEAFLETICONTYPE'], leafletColor=config['VRV_DEFAULT_LEAFLETICONCOLOR'], leafletIconText=None, cesiumIconType=config['VRV_DEFAULT_CESIUMICONTYPE'], cesiumColor=config['VRV_DEFAULT_CESIUMICONCOLOR'], cesiumIconText=None):

	"""
	Given a set of nodes lats and lons, return a node dataframe

	Parameters
	----------
	locs: list of lists, Required, default as None
		A list of locations, in the form of [[lat, lon, alt], [lat, lon, alt], ...] or [[lat, lon], [lat, lon], ...]
	initNodes: :ref:`Nodes`, Optional, default as None
		A dataframe containing an existing set of nodes. If `initNodes` is provided, this function will extend to that dataframe.
	nodeType: string, Optional, default as None
		A user-defined text field that can be used to classify nodes. This field is to categorize a batch of nodes (e.g., "restaurants"). If provided, all nodes generated by the `generateNodes()` function call will be given this value. The nodeType is not used by VeRoViz explicitly. 
	nodeName: string, Optional, default as None
		The name of all nodes that are to be generated by this function call. This field is a more detailed description (e.g., "pizza" or "steakhouse"). The nodeName is not used by VeRoViz explicitly. 
	startNode: int, Optional, default as 1
		The starting node number will be the maximum of startNode and any id values contained in the initNodes dataframe (if provided).  
	incrementName: boolean, Optional, default as False
		Toggle to choose if we add increment after nodeName, e.g. 'customer1', 'customer2',...
	incrementStart: int, Optional, default as 1
		The starting number of the increment.
	popupText: string, Optional, default as None
		Text (or HTML) that will be displayed when a user clicks on the node in either Leaflet or Cesium.  A value of None will result in the node ID being auto-populated in the `popupText` column of the nodes dataframe.
	leafletIconPrefix: string, Optional, default as "glyphicon"
		The collection of Leaflet icons.  Options are "glyphicon", "fa", or "custom". See :ref:`Leaflet style`
	leafletIconType: string, Optional, default as "info-sign"
		The specific icon to be used for all generated nodes.  The list of available options depends on the choice of the leafletIconType. See :ref:`Leaflet style`
	leafletColor: string, Optional, default as "blue"
		The icon color of the generated nodes when displayed in Leaflet. One of a collection of pre-specified colors. See :ref:`Leaflet style`
	leafletIconText: string, Optional, default as None
		Text that will be displayed within the node on a Leaflet map.  This text will only be shown if `leafletIconPrefix` is 'custom' and `leafletIconType` includes a font color and font size.  A value of None will result in the node ID being displayed in the node.  See :ref:`Leaflet style`.
	cesiumIconType: string, Optional, default as "pin"
		'pin' is the only option right now. See :ref:`Cesium style`
	cesiumColor: string, Optional, default as "blue"
		The color of the generated nodes when displayed in Cesium.  One of a collection of pre-specified colors. See :ref:`Cesium style`
	cesiumIconText: string, Optional, default as None
		Text that will be displayed within the node on a Cesium map. See :ref:`Cesium style`

	Return
	------
	:ref:`Nodes`
		A Nodes dataframe with given list of coordinates
	"""

	# Number of nodes
	numNodes = len(locs)

	# Define ids and nodeNames
	if (type(initNodes) is pd.core.frame.DataFrame):
		if (len(initNodes) > 0):
			maxID = max(initNodes['id'])
			startNode = max([maxID + 1, startNode])

	ids = [n for n in range(startNode, startNode + numNodes)]
	if (incrementName):
		nodeNames = [(nodeName + "%s" % (n)) for n in range(incrementStart, incrementStart + numNodes)]
	else:
		nodeNames = [nodeName] * numNodes

	# Snap to road
	# FIXME! - Issue #28 - Multiple nodes might be snapped to the same location 
	if (snapToRoad):
		locs = privGetSnapLocBatch(locs=locs, dataProvider=dataProvider, dataProviderArgs=dataProviderArgs)

	# node dataframe
	nodes = privInitDataframe('Nodes')

	# generate nodes
	dicLocs = locs2Dict(locs)
	for i in range(len(locs)):
		nodes = nodes.append({
			'id': ids[i],
			'lat': dicLocs[i]['lat'],
			'lon': dicLocs[i]['lon'],
			'altMeters': dicLocs[i]['alt'],
			'nodeName': nodeNames[i],
			'nodeType': nodeType,
			'popupText': popupText if (popupText != None) else ids[i],
			'leafletIconPrefix': leafletIconPrefix,
			'leafletIconType': leafletIconType,
			'leafletColor': leafletColor,
			'leafletIconText': leafletIconText if (leafletIconText != None) else ids[i],
			'cesiumIconType': cesiumIconType,
			'cesiumColor': stripCesiumColor(cesiumColor),
			'cesiumIconText': cesiumIconText if (cesiumIconText != None) else ids[i],
			'elevMeters': None
			}, ignore_index=True)

	# if the user provided an initNode dataframe, add the new points after it
	if (type(initNodes) is pd.core.frame.DataFrame):
		nodes = pd.concat([initNodes, nodes], ignore_index=True)

	return nodes

def privCreateArcsFromLocSeq(locSeq=None, initArcs=None, startArc=1, objectID=None, leafletColor=config['VRV_DEFAULT_LEAFLETARCCOLOR'], leafletWeight=config['VRV_DEFAULT_LEAFLETARCWEIGHT'], leafletStyle=config['VRV_DEFAULT_LEAFLETARCSTYLE'], leafletOpacity=config['VRV_DEFAULT_LEAFLETARCOPACITY'],  leafletCurveType=config['VRV_DEFAULT_ARCCURVETYPE'], leafletCurvature=config['VRV_DEFAULT_ARCCURVATURE'], useArrows=True, cesiumColor=config['VRV_DEFAULT_CESIUMPATHCOLOR'], cesiumWeight=config['VRV_DEFAULT_CESIUMPATHWEIGHT'], cesiumStyle=config['VRV_DEFAULT_CESIUMPATHSTYLE'], cesiumOpacity=config['VRV_DEFAULT_CESIUMPATHOPACITY'], popupText=None):

	"""
	See createArcsFromLocSeq for docstring
	"""


	# Number of arcs
	numArcs = len(locSeq)

	# Define odIDs
	if (type(initArcs) is pd.core.frame.DataFrame):
		if (len(initArcs) > 0):
			maxOdID = max(initArcs['odID'])
			startArc = max(maxOdID + 1, startArc)
	odIDs = [n for n in range(startArc, startArc + numArcs)]

	# arc dataframe
	arcs = privInitDataframe('Arcs')

	# generate arcs
	for i in range(len(locSeq) - 1):
		arcs = arcs.append({
			'odID': odIDs[i],
			'objectID': objectID,
			'startLat': locSeq[i][0],
			'startLon': locSeq[i][1],
			'endLat': locSeq[i + 1][0],
			'endLon': locSeq[i + 1][1],
			'leafletColor' : leafletColor,
			'leafletWeight' : leafletWeight,
			'leafletStyle' : leafletStyle,
			'leafletOpacity' : leafletOpacity,
			'leafletCurveType' : leafletCurveType,
			'leafletCurvature' : leafletCurvature,
			'cesiumColor' : stripCesiumColor(cesiumColor),
			'cesiumWeight' : cesiumWeight,
			'cesiumStyle' : cesiumStyle,
			'cesiumOpacity' : cesiumOpacity,
			'useArrows': useArrows,
			'popupText': popupText, 
			'startElevMeters': None,
			'endElevMeters': None
			}, ignore_index=True)

	# if the user provided an initNode dataframe, add the new points after it
	if (type(initArcs) is pd.core.frame.DataFrame):
		arcs = pd.concat([initArcs, arcs], ignore_index=True)

	return arcs