"""
Claude Monitor Integration Service.
Uses the claude_monitor package as a library to provide real usage statistics.
"""

import logging
from datetime import datetime, timedelta
from pathlib import Path
from typing import Dict, List, Any, Optional
from dataclasses import dataclass, asdict

# Import claude_monitor components
from claude_monitor.data.reader import load_usage_entries
from claude_monitor.data.aggregator import UsageAggregator
from claude_monitor.core.models import CostMode, UsageEntry, BurnRate, UsageProjection
from claude_monitor.utils.time_utils import TimezoneHandler
from claude_monitor.core.plans import Plans, PlanType, get_token_limit, get_cost_limit
from claude_monitor.core.calculations import calculate_hourly_burn_rate, BurnRateCalculator

logger = logging.getLogger(__name__)


@dataclass
class UsageStats:
    """Aggregated usage statistics compatible with existing cuti interface."""
    total_tokens: int
    total_cost: float
    total_requests: int
    input_tokens: int
    output_tokens: int
    cache_creation_tokens: int
    cache_read_tokens: int
    tokens_today: int
    cost_today: float
    requests_today: int
    tokens_this_month: int
    cost_this_month: float
    requests_this_month: int
    avg_tokens_per_request: float
    success_rate: float


@dataclass
class DailyUsage:
    """Daily usage data."""
    date: str
    input_tokens: int
    output_tokens: int
    total_tokens: int
    cost: float
    requests: int
    models_used: List[str]


class ClaudeMonitorIntegration:
    """Integration service for claude_monitor package."""
    
    def __init__(self, claude_data_path: Optional[str] = None, plan_type: str = 'pro'):
        """
        Initialize the integration service.
        
        Args:
            claude_data_path: Path to Claude data directory (defaults to ~/.claude/projects)
            plan_type: Claude subscription plan ('pro', 'max5', 'max20', 'custom')
        """
        self.claude_data_path = claude_data_path or "~/.claude/projects"
        self.timezone_handler = TimezoneHandler()
        self.burn_rate_calculator = BurnRateCalculator()
        self.current_plan = plan_type.lower()
        
        # Verify data path exists
        data_path = Path(self.claude_data_path).expanduser()
        if not data_path.exists():
            logger.warning(f"Claude data path does not exist: {data_path}")
        else:
            logger.info(f"Using Claude data path: {data_path}")
    
    def load_usage_data(
        self, 
        hours_back: Optional[int] = None,
        cost_mode: CostMode = CostMode.AUTO
    ) -> List[UsageEntry]:
        """
        Load usage entries from Claude data files.
        
        Args:
            hours_back: Only include entries from last N hours (None for all)
            cost_mode: Cost calculation mode
            
        Returns:
            List of usage entries
        """
        try:
            entries, _ = load_usage_entries(
                data_path=self.claude_data_path,
                hours_back=hours_back,
                mode=cost_mode
            )
            logger.info(f"Loaded {len(entries)} usage entries")
            return entries
        except Exception as e:
            logger.error(f"Failed to load usage data: {e}")
            return []
    
    def get_usage_stats(self, days: int = 30) -> UsageStats:
        """
        Get comprehensive usage statistics.
        
        Args:
            days: Number of days to include in statistics
            
        Returns:
            UsageStats object with aggregated data
        """
        try:
            # Load entries for the specified period
            hours_back = days * 24
            entries = self.load_usage_data(hours_back=hours_back)
            
            if not entries:
                return self._empty_stats()
            
            # Calculate total stats
            total_input = sum(e.input_tokens for e in entries)
            total_output = sum(e.output_tokens for e in entries)
            total_cache_creation = sum(e.cache_creation_tokens for e in entries)
            total_cache_read = sum(e.cache_read_tokens for e in entries)
            total_tokens = total_input + total_output + total_cache_creation + total_cache_read
            total_cost = sum(e.cost_usd for e in entries)
            total_requests = len(entries)
            
            # Calculate today's stats
            today = datetime.now().date()
            today_entries = [e for e in entries if e.timestamp.date() == today]
            tokens_today = sum(e.input_tokens + e.output_tokens + 
                             e.cache_creation_tokens + e.cache_read_tokens 
                             for e in today_entries)
            cost_today = sum(e.cost_usd for e in today_entries)
            requests_today = len(today_entries)
            
            # Calculate this month's stats
            current_month = datetime.now().replace(day=1).date()
            month_entries = [e for e in entries if e.timestamp.date() >= current_month]
            tokens_month = sum(e.input_tokens + e.output_tokens + 
                             e.cache_creation_tokens + e.cache_read_tokens 
                             for e in month_entries)
            cost_month = sum(e.cost_usd for e in month_entries)
            requests_month = len(month_entries)
            
            # Calculate averages
            avg_tokens_per_request = total_tokens / total_requests if total_requests > 0 else 0
            success_rate = 100.0  # Assume all loaded entries are successful
            
            return UsageStats(
                total_tokens=total_tokens,
                total_cost=total_cost,
                total_requests=total_requests,
                input_tokens=total_input,
                output_tokens=total_output,
                cache_creation_tokens=total_cache_creation,
                cache_read_tokens=total_cache_read,
                tokens_today=tokens_today,
                cost_today=cost_today,
                requests_today=requests_today,
                tokens_this_month=tokens_month,
                cost_this_month=cost_month,
                requests_this_month=requests_month,
                avg_tokens_per_request=avg_tokens_per_request,
                success_rate=success_rate
            )
            
        except Exception as e:
            logger.error(f"Failed to get usage stats: {e}")
            return self._empty_stats()
    
    def get_daily_usage(self, days: int = 30) -> List[DailyUsage]:
        """
        Get daily usage breakdown.
        
        Args:
            days: Number of days to include
            
        Returns:
            List of daily usage data
        """
        try:
            # Load entries
            hours_back = days * 24
            entries = self.load_usage_data(hours_back=hours_back)
            
            if not entries:
                return []
            
            # Use claude_monitor's aggregator
            aggregator = UsageAggregator(
                data_path=self.claude_data_path,
                aggregation_mode="daily"
            )
            
            # Get daily aggregated data
            daily_data = aggregator.aggregate_daily(entries)
            
            # Convert to our format
            daily_usage = []
            for day in daily_data:
                daily_usage.append(DailyUsage(
                    date=day['date'],
                    input_tokens=day.get('input_tokens', 0),
                    output_tokens=day.get('output_tokens', 0),
                    total_tokens=(day.get('input_tokens', 0) + 
                                day.get('output_tokens', 0) + 
                                day.get('cache_creation_tokens', 0) + 
                                day.get('cache_read_tokens', 0)),
                    cost=day.get('total_cost', 0.0),
                    requests=day.get('entries_count', 0),
                    models_used=day.get('models_used', [])
                ))
            
            return daily_usage
            
        except Exception as e:
            logger.error(f"Failed to get daily usage: {e}")
            return []
    
    def get_model_breakdown(self, days: int = 30) -> List[Dict[str, Any]]:
        """
        Get usage breakdown by model.
        
        Args:
            days: Number of days to include
            
        Returns:
            List of model usage data
        """
        try:
            # Load entries
            hours_back = days * 24
            entries = self.load_usage_data(hours_back=hours_back)
            
            if not entries:
                return []
            
            # Group by model
            model_stats = {}
            for entry in entries:
                model = entry.model or "unknown"
                if model not in model_stats:
                    model_stats[model] = {
                        'model': model,
                        'input_tokens': 0,
                        'output_tokens': 0,
                        'cache_creation_tokens': 0,
                        'cache_read_tokens': 0,
                        'total_tokens': 0,
                        'cost': 0.0,
                        'requests': 0
                    }
                
                stats = model_stats[model]
                stats['input_tokens'] += entry.input_tokens
                stats['output_tokens'] += entry.output_tokens
                stats['cache_creation_tokens'] += entry.cache_creation_tokens
                stats['cache_read_tokens'] += entry.cache_read_tokens
                stats['total_tokens'] += (entry.input_tokens + entry.output_tokens + 
                                        entry.cache_creation_tokens + entry.cache_read_tokens)
                stats['cost'] += entry.cost_usd
                stats['requests'] += 1
            
            return list(model_stats.values())
            
        except Exception as e:
            logger.error(f"Failed to get model breakdown: {e}")
            return []
    
    def get_monthly_usage(self, months: int = 6) -> List[Dict[str, Any]]:
        """
        Get monthly usage breakdown.
        
        Args:
            months: Number of months to include
            
        Returns:
            List of monthly usage data
        """
        try:
            # Load entries for the specified period
            hours_back = months * 30 * 24  # Approximate
            entries = self.load_usage_data(hours_back=hours_back)
            
            if not entries:
                return []
            
            # Use claude_monitor's aggregator
            aggregator = UsageAggregator(
                data_path=self.claude_data_path,
                aggregation_mode="monthly"
            )
            
            # Get monthly aggregated data
            monthly_data = aggregator.aggregate_monthly(entries)
            
            return monthly_data
            
        except Exception as e:
            logger.error(f"Failed to get monthly usage: {e}")
            return []
    
    def get_usage_trends(self, days: int = 7) -> Dict[str, Any]:
        """
        Get usage trends and patterns.
        
        Args:
            days: Number of days to analyze
            
        Returns:
            Dictionary with trend analysis
        """
        try:
            daily_usage = self.get_daily_usage(days)
            
            if len(daily_usage) < 2:
                return {
                    'token_trend': 'stable',
                    'cost_trend': 'stable',
                    'request_trend': 'stable',
                    'daily_average_tokens': 0,
                    'daily_average_cost': 0,
                    'daily_average_requests': 0
                }
            
            # Calculate trends
            recent_tokens = sum(d.total_tokens for d in daily_usage[:days//2])
            earlier_tokens = sum(d.total_tokens for d in daily_usage[days//2:])
            
            recent_cost = sum(d.cost for d in daily_usage[:days//2])
            earlier_cost = sum(d.cost for d in daily_usage[days//2:])
            
            recent_requests = sum(d.requests for d in daily_usage[:days//2])
            earlier_requests = sum(d.requests for d in daily_usage[days//2:])
            
            # Determine trends
            token_trend = 'increasing' if recent_tokens > earlier_tokens else \
                         'decreasing' if recent_tokens < earlier_tokens else 'stable'
            
            cost_trend = 'increasing' if recent_cost > earlier_cost else \
                        'decreasing' if recent_cost < earlier_cost else 'stable'
            
            request_trend = 'increasing' if recent_requests > earlier_requests else \
                           'decreasing' if recent_requests < earlier_requests else 'stable'
            
            # Calculate averages
            total_days = len(daily_usage)
            daily_avg_tokens = sum(d.total_tokens for d in daily_usage) / total_days
            daily_avg_cost = sum(d.cost for d in daily_usage) / total_days
            daily_avg_requests = sum(d.requests for d in daily_usage) / total_days
            
            return {
                'token_trend': token_trend,
                'cost_trend': cost_trend,
                'request_trend': request_trend,
                'daily_average_tokens': round(daily_avg_tokens),
                'daily_average_cost': round(daily_avg_cost, 4),
                'daily_average_requests': round(daily_avg_requests)
            }
            
        except Exception as e:
            logger.error(f"Failed to get usage trends: {e}")
            return {}
    
    def _empty_stats(self) -> UsageStats:
        """Return empty stats when no data is available."""
        return UsageStats(
            total_tokens=0,
            total_cost=0.0,
            total_requests=0,
            input_tokens=0,
            output_tokens=0,
            cache_creation_tokens=0,
            cache_read_tokens=0,
            tokens_today=0,
            cost_today=0.0,
            requests_today=0,
            tokens_this_month=0,
            cost_this_month=0.0,
            requests_this_month=0,
            avg_tokens_per_request=0.0,
            success_rate=0.0
        )
    
    def get_recent_usage(self, hours: int = 24) -> List[Dict[str, Any]]:
        """
        Get recent usage entries.
        
        Args:
            hours: Number of hours back to look
            
        Returns:
            List of recent usage entries
        """
        try:
            entries = self.load_usage_data(hours_back=hours)
            
            recent_usage = []
            for entry in entries[-50:]:  # Last 50 entries
                recent_usage.append({
                    'timestamp': entry.timestamp.isoformat(),
                    'model': entry.model,
                    'input_tokens': entry.input_tokens,
                    'output_tokens': entry.output_tokens,
                    'cache_creation_tokens': entry.cache_creation_tokens,
                    'cache_read_tokens': entry.cache_read_tokens,
                    'total_tokens': (entry.input_tokens + entry.output_tokens + 
                                   entry.cache_creation_tokens + entry.cache_read_tokens),
                    'cost': entry.cost_usd,
                    'message_id': entry.message_id,
                    'request_id': entry.request_id
                })
            
            return recent_usage
            
        except Exception as e:
            logger.error(f"Failed to get recent usage: {e}")
            return []
    
    def is_data_available(self) -> bool:
        """
        Check if Claude usage data is available.
        
        Returns:
            True if data is available, False otherwise
        """
        try:
            data_path = Path(self.claude_data_path).expanduser()
            if not data_path.exists():
                return False
            
            # Look for .jsonl files
            jsonl_files = list(data_path.rglob("*.jsonl"))
            return len(jsonl_files) > 0
            
        except Exception:
            return False
    
    def get_data_info(self) -> Dict[str, Any]:
        """
        Get information about available data.
        
        Returns:
            Dictionary with data information
        """
        try:
            data_path = Path(self.claude_data_path).expanduser()
            
            info = {
                'data_path': str(data_path),
                'data_path_exists': data_path.exists(),
                'jsonl_files': 0,
                'total_entries': 0,
                'date_range': None
            }
            
            if data_path.exists():
                jsonl_files = list(data_path.rglob("*.jsonl"))
                info['jsonl_files'] = len(jsonl_files)
                
                if jsonl_files:
                    entries = self.load_usage_data()
                    info['total_entries'] = len(entries)
                    
                    if entries:
                        earliest = min(e.timestamp for e in entries)
                        latest = max(e.timestamp for e in entries)
                        info['date_range'] = {
                            'earliest': earliest.isoformat(),
                            'latest': latest.isoformat()
                        }
            
            return info
            
        except Exception as e:
            logger.error(f"Failed to get data info: {e}")
            return {
                'data_path': self.claude_data_path,
                'data_path_exists': False,
                'jsonl_files': 0,
                'total_entries': 0,
                'date_range': None,
                'error': str(e)
            }
    
    def get_usage_predictions(self) -> Dict[str, Any]:
        """Get usage predictions based on current burn rate."""
        try:
            entries = self.load_usage_data(hours_back=24 * 7)  # Last 7 days
            if not entries:
                return self._empty_predictions()
            
            # Get current plan limits
            token_limit = get_token_limit(self.current_plan)
            cost_limit = get_cost_limit(self.current_plan)
            
            # Calculate current usage for the last 5 hours (Claude's reset period)
            now = datetime.now()
            five_hours_ago = now - timedelta(hours=5)
            recent_entries = [e for e in entries if e.timestamp.replace(tzinfo=None) >= five_hours_ago.replace(tzinfo=None)]
            
            # Current 5-hour block usage
            current_tokens = sum(e.input_tokens + e.output_tokens + 
                             e.cache_creation_tokens + e.cache_read_tokens 
                             for e in recent_entries)
            current_cost = sum(e.cost_usd for e in recent_entries)
            
            # Calculate burn rate based on recent usage
            if recent_entries:
                # Use recent entries for burn rate
                hours_in_period = min(5, (now - five_hours_ago).total_seconds() / 3600)
                tokens_per_hour = current_tokens / hours_in_period if hours_in_period > 0 else 0
                cost_per_hour = current_cost / hours_in_period if hours_in_period > 0 else 0
            else:
                # Use 7-day average
                week_tokens = sum(e.input_tokens + e.output_tokens + 
                                e.cache_creation_tokens + e.cache_read_tokens 
                                for e in entries)
                week_cost = sum(e.cost_usd for e in entries)
                tokens_per_hour = week_tokens / (7 * 24) if week_tokens > 0 else 0
                cost_per_hour = week_cost / (7 * 24) if week_cost > 0 else 0
            
            # Calculate time until limits (for 5-hour block)
            remaining_tokens = max(0, token_limit - current_tokens)
            remaining_cost = max(0, cost_limit - current_cost)
            
            hours_until_token_limit = remaining_tokens / tokens_per_hour if tokens_per_hour > 0 else float('inf')
            hours_until_cost_limit = remaining_cost / cost_per_hour if cost_per_hour > 0 else float('inf')
            
            hours_until_limit = min(hours_until_token_limit, hours_until_cost_limit)
            limit_type = 'tokens' if hours_until_token_limit < hours_until_cost_limit else 'cost'
            
            # Project usage for end of 5-hour block
            hours_left_in_block = max(0, 5 - min(5, (now - five_hours_ago).total_seconds() / 3600))
            projected_tokens_block = current_tokens + (tokens_per_hour * hours_left_in_block)
            projected_cost_block = current_cost + (cost_per_hour * hours_left_in_block)
            
            return {
                'burn_rate': {
                    'tokens_per_hour': round(tokens_per_hour),
                    'cost_per_hour': round(cost_per_hour, 2),
                    'tokens_per_minute': round(tokens_per_hour / 60)
                },
                'current_usage': {
                    'tokens': current_tokens,
                    'cost': round(current_cost, 2),
                    'tokens_percentage': round((current_tokens / token_limit) * 100, 1) if token_limit > 0 else 0,
                    'cost_percentage': round((current_cost / cost_limit) * 100, 1) if cost_limit > 0 else 0
                },
                'limits': {
                    'token_limit': token_limit,
                    'cost_limit': cost_limit,
                    'plan': self.current_plan
                },
                'projections': {
                    'tokens_end_of_day': round(projected_tokens_block),
                    'cost_end_of_day': round(projected_cost_block, 2),
                    'hours_until_limit': round(hours_until_limit, 1) if hours_until_limit != float('inf') else None,
                    'limit_type': limit_type if hours_until_limit != float('inf') else None,
                    'will_hit_limit_today': hours_until_limit < hours_left_in_block
                }
            }
        except Exception as e:
            logger.error(f"Error calculating predictions: {e}")
            return self._empty_predictions()
    
    def _empty_predictions(self) -> Dict[str, Any]:
        """Return empty predictions structure."""
        return {
            'burn_rate': {
                'tokens_per_hour': 0,
                'cost_per_hour': 0,
                'tokens_per_minute': 0
            },
            'current_usage': {
                'tokens': 0,
                'cost': 0,
                'tokens_percentage': 0,
                'cost_percentage': 0
            },
            'limits': {
                'token_limit': get_token_limit(self.current_plan),
                'cost_limit': get_cost_limit(self.current_plan),
                'plan': self.current_plan
            },
            'projections': {
                'tokens_end_of_day': 0,
                'cost_end_of_day': 0,
                'hours_until_limit': None,
                'limit_type': None,
                'will_hit_limit_today': False
            }
        }
    
    def set_plan(self, plan_type: str) -> bool:
        """Update the current plan type."""
        try:
            if plan_type.lower() in ['pro', 'max5', 'max20', 'custom']:
                self.current_plan = plan_type.lower()
                return True
            return False
        except Exception:
            return False
    
    def get_available_plans(self) -> List[Dict[str, Any]]:
        """Get list of available plans with their limits."""
        plans = []
        for plan_type in PlanType:
            config = Plans.get_plan(plan_type)
            plans.append({
                'name': config.name,
                'display_name': config.display_name,
                'token_limit': config.token_limit,
                'cost_limit': config.cost_limit,
                'message_limit': config.message_limit,
                'formatted_token_limit': config.formatted_token_limit
            })
        return plans