"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const core_1 = require("@aws-cdk/core");
const kms_generated_1 = require("./kms.generated");
const REQUIRED_ALIAS_PREFIX = 'alias/';
const DISALLOWED_PREFIX = REQUIRED_ALIAS_PREFIX + 'aws/';
class AliasBase extends core_1.Resource {
}
/**
 * Defines a display name for a customer master key (CMK) in AWS Key Management
 * Service (AWS KMS). Using an alias to refer to a key can help you simplify key
 * management. For example, when rotating keys, you can just update the alias
 * mapping instead of tracking and changing key IDs. For more information, see
 * Working with Aliases in the AWS Key Management Service Developer Guide.
 *
 * You can also add an alias for a key by calling `key.addAlias(alias)`.
 *
 * @resource AWS::KMS::Alias
 */
class Alias extends AliasBase {
    static fromAliasAttributes(scope, id, attrs) {
        // tslint:disable-next-line: class-name
        class _Alias extends AliasBase {
            get aliasName() { return attrs.aliasName; }
            get aliasTargetKey() { return attrs.aliasTargetKey; }
        }
        return new _Alias(scope, id);
    }
    constructor(scope, id, props) {
        super(scope, id);
        if (!core_1.Token.isUnresolved(props.aliasName)) {
            if (!props.aliasName.startsWith(REQUIRED_ALIAS_PREFIX)) {
                throw new Error(`Alias must start with the prefix "${REQUIRED_ALIAS_PREFIX}": ${props.aliasName}`);
            }
            if (props.aliasName === REQUIRED_ALIAS_PREFIX) {
                throw new Error(`Alias must include a value after "${REQUIRED_ALIAS_PREFIX}": ${props.aliasName}`);
            }
            if (props.aliasName.startsWith(DISALLOWED_PREFIX)) {
                throw new Error(`Alias cannot start with ${DISALLOWED_PREFIX}: ${props.aliasName}`);
            }
            if (!props.aliasName.match(/^[a-zA-Z0-9:/_-]{1,256}$/)) {
                throw new Error(`Alias name must be between 1 and 256 characters in a-zA-Z0-9:/_-`);
            }
        }
        const resource = new kms_generated_1.CfnAlias(this, 'Resource', {
            aliasName: props.aliasName,
            targetKeyId: props.targetKey.keyArn
        });
        this.aliasName = resource.aliasName;
    }
}
exports.Alias = Alias;
//# sourceMappingURL=data:application/json;base64,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