# Copyright (C) 2023 - 2025 ANSYS, Inc. and/or its affiliates.
# SPDX-License-Identifier: MIT
#
#
# Permission is hereby granted, free of charge, to any person obtaining a copy
# of this software and associated documentation files (the "Software"), to deal
# in the Software without restriction, including without limitation the rights
# to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
# copies of the Software, and to permit persons to whom the Software is
# furnished to do so, subject to the following conditions:
#
# The above copyright notice and this permission notice shall be included in all
# copies or substantial portions of the Software.
#
# THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
# IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
# FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
# AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
# LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
# OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
# SOFTWARE.

"""Produces the def file for declaring the exported functions of the DLL."""

from pathlib import Path

import ansys.scade.python_wrapper.pydata as data


def generate_def(model: data.Model, def_pathname: Path, cosim: bool, banner: str = '') -> None:
    """Generate the C definition file for the DLL."""

    def add_export(function: data.Function):
        nonlocal f, i
        if function:
            f.write('\t%s @ %d;\n' % (function.c_name, i))
            i += 1

    with def_pathname.open('w') as f:
        if banner:
            f.write('; generated by %s\n' % banner)
            f.write('\n')

        f.write('EXPORTS\n')
        i = 1
        for op in model.operators:
            add_export(op.cycle)
            add_export(op.init)
            add_export(op.reset)
        for sensor in model.sensors:
            f.write('\t%s @ %d;\n' % (sensor.c_name, i))
            i += 1

        if cosim:
            # SsmSlaveLib
            # unselected functions: SsmGetLastError SsmGetOutput SsmGetOutputVector
            functions = """
            SsmCloseScadeSimulator SsmDbgStep SsmGuiActivate SsmGuiRefresh SsmOpenScadeSimulator
            SsmSetInput SsmSetInputVector SsmSetConnectRetry SsmSetHostName
            SsmSetScadeInstallPath SsmStep SsmPause SsmSaveScenario
            """
            for name in functions.split():
                f.write('\t%s @ %d;\n' % (name, i))
                i += 1

            c_types = {_.c_type for _ in model.sensors}
            for op in model.operators:
                c_types.update({_.c_type for _ in op.ios})
            for c_type_ in sorted(c_types):
                f.write('\t%s_to_string @ %d;\n' % (c_type_, i))
                i += 1

        # note: the SDY functions are exported by defining the macro DLL_EXPORTS
        # cf. declare_target()

        if model.elaboration:
            f.write('\t%s @ %d;\n' % (model.elaboration, i))
            i += 1

        f.write('\n')
        f.write('; end of file\n')
