# SimpliQ MCP Server - OAuth 2.0 Client Module

> OAuth 2.0 authentication client for validating tokens from User Manager API

**Status:** ✅ Production-ready
**Version:** 1.0.0 (Phase 2)
**Created:** 25 November 2025

---

## 📖 Overview
- [PHASE2_IMPLEMENTATION_SUMMARY.md](../../docs/auth/OAUTH_GUIDE.md) - See consolidated guide
 - [OAuth Guide (consolidado)](../../docs/auth/OAUTH_GUIDE.md) - Comprehensive testing & reference (consolidado)

### Key Features

- ✅ **Token Validation** - Validates Bearer tokens via User Manager API
- ✅ **Token Caching** - Caches validated tokens (configurable TTL)
- ✅ **Context Propagation** - Propagates user context to Flask's `g` object
- ✅ **Error Handling** - Robust error handling with detailed logging
- ✅ **Performance** - Minimal overhead with smart caching

---

## 🏗️ Architecture

```
┌─────────────────────────────────────────────────────────────┐
│ MCP Client (e.g., Claude Desktop)                          │
│ - Sends Authorization: Bearer <token>                       │
└────────────────────┬────────────────────────────────────────┘
                     │
                     ▼
┌─────────────────────────────────────────────────────────────┐
│ MCP Server (mcp_server.py)                                  │
│ - Receives request with Bearer token                        │
│ - Passes to OAuthMiddleware                                 │
└────────────────────┬────────────────────────────────────────┘
                     │
                     ▼
┌─────────────────────────────────────────────────────────────┐
│ OAuthMiddleware (oauth/middleware.py)                       │
│ 1. Extract Bearer token from Authorization header           │
│ 2. Check cache for validated token                          │
│ 3. If not cached, validate via OAuthClient                  │
│ 4. Cache result (TTL: 5 minutes)                            │
│ 5. Propagate context to Flask's g                           │
└────────────────────┬────────────────────────────────────────┘
                     │
                     ▼
┌─────────────────────────────────────────────────────────────┐
│ OAuthClient (oauth/client.py)                               │
│ - Calls User Manager API: GET /oauth/userinfo              │
│ - Returns user info if token is valid                       │
└────────────────────┬────────────────────────────────────────┘
                     │
                     ▼
┌─────────────────────────────────────────────────────────────┐
│ User Manager API (OAuth Provider)                           │
│ - Validates JWT signature and expiration                    │
│ - Returns user info: user_id, client_id, org_id, scope     │
└─────────────────────────────────────────────────────────────┘
```

---

## 📁 Module Structure

```
simpliq_server/oauth/
├── __init__.py              # Module exports
├── client.py                # OAuthClient - HTTP client for User Manager API
└── middleware.py            # OAuthMiddleware - token validation and caching
```

---

## 🔧 Components

### 1. OAuthClient (`client.py`)

HTTP client for communicating with User Manager OAuth Provider.

**Methods:**
- `validate_token(access_token: str) -> Optional[Dict]`
  - Validates token via `/oauth/userinfo` endpoint
  - Returns user info if valid, None otherwise

- `introspect_token(token: str, token_type_hint: str) -> Optional[Dict]`
  - Introspects token (RFC 7662)
  - Not used in Phase 2, available for future

- `refresh_access_token(refresh_token: str, client_id: str, client_secret: str) -> Optional[Dict]`
  - Refreshes access token using refresh token
  - Returns new token response

- `revoke_token(token: str, token_type_hint: str) -> bool`
  - Revokes an OAuth token
  - Returns True if successful

**Example:**
```python
from oauth.client import OAuthClient

client = OAuthClient(provider_url="http://localhost:8002")
user_info = client.validate_token("eyJhbGc...")

if user_info:
    print(f"User: {user_info['sub']}")
    print(f"Org: {user_info['org_id']}")
```

### 2. OAuthMiddleware (`middleware.py`)

Middleware for validating OAuth tokens in Flask requests.

**Methods:**
- `validate_token(access_token: str, use_cache: bool) -> Optional[Dict]`
  - Validates token with optional caching
  - Cache TTL configurable (default: 5 minutes)

- `validate_request(flask_request=None) -> tuple[bool, Optional[Dict], Optional[str]]`
  - Validates entire Flask request
  - Returns (is_valid, user_info, error_message)

- `extract_token_from_request(flask_request=None) -> Optional[str]`
  - Extracts Bearer token from Authorization header

- `require_oauth` (decorator)
  - Decorator to protect Flask routes
  - Automatically validates token and propagates context

**Example:**
```python
from oauth.middleware import OAuthMiddleware

middleware = OAuthMiddleware(
    provider_url="http://localhost:8002",
    cache_ttl=300,
    enabled=True
)

# In Flask route handler
is_valid, user_info, error = middleware.validate_request()
if not is_valid:
    return jsonify({"error": error}), 401
```

**Context Propagation:**

After successful validation, the following are available in Flask's `g`:
- `g.oauth_user_id` - User ID from token
- `g.oauth_client_id` - Client ID from token
- `g.oauth_org_id` - Organization ID from token
- `g.oauth_scope` - Space-separated scopes
- `g.oauth_user_info` - Full user info dict

---

## ⚙️ Configuration

**File:** `simpliq_server/config.yml`

```yaml
oauth:
  # Enable/disable OAuth validation
  enabled: true

  # User Manager API base URL
  provider_url: "http://localhost:8002"

  # Token validation settings
  token_validation:
    # Cache TTL in seconds (default: 300 = 5 minutes)
    cache_ttl: 300

  # Development mode (DANGER!)
  dev_mode:
    enabled: false
    skip_validation: false  # Never enable in production!
```

### Configuration Options

| Option | Default | Description |
|--------|---------|-------------|
| `oauth.enabled` | `true` | Enable OAuth validation |
| `oauth.provider_url` | `http://localhost:8002` | User Manager API URL |
| `oauth.token_validation.cache_ttl` | `300` | Token cache TTL (seconds) |
| `oauth.dev_mode.skip_validation` | `false` | Skip validation (DEV ONLY!) |

---

## 🚀 Usage

### Integration with MCP Server

The OAuth middleware is automatically initialized in `mcp_server.py` on startup:

```python
# mcp_server.py (lines 256-290)

oauth_middleware = None
try:
    with open(CONFIG_FILE, 'r') as f:
        config = yaml.safe_load(f)
        oauth_config = config.get('oauth', {})

        if oauth_config.get('enabled', False):
            provider_url = oauth_config.get('provider_url', 'http://localhost:8002')
            cache_ttl = oauth_config.get('token_validation', {}).get('cache_ttl', 300)

            oauth_middleware = OAuthMiddleware(
                provider_url=provider_url,
                cache_ttl=cache_ttl,
                enabled=True
            )
```

### Token Validation in tools/call

OAuth validation occurs automatically for all `tools/call` requests:

```python
# mcp_server.py (lines 1814-1840)

if oauth_middleware and oauth_middleware.enabled:
    is_valid, oauth_user_info, error_msg = oauth_middleware.validate_request()

    if not is_valid:
        return jsonify({
            "jsonrpc": "2.0",
            "id": rpc_id,
            "error": {
                "code": -32001,
                "message": "OAuth authentication required",
                "data": {"detail": error_msg}
            }
        }), 401

    # Propagate OAuth context
    g.oauth_user_id = oauth_user_info.get("sub")
    g.oauth_client_id = oauth_user_info.get("client_id")
    g.oauth_org_id = oauth_user_info.get("org_id")
    g.oauth_scope = oauth_user_info.get("scope", "")
```

---

## 🧪 Testing

### Unit Tests

```bash
cd simpliq_server
pytest tests/oauth/test_oauth_integration.py -v
```

### Integration Example

```bash
# Requires User Manager API running on localhost:8002
python tests/oauth/test_mcp_oauth_example.py
```

**Expected output:**
```
======================================================================
STEP 1: Creating OAuth Client in User Manager
[OK] Client created: client-abc123...

STEP 2: Obtaining Access Token (Client Credentials)
[OK] Access token obtained

STEP 3: Calling MCP Tool with OAuth Token
[OK] Retrieved 50+ MCP tools

[SUCCESS] OAuth 2.0 + MCP Integration Test Completed!
```

---

## 📊 Performance

### Token Caching

- **Cache TTL:** 5 minutes (configurable)
- **Cache Type:** In-memory dict
- **Eviction:** Automatic on expiry
- **Cleanup:** Periodic cleanup of expired entries

### Benchmarks

| Operation | Cached | Uncached |
|-----------|--------|----------|
| Token validation | < 1 ms | ~50-100 ms |
| Request overhead | ~1-2 ms | ~50-100 ms |

**Recommendations:**
- Use caching in production (always enabled by default)
- Adjust `cache_ttl` based on security vs. performance trade-off
- Monitor cache hit ratio with `get_cache_stats()`

---

## 🔒 Security Considerations

### Token Transport
- ✅ Tokens transmitted via HTTPS (use `ssl_enabled: true` in production)
- ✅ Bearer tokens in Authorization header (not in URL)
- ✅ No token logging (only first 20 chars in debug logs)

### Token Validation
- ✅ Signature verification (JWT with HS256)
- ✅ Expiration check (exp claim)
- ✅ Revocation support
- ✅ Scope validation (future enhancement)

### Caching
- ✅ Cache respects token expiration
- ✅ Automatic cleanup of expired entries
- ✅ Cache cleared on server restart

---

## 🐛 Troubleshooting

### OAuth validation fails with "Missing or invalid Authorization header"

**Cause:** No Authorization header or incorrect format

**Solution:**
```bash
# Correct format:
Authorization: Bearer eyJhbGciOiJIUzI1NiIsInR5cCI6IkpXVCJ9...

# NOT:
Authorization: eyJhbGc...  # Missing "Bearer"
```

### OAuth validation fails with "Invalid or expired access token"

**Cause:** Token invalid, expired, or revoked

**Solution:**
1. Check token expiration (default: 1 hour)
2. Obtain new token from User Manager
3. Verify User Manager is running
4. Check `provider_url` in config.yml

### High latency on token validation

**Cause:** Cache disabled or very low TTL

**Solution:**
```yaml
oauth:
  token_validation:
    cache_ttl: 300  # Increase to 5-10 minutes
```

---

## 📚 References

- [PHASE2_IMPLEMENTATION_SUMMARY.md](../../docs/auth/OAUTH_GUIDE.md) - See consolidated guide
 - [OAuth Guide (consolidado)](../../docs/auth/OAUTH_GUIDE.md) - Comprehensive testing & reference (consolidado)
- [RFC 6750](https://datatracker.ietf.org/doc/html/rfc6750) - OAuth 2.0 Bearer Token Usage

---

## 📝 Changelog

### Version 1.0.0 (2025-11-25) - Phase 2 Complete
- ✅ OAuthClient implemented
- ✅ OAuthMiddleware implemented
- ✅ Token caching with configurable TTL
- ✅ Integration with MCP Server
- ✅ Context propagation to Flask's g
- ✅ Unit tests and integration examples
- ✅ Complete documentation

---

**OAuth 2.0 Client Module - SimpliQ MCP Server** 🔐

**Developed by:** Claude AI (Sonnet 4.5) & Gerson Amorim
**Project:** SimpliQ - Intelligent ERP Data Platform
**License:** MIT
