from typing import Optional


class AmazonBedrockError(Exception):
    """
    Any error generated by the Amazon Bedrock integration.

    This error wraps its source transparently in such a way that its attributes
    can be accessed directly: for example, if the original error has a `message` attribute,
    `AmazonBedrockError.message` will exist and have the expected content.
    """

    def __init__(
        self,
        message: Optional[str] = None,
    ):
        super().__init__()
        if message:
            self.message = message

    def __getattr__(self, attr):
        # If self.__cause__ is None, it will raise the expected AttributeError
        getattr(self.__cause__, attr)

    def __str__(self):
        return self.message

    def __repr__(self):
        return str(self)


class AWSConfigurationError(AmazonBedrockError):
    """Exception raised when AWS is not configured correctly"""

    def __init__(self, message: Optional[str] = None):
        super().__init__(message=message)


class AmazonBedrockConfigurationError(AmazonBedrockError):
    """Exception raised when AmazonBedrock node is not configured correctly"""

    def __init__(self, message: Optional[str] = None):
        super().__init__(message=message)


class AmazonBedrockInferenceError(AmazonBedrockError):
    """Exception for issues that occur in the Bedrock inference node"""

    def __init__(self, message: Optional[str] = None):
        super().__init__(message=message)
