"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.deserialize = exports.serialize = void 0;
const yaml = require("yaml");
const yaml_types = require("yaml/types");
/**
 * Serializes the given data structure into valid YAML.
 *
 * @param obj the data structure to serialize
 * @returns a string containing the YAML representation of {@param obj}
 */
function serialize(obj) {
    const oldFold = yaml_types.strOptions.fold.lineWidth;
    try {
        yaml_types.strOptions.fold.lineWidth = 0;
        return yaml.stringify(obj, { schema: 'yaml-1.1' });
    }
    finally {
        yaml_types.strOptions.fold.lineWidth = oldFold;
    }
}
exports.serialize = serialize;
/**
 * Deserialize the YAML into the appropriate data structure.
 *
 * @param str the string containing YAML
 * @returns the data structure the YAML represents
 *   (most often in case of CloudFormation, an object)
 */
function deserialize(str) {
    return parseYamlStrWithCfnTags(str);
}
exports.deserialize = deserialize;
function makeTagForCfnIntrinsic(intrinsicName, addFnPrefix = true, resolveFun) {
    return {
        identify(value) { return typeof value === 'string'; },
        tag: `!${intrinsicName}`,
        resolve: resolveFun || ((_doc, cstNode) => {
            const ret = {};
            ret[addFnPrefix ? `Fn::${intrinsicName}` : intrinsicName] =
                // the +1 is to account for the ! the short form begins with
                parseYamlStrWithCfnTags(cstNode.toString().substring(intrinsicName.length + 1));
            return ret;
        }),
    };
}
const shortForms = [
    'Base64', 'Cidr', 'FindInMap', 'GetAZs', 'ImportValue', 'Join', 'Sub',
    'Select', 'Split', 'Transform', 'And', 'Equals', 'If', 'Not', 'Or',
].map(name => makeTagForCfnIntrinsic(name)).concat(makeTagForCfnIntrinsic('Ref', false), makeTagForCfnIntrinsic('Condition', false), makeTagForCfnIntrinsic('GetAtt', true, (_doc, cstNode) => {
    const parsedArguments = parseYamlStrWithCfnTags(cstNode.toString().substring('!GetAtt'.length));
    let value;
    if (typeof parsedArguments === 'string') {
        // if the arguments to !GetAtt are a string,
        // the part before the first '.' is the logical ID,
        // and the rest is the attribute name
        // (which can contain '.')
        const firstDot = parsedArguments.indexOf('.');
        if (firstDot === -1) {
            throw new Error(`Short-form Fn::GetAtt must contain a '.' in its string argument, got: '${parsedArguments}'`);
        }
        value = [
            parsedArguments.substring(0, firstDot),
            parsedArguments.substring(firstDot + 1),
        ];
    }
    else {
        // this is the form where the arguments to Fn::GetAtt are already an array -
        // in this case, nothing more to do
        value = parsedArguments;
    }
    return { 'Fn::GetAtt': value };
}));
function parseYamlStrWithCfnTags(text) {
    return yaml.parse(text, {
        customTags: shortForms,
        schema: 'yaml-1.1',
    });
}
//# sourceMappingURL=data:application/json;base64,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