"""Configuration export command.

This module provides the 'framework export-config' command which displays
the framework's default configuration template. This helps users understand what
configuration options are available when creating new projects.
"""

import click
import yaml
from pathlib import Path
from rich.console import Console
from rich.syntax import Syntax
from jinja2 import Template


console = Console()


@click.command(name="export-config")
@click.option(
    "--output", "-o",
    type=click.Path(),
    help="Output file (default: print to console)"
)
@click.option(
    "--format",
    type=click.Choice(["yaml", "json"]),
    default="yaml",
    help="Output format (default: yaml)"
)
def export_config(output: str, format: str):
    """Export framework default configuration template.
    
    Displays the framework's default configuration template that is used
    when creating new projects with 'framework init'. This is useful for:
    
    \b
      - Understanding available configuration options
      - Seeing default values for models, services, etc.
      - Debugging configuration issues
      - Creating custom configurations
    
    The exported configuration shows the complete framework template
    rendered with example values.
    
    Examples:
    
    \b
      # Display to console with syntax highlighting
      $ framework export-config
      
      # Save to file
      $ framework export-config -o framework-defaults.yml
      
      # Export as JSON
      $ framework export-config --format json -o framework-defaults.json
    """
    try:
        # Load framework's configuration template (known location in framework structure)
        template_path = Path(__file__).parent.parent / "templates" / "project" / "config.yml.j2"
        
        if not template_path.exists():
            console.print(
                "❌ Could not locate framework configuration template.",
                style="red"
            )
            console.print(
                f"   Expected at: {template_path}",
                style="dim"
            )
            raise click.Abort()
        
        # Read and render the template with example values
        with open(template_path, 'r') as f:
            template_content = f.read()
        
        template = Template(template_content)
        rendered_config = template.render(
            project_name="example_project",
            package_name="example_project",
            project_root="/path/to/example_project",
            hostname="localhost",
            default_provider="cborg",
            default_model="google/gemini-flash"
        )
        
        # Parse the rendered config as YAML
        config_data = yaml.safe_load(rendered_config)
        
        # Format output based on requested format
        if format == "yaml":
            output_str = yaml.dump(
                config_data,
                default_flow_style=False,
                sort_keys=False,
                allow_unicode=True
            )
        else:  # json
            import json
            output_str = json.dumps(config_data, indent=2, ensure_ascii=False)
        
        # Output to file or console
        if output:
            output_path = Path(output)
            output_path.write_text(output_str)
            console.print(
                f"✅ Configuration exported to: [bold]{output_path}[/bold]"
            )
        else:
            # Print to console with syntax highlighting
            console.print("\n[bold]Framework Default Configuration:[/bold]\n")
            syntax = Syntax(
                output_str,
                format,
                theme="monokai",
                line_numbers=False,
                word_wrap=True
            )
            console.print(syntax)
            console.print(
                f"\n[dim]💡 Tip: Save to file with --output flag[/dim]"
            )
            
    except KeyboardInterrupt:
        console.print("\n⚠️  Operation cancelled", style="yellow")
        raise click.Abort()
    except Exception as e:
        console.print(f"❌ Failed to export configuration: {e}", style="red")
        import os
        if os.environ.get("DEBUG"):
            import traceback
            console.print(traceback.format_exc(), style="dim")
        raise click.Abort()


if __name__ == "__main__":
    export_config()

