"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.Role = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const core_1 = require("@aws-cdk/core");
const constructs_1 = require("constructs");
const grant_1 = require("./grant");
const iam_generated_1 = require("./iam.generated");
const policy_1 = require("./policy");
const policy_document_1 = require("./policy-document");
const policy_statement_1 = require("./policy-statement");
const principals_1 = require("./principals");
const immutable_role_1 = require("./private/immutable-role");
const util_1 = require("./util");
/**
 * IAM Role.
 *
 * Defines an IAM role. The role is created with an assume policy document associated with
 * the specified AWS service principal defined in `serviceAssumeRole`.
 *
 * @stability stable
 */
class Role extends core_1.Resource {
    /**
     * @stability stable
     */
    constructor(scope, id, props) {
        var _b;
        super(scope, id, {
            physicalName: props.roleName,
        });
        /**
         * The principal to grant permissions to.
         *
         * @stability stable
         */
        this.grantPrincipal = this;
        /**
         * The AWS account ID of this principal.
         *
         * Can be undefined when the account is not known
         * (for example, for service principals).
         * Can be a Token - in that case,
         * it's assumed to be AWS::AccountId.
         *
         * @stability stable
         */
        this.principalAccount = this.env.account;
        /**
         * When this Principal is used in an AssumeRole policy, the action to use.
         *
         * @stability stable
         */
        this.assumeRoleAction = 'sts:AssumeRole';
        this.managedPolicies = [];
        this.attachedPolicies = new util_1.AttachedPolicies();
        const externalIds = props.externalIds || [];
        if (props.externalId) {
            externalIds.push(props.externalId);
        }
        this.assumeRolePolicy = createAssumeRolePolicy(props.assumedBy, externalIds);
        this.managedPolicies.push(...props.managedPolicies || []);
        this.inlinePolicies = props.inlinePolicies || {};
        this.permissionsBoundary = props.permissionsBoundary;
        const maxSessionDuration = props.maxSessionDuration && props.maxSessionDuration.toSeconds();
        validateMaxSessionDuration(maxSessionDuration);
        const description = (props.description && ((_b = props.description) === null || _b === void 0 ? void 0 : _b.length) > 0) ? props.description : undefined;
        if (description && description.length > 1000) {
            throw new Error('Role description must be no longer than 1000 characters.');
        }
        const role = new iam_generated_1.CfnRole(this, 'Resource', {
            assumeRolePolicyDocument: this.assumeRolePolicy,
            managedPolicyArns: util_1.UniqueStringSet.from(() => this.managedPolicies.map(p => p.managedPolicyArn)),
            policies: _flatten(this.inlinePolicies),
            path: props.path,
            permissionsBoundary: this.permissionsBoundary ? this.permissionsBoundary.managedPolicyArn : undefined,
            roleName: this.physicalName,
            maxSessionDuration,
            description,
        });
        this.roleId = role.attrRoleId;
        this.roleArn = this.getResourceArnAttribute(role.attrArn, {
            region: '',
            service: 'iam',
            resource: 'role',
            resourceName: this.physicalName,
        });
        this.roleName = this.getResourceNameAttribute(role.ref);
        this.policyFragment = new principals_1.ArnPrincipal(this.roleArn).policyFragment;
        function _flatten(policies) {
            if (policies == null || Object.keys(policies).length === 0) {
                return undefined;
            }
            const result = new Array();
            for (const policyName of Object.keys(policies)) {
                const policyDocument = policies[policyName];
                result.push({ policyName, policyDocument });
            }
            return result;
        }
    }
    /**
     * Import an external role by ARN.
     *
     * If the imported Role ARN is a Token (such as a
     * `CfnParameter.valueAsString` or a `Fn.importValue()`) *and* the referenced
     * role has a `path` (like `arn:...:role/AdminRoles/Alice`), the
     * `roleName` property will not resolve to the correct value. Instead it
     * will resolve to the first path component. We unfortunately cannot express
     * the correct calculation of the full path name as a CloudFormation
     * expression. In this scenario the Role ARN should be supplied without the
     * `path` in order to resolve the correct role resource.
     *
     * @param scope construct scope.
     * @param id construct id.
     * @param roleArn the ARN of the role to import.
     * @param options allow customizing the behavior of the returned role.
     * @stability stable
     */
    static fromRoleArn(scope, id, roleArn, options = {}) {
        const scopeStack = core_1.Stack.of(scope);
        const parsedArn = scopeStack.parseArn(roleArn);
        const resourceName = parsedArn.resourceName;
        const roleAccount = parsedArn.account;
        // service roles have an ARN like 'arn:aws:iam::<account>:role/service-role/<roleName>'
        // or 'arn:aws:iam::<account>:role/service-role/servicename.amazonaws.com/service-role/<roleName>'
        // we want to support these as well, so we just use the element after the last slash as role name
        const roleName = resourceName.split('/').pop();
        class Import extends core_1.Resource {
            constructor(_scope, _id) {
                super(_scope, _id, {
                    account: roleAccount,
                });
                this.grantPrincipal = this;
                this.principalAccount = roleAccount;
                this.assumeRoleAction = 'sts:AssumeRole';
                this.policyFragment = new principals_1.ArnPrincipal(roleArn).policyFragment;
                this.roleArn = roleArn;
                this.roleName = roleName;
                this.attachedPolicies = new util_1.AttachedPolicies();
            }
            addToPolicy(statement) {
                return this.addToPrincipalPolicy(statement).statementAdded;
            }
            addToPrincipalPolicy(statement) {
                if (!this.defaultPolicy) {
                    this.defaultPolicy = new policy_1.Policy(this, 'Policy');
                    this.attachInlinePolicy(this.defaultPolicy);
                }
                this.defaultPolicy.addStatements(statement);
                return { statementAdded: true, policyDependable: this.defaultPolicy };
            }
            attachInlinePolicy(policy) {
                const thisAndPolicyAccountComparison = core_1.Token.compareStrings(this.env.account, policy.env.account);
                const equalOrAnyUnresolved = thisAndPolicyAccountComparison === core_1.TokenComparison.SAME ||
                    thisAndPolicyAccountComparison === core_1.TokenComparison.BOTH_UNRESOLVED ||
                    thisAndPolicyAccountComparison === core_1.TokenComparison.ONE_UNRESOLVED;
                if (equalOrAnyUnresolved) {
                    this.attachedPolicies.attach(policy);
                    policy.attachToRole(this);
                }
            }
            addManagedPolicy(_policy) {
                // FIXME: Add warning that we're ignoring this
            }
            /**
             * Grant permissions to the given principal to pass this role.
             */
            grantPassRole(identity) {
                return this.grant(identity, 'iam:PassRole');
            }
            /**
             * Grant the actions defined in actions to the identity Principal on this resource.
             */
            grant(grantee, ...actions) {
                return grant_1.Grant.addToPrincipal({
                    grantee,
                    actions,
                    resourceArns: [this.roleArn],
                    scope: this,
                });
            }
        }
        const importedRole = new Import(scope, id);
        const roleArnAndScopeStackAccountComparison = core_1.Token.compareStrings(importedRole.env.account, scopeStack.account);
        const equalOrAnyUnresolved = roleArnAndScopeStackAccountComparison === core_1.TokenComparison.SAME ||
            roleArnAndScopeStackAccountComparison === core_1.TokenComparison.BOTH_UNRESOLVED ||
            roleArnAndScopeStackAccountComparison === core_1.TokenComparison.ONE_UNRESOLVED;
        // we only return an immutable Role if both accounts were explicitly provided, and different
        return options.mutable !== false && equalOrAnyUnresolved
            ? importedRole
            : new immutable_role_1.ImmutableRole(scope, `ImmutableRole${id}`, importedRole);
    }
    /**
     * Adds a permission to the role's default policy document.
     *
     * If there is no default policy attached to this role, it will be created.
     *
     * @param statement The permission statement to add to the policy document.
     * @stability stable
     */
    addToPrincipalPolicy(statement) {
        if (!this.defaultPolicy) {
            this.defaultPolicy = new policy_1.Policy(this, 'DefaultPolicy');
            this.attachInlinePolicy(this.defaultPolicy);
        }
        this.defaultPolicy.addStatements(statement);
        return { statementAdded: true, policyDependable: this.defaultPolicy };
    }
    /**
     * Add to the policy of this principal.
     *
     * @stability stable
     */
    addToPolicy(statement) {
        return this.addToPrincipalPolicy(statement).statementAdded;
    }
    /**
     * Attaches a managed policy to this role.
     *
     * @param policy The the managed policy to attach.
     * @stability stable
     */
    addManagedPolicy(policy) {
        if (this.managedPolicies.find(mp => mp === policy)) {
            return;
        }
        this.managedPolicies.push(policy);
    }
    /**
     * Attaches a policy to this role.
     *
     * @param policy The policy to attach.
     * @stability stable
     */
    attachInlinePolicy(policy) {
        this.attachedPolicies.attach(policy);
        policy.attachToRole(this);
    }
    /**
     * Grant the actions defined in actions to the identity Principal on this resource.
     *
     * @stability stable
     */
    grant(grantee, ...actions) {
        return grant_1.Grant.addToPrincipal({
            grantee,
            actions,
            resourceArns: [this.roleArn],
            scope: this,
        });
    }
    /**
     * Grant permissions to the given principal to pass this role.
     *
     * @stability stable
     */
    grantPassRole(identity) {
        return this.grant(identity, 'iam:PassRole');
    }
    /**
     * Return a copy of this Role object whose Policies will not be updated.
     *
     * Use the object returned by this method if you want this Role to be used by
     * a construct without it automatically updating the Role's Policies.
     *
     * If you do, you are responsible for adding the correct statements to the
     * Role's policies yourself.
     *
     * @stability stable
     */
    withoutPolicyUpdates() {
        if (!this.immutableRole) {
            this.immutableRole = new immutable_role_1.ImmutableRole(constructs_1.Node.of(this).scope, `ImmutableRole${this.node.id}`, this);
        }
        return this.immutableRole;
    }
    /**
     * Validate the current construct.
     *
     * This method can be implemented by derived constructs in order to perform
     * validation logic. It is called on all constructs before synthesis.
     *
     * @stability stable
     */
    validate() {
        var _b;
        const errors = super.validate();
        errors.push(...((_b = this.assumeRolePolicy) === null || _b === void 0 ? void 0 : _b.validateForResourcePolicy()) || []);
        for (const policy of Object.values(this.inlinePolicies)) {
            errors.push(...policy.validateForIdentityPolicy());
        }
        return errors;
    }
}
exports.Role = Role;
_a = JSII_RTTI_SYMBOL_1;
Role[_a] = { fqn: "@aws-cdk/aws-iam.Role", version: "1.98.0" };
function createAssumeRolePolicy(principal, externalIds) {
    const statement = new AwsStarStatement();
    statement.addPrincipals(principal);
    statement.addActions(principal.assumeRoleAction);
    if (externalIds.length) {
        statement.addCondition('StringEquals', { 'sts:ExternalId': externalIds.length === 1 ? externalIds[0] : externalIds });
    }
    const doc = new policy_document_1.PolicyDocument();
    doc.addStatements(statement);
    return doc;
}
function validateMaxSessionDuration(duration) {
    if (duration === undefined) {
        return;
    }
    if (duration < 3600 || duration > 43200) {
        throw new Error(`maxSessionDuration is set to ${duration}, but must be >= 3600sec (1hr) and <= 43200sec (12hrs)`);
    }
}
/**
 * A PolicyStatement that normalizes its Principal field differently
 *
 * Normally, "anyone" is normalized to "Principal: *", but this statement
 * normalizes to "Principal: { AWS: * }".
 */
class AwsStarStatement extends policy_statement_1.PolicyStatement {
    toStatementJson() {
        const stat = super.toStatementJson();
        if (stat.Principal === '*') {
            stat.Principal = { AWS: '*' };
        }
        return stat;
    }
}
//# sourceMappingURL=data:application/json;base64,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