"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.UniqueStringSet = exports.mergePrincipal = exports.AttachedPolicies = exports.generatePolicyName = exports.undefinedIfEmpty = exports.LITERAL_STRING_KEY = void 0;
const core_1 = require("@aws-cdk/core");
const MAX_POLICY_NAME_LEN = 128;
exports.LITERAL_STRING_KEY = 'LiteralString';
function undefinedIfEmpty(f) {
    return core_1.Lazy.list({
        produce: () => {
            const array = f();
            return (array && array.length > 0) ? array : undefined;
        },
    });
}
exports.undefinedIfEmpty = undefinedIfEmpty;
/**
 * Used to generate a unique policy name based on the policy resource construct.
 * The logical ID of the resource is a great candidate as long as it doesn't exceed
 * 128 characters, so we take the last 128 characters (in order to make sure the hash
 * is there).
 */
function generatePolicyName(scope, logicalId) {
    // as logicalId is itself a Token, resolve it first
    const resolvedLogicalId = core_1.Tokenization.resolve(logicalId, {
        scope,
        resolver: new core_1.DefaultTokenResolver(new core_1.StringConcat()),
    });
    return lastNCharacters(resolvedLogicalId, MAX_POLICY_NAME_LEN);
}
exports.generatePolicyName = generatePolicyName;
/**
 * Returns a string composed of the last n characters of str.
 * If str is shorter than n, returns str.
 *
 * @param str the string to return the last n characters of
 * @param n how many characters to return
 */
function lastNCharacters(str, n) {
    const startIndex = Math.max(str.length - n, 0);
    return str.substring(startIndex, str.length);
}
/**
 * Helper class that maintains the set of attached policies for a principal.
 */
class AttachedPolicies {
    constructor() {
        this.policies = new Array();
    }
    /**
     * Adds a policy to the list of attached policies.
     *
     * If this policy is already, attached, returns false.
     * If there is another policy attached with the same name, throws an exception.
     */
    attach(policy) {
        if (this.policies.find(p => p === policy)) {
            return; // already attached
        }
        if (this.policies.find(p => p.policyName === policy.policyName)) {
            throw new Error(`A policy named "${policy.policyName}" is already attached`);
        }
        this.policies.push(policy);
    }
}
exports.AttachedPolicies = AttachedPolicies;
/**
 * Merge two dictionaries that represent IAM principals
 *
 * Does an in-place merge.
 */
function mergePrincipal(target, source) {
    var _a;
    // If one represents a literal string, the other one must be empty
    if ((exports.LITERAL_STRING_KEY in source && !isEmptyObject(target)) ||
        (exports.LITERAL_STRING_KEY in target && !isEmptyObject(source))) {
        throw new Error(`Cannot merge principals ${JSON.stringify(target)} and ${JSON.stringify(source)}; if one uses a literal principal string the other one must be empty`);
    }
    for (const key of Object.keys(source)) {
        target[key] = (_a = target[key]) !== null && _a !== void 0 ? _a : [];
        let value = source[key];
        if (!Array.isArray(value)) {
            value = [value];
        }
        target[key].push(...value);
    }
    return target;
}
exports.mergePrincipal = mergePrincipal;
/**
 * Lazy string set token that dedupes entries
 *
 * Needs to operate post-resolve, because the inputs could be
 * `[ '${Token[TOKEN.9]}', '${Token[TOKEN.10]}', '${Token[TOKEN.20]}' ]`, which
 * still all resolve to the same string value.
 *
 * Needs to JSON.stringify() results because strings could resolve to literal
 * strings but could also resolve to `{ Fn::Join: [...] }`.
 */
class UniqueStringSet {
    constructor(fn) {
        this.fn = fn;
        this.creationStack = core_1.captureStackTrace();
    }
    static from(fn) {
        return core_1.Token.asList(new UniqueStringSet(fn));
    }
    resolve(context) {
        context.registerPostProcessor(this);
        return this.fn();
    }
    postProcess(input, _context) {
        if (!Array.isArray(input)) {
            return input;
        }
        if (input.length === 0) {
            return undefined;
        }
        const uniq = {};
        for (const el of input) {
            uniq[JSON.stringify(el)] = el;
        }
        return Object.values(uniq);
    }
    toString() {
        return core_1.Token.asString(this);
    }
}
exports.UniqueStringSet = UniqueStringSet;
function isEmptyObject(x) {
    return Object.keys(x).length === 0;
}
//# sourceMappingURL=data:application/json;base64,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