"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.ContextProvider = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const cxapi = require("@aws-cdk/cx-api");
const constructs_1 = require("constructs");
const annotations_1 = require("./annotations");
const stack_1 = require("./stack");
const token_1 = require("./token");
/**
 * (experimental) Base class for the model side of context providers.
 *
 * Instances of this class communicate with context provider plugins in the 'cdk
 * toolkit' via context variables (input), outputting specialized queries for
 * more context variables (output).
 *
 * ContextProvider needs access to a Construct to hook into the context mechanism.
 *
 * @experimental
 */
class ContextProvider {
    constructor() { }
    /**
     * @returns the context key or undefined if a key cannot be rendered (due to tokens used in any of the props)
     * @experimental
     */
    static getKey(scope, options) {
        const stack = stack_1.Stack.of(scope);
        const props = {
            account: stack.account,
            region: stack.region,
            ...options.props || {},
        };
        if (Object.values(props).find(x => token_1.Token.isUnresolved(x))) {
            throw new Error(`Cannot determine scope for context provider ${options.provider}.\n` +
                'This usually happens when one or more of the provider props have unresolved tokens');
        }
        const propStrings = propsToArray(props);
        return {
            key: `${options.provider}:${propStrings.join(':')}`,
            props,
        };
    }
    /**
     * @experimental
     */
    static getValue(scope, options) {
        const stack = stack_1.Stack.of(scope);
        if (token_1.Token.isUnresolved(stack.account) || token_1.Token.isUnresolved(stack.region)) {
            throw new Error(`Cannot retrieve value from context provider ${options.provider} since account/region ` +
                'are not specified at the stack level. Either configure "env" with explicit account and region when ' +
                'you define your stack, or use the environment variables "CDK_DEFAULT_ACCOUNT" and "CDK_DEFAULT_REGION" ' +
                'to inherit environment information from the CLI (not recommended for production stacks)');
        }
        const { key, props } = this.getKey(scope, options);
        const value = constructs_1.Node.of(scope).tryGetContext(key);
        const providerError = extractProviderError(value);
        // if context is missing or an error occurred during context retrieval,
        // report and return a dummy value.
        if (value === undefined || providerError !== undefined) {
            stack.reportMissingContext({
                key,
                provider: options.provider,
                props: props,
            });
            if (providerError !== undefined) {
                annotations_1.Annotations.of(scope).addError(providerError);
            }
            return { value: options.dummyValue };
        }
        return { value };
    }
}
exports.ContextProvider = ContextProvider;
_a = JSII_RTTI_SYMBOL_1;
ContextProvider[_a] = { fqn: "@aws-cdk/core.ContextProvider", version: "1.90.0" };
/**
 * If the context value represents an error, return the error message
 */
function extractProviderError(value) {
    if (typeof value === 'object' && value !== null) {
        return value[cxapi.PROVIDER_ERROR_KEY];
    }
    return undefined;
}
/**
 * Quote colons in all strings so that we can undo the quoting at a later point
 *
 * We'll use $ as a quoting character, for no particularly good reason other
 * than that \ is going to lead to quoting hell when the keys are stored in JSON.
 */
function colonQuote(xs) {
    return xs.replace('$', '$$').replace(':', '$:');
}
function propsToArray(props, keyPrefix = '') {
    const ret = [];
    for (const key of Object.keys(props)) {
        // skip undefined values
        if (props[key] === undefined) {
            continue;
        }
        switch (typeof props[key]) {
            case 'object': {
                ret.push(...propsToArray(props[key], `${keyPrefix}${key}.`));
                break;
            }
            case 'string': {
                ret.push(`${keyPrefix}${key}=${colonQuote(props[key])}`);
                break;
            }
            default: {
                ret.push(`${keyPrefix}${key}=${JSON.stringify(props[key])}`);
                break;
            }
        }
    }
    ret.sort();
    return ret;
}
//# sourceMappingURL=data:application/json;base64,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