"""
This module defines the EnisaIdEnisaVulnerabilityRef class, which represents a reference to an ENISA vulnerability.
"""

from typing import Any
import uuid

from euvd.models._baseenisaidreference import _BaseEnisaIdReference
from euvd.models.vulnerability import Vulnerability


class EnisaIdEnisaVulnerabilityRef(_BaseEnisaIdReference):
    """
    Represents a reference to an ENISA vulnerability.

    Attributes:
        vulnerability (Vulnerability): The vulnerability associated with the ENISA reference.
    """

    vulnerability: Vulnerability

    def __hash__(self):
        """
        Generates a hash value for the ENISA vulnerability reference instance.

        Returns:
            int: The hash value based on the ID and associated vulnerability.
        """
        return hash((self.id, self.vulnerability))

    @classmethod
    def from_dict(cls, data: dict[str, Any]) -> 'EnisaIdEnisaVulnerabilityRef':
        """
        Parses a dictionary into an EnisaIdEnisaVulnerabilityRef instance.

        Args:
            data (dict[str, Any]): The vulnerability reference data to parse.

        Returns:
            EnisaIdEnisaVulnerabilityRef: An instance of the EnisaIdEnisaVulnerabilityRef class.
        """
        return EnisaIdEnisaVulnerabilityRef(
            id=uuid.UUID(data['id']),
            vulnerability=Vulnerability.from_dict(data['enisaId']),
        )


__all__ = [
    'EnisaIdEnisaVulnerabilityRef',
]
