'''
Script for regenerating pseudopotentials missing PP_PSWFC fields with oncvpsp(r).x
Written by Edward Linscott, March 2022
'''

import subprocess
import xml.etree.ElementTree as ET
from pathlib import Path

for fname in Path().rglob('*.upf'):
    try:
        pseudo = ET.parse(fname).getroot()
    except Exception:
        raise ValueError(f'Failed to read `{fname}`')

    # Do not touch files not generated by oncvpsp.x
    pp_info = pseudo.find('PP_INFO')
    if pp_info is None:
        raise ValueError(f'`{fname}` is missing a `PP_INFO` block')
    if pp_info.text is None or 'ONCVPSP ' not in pp_info.text:
        continue

    # Do not touch files that have already got the appropriate fields
    pp_header = pseudo.find('PP_HEADER')
    if pp_header is None:
        raise ValueError(f'`{fname}` is missing a `PP_HEADER` block')
    n_wfc = int(pp_header.get('number_of_wfc', 0))
    if n_wfc > 0:
        continue

    # Do not touch files that have already got a wfc counterpart
    new_fname = fname.parent / (fname.stem + '_wfc.upf')
    if new_fname.exists():
        continue

    # Printing out info to screen
    print(f'Generating {new_fname}...', end='', flush=True)

    # Generate a oncvpsp.x input file
    pp_inputfile = pp_info.find('PP_INPUTFILE')
    if pp_inputfile is None:
        raise ValueError(f'`{fname}` is missing a `PP_INPUTFILE` subblock')
    if pp_inputfile.text is None:
        raise ValueError(f'`{fname}` has an empty `PP_INPUTFILE` subblock')
    with open('tmp.in', 'w') as fd:
        fd.write(pp_inputfile.text)

    # Work out the appropriate program
    if pp_header.get('relativistic') == 'full':
        prog = 'oncvpspr.x'
    else:
        prog = 'oncvpsp.x'
    # pp_header.

    # Run oncvpsp.x
    proc = subprocess.run(f'{prog} < tmp.in', shell=True, capture_output=True, text=True)
    flines = proc.stdout.split('\n')
    try:
        istart = flines.index('Begin PSP_UPF') + 1
        iend = flines.index('</UPF>') + 1
    except ValueError:
        print(' failed!')
        continue

    # Save the new pseudo to file
    with open(new_fname, 'w') as fd:
        fd.write('\n'.join(flines[istart:iend]))

    print(' done')

# Remove tmp.in
Path('tmp.in').unlink()
