# coding: utf-8

import json
import uuid

from stix2.canonicalization.Canonicalize import canonicalize


class Vulnerability:
    def __init__(self, opencti):
        self.opencti = opencti
        self.properties = """
            id
            standard_id
            entity_type
            parent_types
            spec_version
            created_at
            updated_at
            createdBy {
                ... on Identity {
                    id
                    standard_id
                    entity_type
                    parent_types
                    spec_version
                    identity_class
                    name
                    description
                    roles
                    contact_information
                    x_opencti_aliases
                    created
                    modified
                    objectLabel {
                        edges {
                            node {
                                id
                                value
                                color
                            }
                        }
                    }
                }
                ... on Organization {
                    x_opencti_organization_type
                    x_opencti_reliability
                }
                ... on Individual {
                    x_opencti_firstname
                    x_opencti_lastname
                }
            }
            objectMarking {
                edges {
                    node {
                        id
                        standard_id
                        entity_type
                        definition_type
                        definition
                        created
                        modified
                        x_opencti_order
                        x_opencti_color
                    }
                }
            }
            objectLabel {
                edges {
                    node {
                        id
                        value
                        color
                    }
                }
            }
            externalReferences {
                edges {
                    node {
                        id
                        standard_id
                        entity_type
                        source_name
                        description
                        url
                        hash
                        external_id
                        created
                        modified
                        importFiles {
                            edges {
                                node {
                                    id
                                    name
                                    size
                                    metaData {
                                        mimetype
                                        version
                                    }
                                }
                            }
                        }
                    }
                }
            }
            revoked
            confidence
            created
            modified
            name
            description
            x_opencti_aliases
            x_opencti_base_score
            x_opencti_base_severity
            x_opencti_attack_vector
            x_opencti_integrity_impact
            x_opencti_availability_impact
            x_opencti_confidentiality_impact
            importFiles {
                edges {
                    node {
                        id
                        name
                        size
                        metaData {
                            mimetype
                            version
                        }
                    }
                }
            }
        """

    @staticmethod
    def generate_id(name):
        name = name.lower().strip()
        data = {"name": name}
        data = canonicalize(data, utf8=False)
        id = str(uuid.uuid5(uuid.UUID("00abedb4-aa42-466c-9c01-fed23315a9b7"), data))
        return "vulnerability--" + id

    """
        List Vulnerability objects

        :param filters: the filters to apply
        :param search: the search keyword
        :param first: return the first n rows from the after ID (or the beginning if not set)
        :param after: ID of the first row for pagination
        :return List of Vulnerability objects
    """

    def list(self, **kwargs):
        filters = kwargs.get("filters", None)
        search = kwargs.get("search", None)
        first = kwargs.get("first", 100)
        after = kwargs.get("after", None)
        order_by = kwargs.get("orderBy", None)
        order_mode = kwargs.get("orderMode", None)
        custom_attributes = kwargs.get("customAttributes", None)
        get_all = kwargs.get("getAll", False)
        with_pagination = kwargs.get("withPagination", False)
        if get_all:
            first = 100

        self.opencti.app_logger.info(
            "Listing Vulnerabilities with filters", {"filters": json.dumps(filters)}
        )
        query = (
            """
            query Vulnerabilities($filters: FilterGroup, $search: String, $first: Int, $after: ID, $orderBy: VulnerabilitiesOrdering, $orderMode: OrderingMode) {
                vulnerabilities(filters: $filters, search: $search, first: $first, after: $after, orderBy: $orderBy, orderMode: $orderMode) {
                    edges {
                        node {
                            """
            + (custom_attributes if custom_attributes is not None else self.properties)
            + """
                        }
                    }
                    pageInfo {
                        startCursor
                        endCursor
                        hasNextPage
                        hasPreviousPage
                        globalCount
                    }
                }
            }
        """
        )
        result = self.opencti.query(
            query,
            {
                "filters": filters,
                "search": search,
                "first": first,
                "after": after,
                "orderBy": order_by,
                "orderMode": order_mode,
            },
        )

        if get_all:
            final_data = []
            data = self.opencti.process_multiple(result["data"]["vulnerabilities"])
            final_data = final_data + data
            while result["data"]["vulnerabilities"]["pageInfo"]["hasNextPage"]:
                after = result["data"]["vulnerabilities"]["pageInfo"]["endCursor"]
                self.opencti.app_logger.info(
                    "Listing Vulnerabilities", {"after": after}
                )
                result = self.opencti.query(
                    query,
                    {
                        "filters": filters,
                        "search": search,
                        "first": first,
                        "after": after,
                        "orderBy": order_by,
                        "orderMode": order_mode,
                    },
                )
                data = self.opencti.process_multiple(result["data"]["vulnerabilities"])
                final_data = final_data + data
            return final_data
        else:
            return self.opencti.process_multiple(
                result["data"]["vulnerabilities"], with_pagination
            )

    """
        Read a Vulnerability object

        :param id: the id of the Vulnerability
        :param filters: the filters to apply if no id provided
        :return Vulnerability object
    """

    def read(self, **kwargs):
        id = kwargs.get("id", None)
        filters = kwargs.get("filters", None)
        custom_attributes = kwargs.get("customAttributes", None)
        if id is not None:
            self.opencti.app_logger.info("Reading Vulnerability", {"id": id})
            query = (
                """
                query Vulnerability($id: String!) {
                    vulnerability(id: $id) {
                        """
                + (
                    custom_attributes
                    if custom_attributes is not None
                    else self.properties
                )
                + """
                    }
                }
             """
            )
            result = self.opencti.query(query, {"id": id})
            return self.opencti.process_multiple_fields(result["data"]["vulnerability"])
        elif filters is not None:
            result = self.list(filters=filters)
            if len(result) > 0:
                return result[0]
            else:
                return None
        else:
            self.opencti.app_logger.error(
                "[opencti_tool] Missing parameters: id or filters"
            )
            return None

    """
        Create a Vulnerability object

        :param name: the name of the Vulnerability
        :return Vulnerability object
    """

    def create(self, **kwargs):
        stix_id = kwargs.get("stix_id", None)
        created_by = kwargs.get("createdBy", None)
        object_marking = kwargs.get("objectMarking", None)
        object_label = kwargs.get("objectLabel", None)
        external_references = kwargs.get("externalReferences", None)
        revoked = kwargs.get("revoked", None)
        confidence = kwargs.get("confidence", None)
        lang = kwargs.get("lang", None)
        created = kwargs.get("created", None)
        modified = kwargs.get("modified", None)
        name = kwargs.get("name", None)
        description = kwargs.get("description", None)
        x_opencti_aliases = kwargs.get("x_opencti_aliases", None)
        x_opencti_base_score = kwargs.get("x_opencti_base_score", None)
        x_opencti_base_severity = kwargs.get("x_opencti_base_severity", None)
        x_opencti_attack_vector = kwargs.get("x_opencti_attack_vector", None)
        x_opencti_integrity_impact = kwargs.get("x_opencti_integrity_impact", None)
        x_opencti_availability_impact = kwargs.get(
            "x_opencti_availability_impact", None
        )
        x_opencti_confidentiality_impact = kwargs.get(
            "x_opencti_confidentiality_impact", None
        )
        x_opencti_stix_ids = kwargs.get("x_opencti_stix_ids", None)
        granted_refs = kwargs.get("objectOrganization", None)
        update = kwargs.get("update", False)

        if name is not None:
            self.opencti.app_logger.info("Creating Vulnerability", {"name": name})
            query = """
                mutation VulnerabilityAdd($input: VulnerabilityAddInput!) {
                    vulnerabilityAdd(input: $input) {
                        id
                        standard_id
                        entity_type
                        parent_types
                    }
                }
            """
            result = self.opencti.query(
                query,
                {
                    "input": {
                        "stix_id": stix_id,
                        "createdBy": created_by,
                        "objectMarking": object_marking,
                        "objectLabel": object_label,
                        "objectOrganization": granted_refs,
                        "externalReferences": external_references,
                        "revoked": revoked,
                        "confidence": confidence,
                        "lang": lang,
                        "created": created,
                        "modified": modified,
                        "name": name,
                        "description": description,
                        "x_opencti_aliases": x_opencti_aliases,
                        "x_opencti_base_score": x_opencti_base_score,
                        "x_opencti_base_severity": x_opencti_base_severity,
                        "x_opencti_attack_vector": x_opencti_attack_vector,
                        "x_opencti_integrity_impact": x_opencti_integrity_impact,
                        "x_opencti_availability_impact": x_opencti_availability_impact,
                        "x_opencti_confidentiality_impact": x_opencti_confidentiality_impact,
                        "x_opencti_stix_ids": x_opencti_stix_ids,
                        "update": update,
                    }
                },
            )
            return self.opencti.process_multiple_fields(
                result["data"]["vulnerabilityAdd"]
            )
        else:
            self.opencti.app_logger.error(
                "[opencti_vulnerability] Missing parameters: name and description"
            )

    """
        Import an Vulnerability object from a STIX2 object

        :param stixObject: the Stix-Object Vulnerability
        :return Vulnerability object
    """

    def import_from_stix2(self, **kwargs):
        stix_object = kwargs.get("stixObject", None)
        extras = kwargs.get("extras", {})
        update = kwargs.get("update", False)
        if stix_object is not None:
            # Search in extensions
            if "x_opencti_aliases" not in stix_object:
                stix_object[
                    "x_opencti_aliases"
                ] = self.opencti.get_attribute_in_extension("aliases", stix_object)
            if "x_opencti_base_score" not in stix_object:
                stix_object[
                    "x_opencti_base_score"
                ] = self.opencti.get_attribute_in_extension("base_score", stix_object)
            if "x_opencti_base_severity" not in stix_object:
                stix_object[
                    "x_opencti_base_severity"
                ] = self.opencti.get_attribute_in_extension(
                    "base_severity", stix_object
                )
            if "x_opencti_attack_vector" not in stix_object:
                stix_object[
                    "x_opencti_attack_vector"
                ] = self.opencti.get_attribute_in_extension(
                    "attack_vector", stix_object
                )
            if "x_opencti_integrity_impact" not in stix_object:
                stix_object[
                    "x_opencti_integrity_impact"
                ] = self.opencti.get_attribute_in_extension(
                    "integrity_impact", stix_object
                )
            if "x_opencti_availability_impact" not in stix_object:
                stix_object[
                    "x_opencti_availability_impact"
                ] = self.opencti.get_attribute_in_extension(
                    "availability_impact", stix_object
                )
            if "x_opencti_confidentiality_impact" not in stix_object:
                stix_object[
                    "x_opencti_confidentiality_impact"
                ] = self.opencti.get_attribute_in_extension(
                    "confidentiality_impact", stix_object
                )
            if "x_opencti_stix_ids" not in stix_object:
                stix_object[
                    "x_opencti_stix_ids"
                ] = self.opencti.get_attribute_in_extension("stix_ids", stix_object)
            if "x_opencti_granted_refs" not in stix_object:
                stix_object[
                    "x_opencti_granted_refs"
                ] = self.opencti.get_attribute_in_extension("granted_refs", stix_object)
            return self.create(
                stix_id=stix_object["id"],
                createdBy=extras["created_by_id"]
                if "created_by_id" in extras
                else None,
                objectMarking=extras["object_marking_ids"]
                if "object_marking_ids" in extras
                else None,
                objectLabel=extras["object_label_ids"]
                if "object_label_ids" in extras
                else None,
                externalReferences=extras["external_references_ids"]
                if "external_references_ids" in extras
                else None,
                revoked=stix_object["revoked"] if "revoked" in stix_object else None,
                confidence=stix_object["confidence"]
                if "confidence" in stix_object
                else None,
                lang=stix_object["lang"] if "lang" in stix_object else None,
                created=stix_object["created"] if "created" in stix_object else None,
                modified=stix_object["modified"] if "modified" in stix_object else None,
                name=stix_object["name"],
                description=self.opencti.stix2.convert_markdown(
                    stix_object["description"]
                )
                if "description" in stix_object
                else None,
                x_opencti_aliases=stix_object["x_opencti_aliases"]
                if "x_opencti_aliases" in stix_object
                else None,
                x_opencti_base_score=stix_object["x_opencti_base_score"]
                if "x_opencti_base_score" in stix_object
                else None,
                x_opencti_base_severity=stix_object["x_opencti_base_severity"]
                if "x_opencti_base_severity" in stix_object
                else None,
                x_opencti_attack_vector=stix_object["x_opencti_attack_vector"]
                if "x_opencti_attack_vector" in stix_object
                else None,
                x_opencti_integrity_impact=stix_object["x_opencti_integrity_impact"]
                if "x_opencti_integrity_impact" in stix_object
                else None,
                x_opencti_availability_impact=stix_object[
                    "x_opencti_availability_impact"
                ]
                if "x_opencti_availability_impact" in stix_object
                else None,
                x_opencti_confidentiality_impact=stix_object[
                    "x_opencti_confidentiality_impact"
                ]
                if "x_opencti_confidentiality_impact" in stix_object
                else None,
                x_opencti_stix_ids=stix_object["x_opencti_stix_ids"]
                if "x_opencti_stix_ids" in stix_object
                else None,
                objectOrganization=stix_object["x_opencti_granted_refs"]
                if "x_opencti_granted_refs" in stix_object
                else None,
                update=update,
            )
        else:
            self.opencti.app_logger.error(
                "[opencti_vulnerability] Missing parameters: stixObject"
            )
