#  Pyrogram - Telegram MTProto API Client Library for Python
#  Copyright (C) 2017-present Dan <https://github.com/delivrance>
#
#  This file is part of Pyrogram.
#
#  Pyrogram is free software: you can redistribute it and/or modify
#  it under the terms of the GNU Lesser General Public License as published
#  by the Free Software Foundation, either version 3 of the License, or
#  (at your option) any later version.
#
#  Pyrogram is distributed in the hope that it will be useful,
#  but WITHOUT ANY WARRANTY; without even the implied warranty of
#  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#  GNU Lesser General Public License for more details.
#
#  You should have received a copy of the GNU Lesser General Public License
#  along with Pyrogram.  If not, see <http://www.gnu.org/licenses/>.

import logging
from typing import Union, List, Iterable

import pyrogram
from pyrogram import raw
from pyrogram import types

log = logging.getLogger(__name__)


class GetForumTopicsByID:
    async def get_forum_topics_by_id(
        self: "pyrogram.Client",
        chat_id: Union[int, str],
        topic_ids: Union[int, Iterable[int]]
    ) -> Union["types.ForumTopic", List["types.ForumTopic"]]:
        """Get one or more topic from a chat by using topic identifiers.

        .. include:: /_includes/usable-by/users.rst

        Parameters:
            chat_id (``int`` | ``str``):
                Unique identifier (int) or username (str) of the target chat.

            topic_ids (``int`` | Iterable of ``int``, *optional*):
                Pass a single topic identifier or an iterable of topic ids (as integers) to get the information of the
                topic themselves.

        Returns:
            :obj:`~pyrogram.types.ForumTopic` | List of :obj:`~pyrogram.types.ForumTopic`: In case *topic_ids* was not
            a list, a single topic is returned, otherwise a list of topics is returned.

        Example:
            .. code-block:: python

                # Get one topic
                await app.get_forum_topics_by_id(chat_id, 12345)

                # Get more than one topic (list of topics)
                await app.get_forum_topics_by_id(chat_id, [12345, 12346])

        Raises:
            ValueError: In case of invalid arguments.
        """
        is_iterable = not isinstance(topic_ids, int)
        ids = list(topic_ids) if is_iterable else [topic_ids]

        r = await self.invoke(
            raw.functions.channels.GetForumTopicsByID(
                channel=await self.resolve_peer(chat_id),
                topics=ids
            )
        )

        topics_list = []
        parsed_messages = {}
        users = {u.id: u for u in getattr(r, "users", [])}
        chats = {c.id: c for c in getattr(r, "chats", [])}
        topics = {t.id: t for t in getattr(r, "topics", [])}
        messages = {m.id: m for m in getattr(r, "messages", [])}

        for message in messages.values():
            if isinstance(message, raw.types.MessageEmpty):
                continue

            try:
                parsed_messages[message.id] = await types.Message._parse(
                    client=self, 
                    message=message, 
                    users=users,
                    chats=chats,
                    replies=0,
                    from_topic=True if len(topics_list) != 1 else topics_list[topic_ids[0]]
                )
            except: pass  # noqa: E701, E722

        for current in topics.values():
            topics_list.append(types.ForumTopic._parse(
                self, 
                forum_topic=current,
                messages=parsed_messages,
                users=users, 
                chats=chats
            ))

        return topics if is_iterable else topics[0] if topics else None
